# -*- coding: utf-8 -*-
# :Project:   PatchDB -- SQL statements test
# :Created:   lun 22 feb 2016 18:35:06 CET
# :Author:    Lele Gaifax <lele@metapensiero.it>
# :License:   GNU General Public License version 3 or later
# :Copyright: © 2016, 2017, 2022 Lele Gaifax
#

import fixtures


class TestSingleSQLScript(fixtures.BaseTestCase):
    TEST_TXT = """
    Basic Test
    ==========

    .. patchdb:script:: Create first table

       create table sl_test (
         id integer primary key
       )
    """


class TestMultiSQLScript(fixtures.BaseTestCase):
    TEST_TXT = """
    Multiple SQL statements in one script
    =====================================

    .. patchdb:script:: Create table and index

       create table persons (
         id integer not null primary key,
         fullname varchar(50)
       )
       ;;
       create unique index persons_fullname on persons(fullname)
    """


class TestMultiSQLScriptWithDrop(fixtures.BaseTestCase):
    TEST_TXT = """
    Multiple SQL statements in one script
    =====================================

    .. patchdb:script:: Recreate table and index

       drop table persons
       ;;
       create table persons (
         id integer not null primary key,
         fullname varchar(50)
       )
       ;;
       create unique index persons_fullname on persons(fullname)
    """


class TestMultiSQLScriptIgnoringErrors(fixtures.BaseTestCase):
    TEST_TXT = """
    Multiple SQL statements in one script
    =====================================

    .. patchdb:script:: Create table and index
       :onerror: ignore

       create table persons (
         id integer not null primary key,
         fullname varchar(50)
       )
       ;;
       create index persons_notexisting on persons(notexisting)
       ;;
       create unique index persons_fullname on persons(fullname)
    """

    def test(self):
        output = self.patchdb()
        self.assertIn('Done, applied 1 script', output)
        self.assertIn('Ignoring error generated by statement at line 6', self.patchdb_output)
        output = self.patchdb()
        self.assertIn('Done, applied 0 scripts', output)


class TestMultiSQLScriptSkippingOnErrors(fixtures.BaseTestCase):
    TEST_TXT = """
    Multiple SQL statements in one script
    =====================================

    .. patchdb:script:: Create table and index
       :onerror: skip

       create table persons (
         id integer not null primary key,
         fullname varchar(50)
       )
       ;;
       create index persons_notexisting on persons(notexisting)
       ;;
       create unique index persons_fullname on persons(fullname)
    """

    def test(self):
        output = self.patchdb()
        self.assertIn('Done, applied 1 script', output)
        self.assertIn('Skipping succeding statements due to error executing statement'
                      ' at line 6', self.patchdb_output)
        output = self.patchdb()
        self.assertIn('Done, applied 0 scripts', output)


class TestFakeDomains(fixtures.BaseTestCase):
    TEST_TXT = """
    SQLite fake domains
    ===================

    .. patchdb:script:: Data domains
       :always: first

       CREATE DOMAIN id_t integer
       ;;
       CREATE DOMAIN value_t varchar(10)
       ;;
       CREATE domain "VALUE_T" varchar(100)

    .. patchdb:script:: Create table
       :depends: Data domains

       CREATE TABLE test (
         id id_t NOT NULL PRIMARY KEY,
         value value_t default 'VALUE_T',
         value_1 VALUE_T,
         longvalue "VALUE_T"
       )
    """

    def test(self):
        output = self.patchdb()
        self.assertIn('Done, applied 2 script', output)
        output = self.patchdb()
        self.assertIn('Done, applied 1 script', output)
        c, _ = self.get_connection_and_base_exception()
        try:
            q = c.cursor()
            q.execute('PRAGMA table_info(test)')
            r = q.fetchone()
            # Uhm, for some reason now the "id" type comes out in uppercase
            self.assertEqual(r[2].lower(), 'integer')
            r = q.fetchone()
            self.assertEqual(r[2], 'varchar(10)')
            self.assertEqual(r[4], "'VALUE_T'")
            r = q.fetchone()
            self.assertEqual(r[2], 'varchar(10)')
            r = q.fetchone()
            self.assertEqual(r[2], 'varchar(100)')
        finally:
            c.close()


class TestDropNonExistingTable(fixtures.BaseTestCase):
    TEST_TXT = """
    Ignore non existing object test
    ===============================

    .. patchdb:script:: Drop non existing table

       drop table does_not_exist
    """
