"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractTokenDouble = exports.createTokenDouble = exports.unresolved = exports.containsListTokenElement = exports.NullConcat = exports.regexQuote = exports.TokenString = exports.VALID_KEY_CHARS = exports.END_TOKEN_MARKER = exports.BEGIN_LIST_TOKEN_MARKER = exports.BEGIN_STRING_TOKEN_MARKER = void 0;
const string_fragments_1 = require("../string-fragments");
const token_1 = require("../token");
// Details for encoding and decoding Tokens into native types; should not be exported
exports.BEGIN_STRING_TOKEN_MARKER = '${Token[';
exports.BEGIN_LIST_TOKEN_MARKER = '#{Token[';
exports.END_TOKEN_MARKER = ']}';
exports.VALID_KEY_CHARS = 'a-zA-Z0-9:._-';
const QUOTED_BEGIN_STRING_TOKEN_MARKER = regexQuote(exports.BEGIN_STRING_TOKEN_MARKER);
const QUOTED_BEGIN_LIST_TOKEN_MARKER = regexQuote(exports.BEGIN_LIST_TOKEN_MARKER);
const QUOTED_END_TOKEN_MARKER = regexQuote(exports.END_TOKEN_MARKER);
const STRING_TOKEN_REGEX = new RegExp(`${QUOTED_BEGIN_STRING_TOKEN_MARKER}([${exports.VALID_KEY_CHARS}]+)${QUOTED_END_TOKEN_MARKER}`, 'g');
const LIST_TOKEN_REGEX = new RegExp(`${QUOTED_BEGIN_LIST_TOKEN_MARKER}([${exports.VALID_KEY_CHARS}]+)${QUOTED_END_TOKEN_MARKER}`, 'g');
/**
 * A string with markers in it that can be resolved to external values
 */
class TokenString {
    constructor(str, re) {
        this.str = str;
        this.re = re;
    }
    /**
     * Returns a `TokenString` for this string.
     */
    static forString(s) {
        return new TokenString(s, STRING_TOKEN_REGEX);
    }
    /**
     * Returns a `TokenString` for this string (must be the first string element of the list)
     */
    static forListToken(s) {
        return new TokenString(s, LIST_TOKEN_REGEX);
    }
    /**
     * Split string on markers, substituting markers with Tokens
     */
    split(lookup) {
        const ret = new string_fragments_1.TokenizedStringFragments();
        let rest = 0;
        this.re.lastIndex = 0; // Reset
        let m = this.re.exec(this.str);
        while (m) {
            if (m.index > rest) {
                ret.addLiteral(this.str.substring(rest, m.index));
            }
            ret.addToken(lookup(m[1]));
            rest = this.re.lastIndex;
            m = this.re.exec(this.str);
        }
        if (rest < this.str.length) {
            ret.addLiteral(this.str.substring(rest));
        }
        return ret;
    }
    /**
     * Indicates if this string includes tokens.
     */
    test() {
        this.re.lastIndex = 0; // Reset
        return this.re.test(this.str);
    }
}
exports.TokenString = TokenString;
/**
 * Quote a string for use in a regex
 */
function regexQuote(s) {
    return s.replace(/[.?*+^$[\]\\(){}|-]/g, '\\$&');
}
exports.regexQuote = regexQuote;
/**
 * Concatenator that disregards the input
 *
 * Can be used when traversing the tokens is important, but the
 * result isn't.
 */
class NullConcat {
    join(_left, _right) {
        return undefined;
    }
}
exports.NullConcat = NullConcat;
function containsListTokenElement(xs) {
    return xs.some(x => typeof (x) === 'string' && TokenString.forListToken(x).test());
}
exports.containsListTokenElement = containsListTokenElement;
/**
 * Returns true if obj is a token (i.e. has the resolve() method or is a string
 * that includes token markers), or it's a listifictaion of a Token string.
 *
 * @param obj The object to test.
 */
function unresolved(obj) {
    if (typeof (obj) === 'string') {
        return TokenString.forString(obj).test();
    }
    else if (typeof obj === 'number') {
        return extractTokenDouble(obj) !== undefined;
    }
    else if (Array.isArray(obj) && obj.length === 1) {
        return typeof (obj[0]) === 'string' && TokenString.forListToken(obj[0]).test();
    }
    else {
        return token_1.isResolvableObject(obj);
    }
}
exports.unresolved = unresolved;
/**
 * Bit pattern in the top 16 bits of a double to indicate a Token
 *
 * An IEEE double in LE memory order looks like this (grouped
 * into octets, then grouped into 32-bit words):
 *
 * mmmmmmmm.mmmmmmmm.mmmmmmmm.mmmmmmmm | mmmmmmmm.mmmmmmmm.EEEEmmmm.sEEEEEEE
 *
 * - m: mantissa (52 bits)
 * - E: exponent (11 bits)
 * - s: sign (1 bit)
 *
 * We put the following marker into the top 16 bits (exponent and sign), and
 * use the mantissa part to encode the token index. To save some bit twiddling
 * we use all top 16 bits for the tag. That loses us 4 mantissa bits to store
 * information in but we still have 48, which is going to be plenty for any
 * number of tokens to be created during the lifetime of any CDK application.
 *
 * Can't have all bits set because that makes a NaN, so unset the least
 * significant exponent bit.
 *
 * Currently not supporting BE architectures.
 */
// tslint:disable-next-line:no-bitwise
const DOUBLE_TOKEN_MARKER_BITS = 0xFBFF << 16;
/**
 * Highest encodable number
 */
const MAX_ENCODABLE_INTEGER = Math.pow(2, 48) - 1;
/**
 * Get 2^32 as a number, so we can do multiplication and div instead of bit shifting
 *
 * Necessary because in JavaScript, bit operations implicitly convert
 * to int32 and we need them to work on "int64"s.
 *
 * So instead of x >> 32, we do Math.floor(x / 2^32), and vice versa.
 */
const BITS32 = Math.pow(2, 32);
/**
 * Return a special Double value that encodes the given nonnegative integer
 *
 * We use this to encode Token ordinals.
 */
function createTokenDouble(x) {
    if (Math.floor(x) !== x || x < 0) {
        throw new Error('Can only encode positive integers');
    }
    if (x > MAX_ENCODABLE_INTEGER) {
        throw new Error(`Got an index too large to encode: ${x}`);
    }
    const buf = new ArrayBuffer(8);
    const ints = new Uint32Array(buf);
    // tslint:disable:no-bitwise
    ints[0] = x & 0x0000FFFFFFFF; // Bottom 32 bits of number
    // This needs an "x >> 32" but that will make it a 32-bit number instead
    // of a 64-bit number.
    ints[1] = (shr32(x) & 0xFFFF) | DOUBLE_TOKEN_MARKER_BITS; // Top 16 bits of number and the mask
    // tslint:enable:no-bitwise
    return (new Float64Array(buf))[0];
}
exports.createTokenDouble = createTokenDouble;
/**
 * Shift a 64-bit int right 32 bits
 */
function shr32(x) {
    return Math.floor(x / BITS32);
}
/**
 * Shift a 64-bit left 32 bits
 */
function shl32(x) {
    return x * BITS32;
}
/**
 * Extract the encoded integer out of the special Double value
 *
 * Returns undefined if the float is a not an encoded token.
 */
function extractTokenDouble(encoded) {
    const buf = new ArrayBuffer(8);
    (new Float64Array(buf))[0] = encoded;
    const ints = new Uint32Array(buf);
    // tslint:disable:no-bitwise
    if ((ints[1] & 0xFFFF0000) !== DOUBLE_TOKEN_MARKER_BITS) {
        return undefined;
    }
    // Must use + instead of | here (bitwise operations
    // will force 32-bits integer arithmetic, + will not).
    return ints[0] + shl32(ints[1] & 0xFFFF);
    // tslint:enable:no-bitwise
}
exports.extractTokenDouble = extractTokenDouble;
//# sourceMappingURL=data:application/json;base64,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