"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.shouldFollow = exports.shouldExclude = void 0;
const fs = require("fs");
const minimatch = require("minimatch");
const path = require("path");
const options_1 = require("./options");
/**
 * Determines whether a given file should be excluded or not based on given
 * exclusion glob patterns.
 *
 * @param exclude  exclusion patterns
 * @param filePath file path to be assessed against the pattern
 *
 * @returns `true` if the file should be excluded
 */
function shouldExclude(exclude, filePath) {
    let excludeOutput = false;
    for (const pattern of exclude) {
        const negate = pattern.startsWith('!');
        const match = minimatch(filePath, pattern, { matchBase: true, flipNegate: true });
        if (!negate && match) {
            excludeOutput = true;
        }
        if (negate && match) {
            excludeOutput = false;
        }
    }
    return excludeOutput;
}
exports.shouldExclude = shouldExclude;
/**
 * Determines whether a symlink should be followed or not, based on a FollowMode.
 *
 * @param mode       the FollowMode.
 * @param sourceRoot the root of the source tree.
 * @param realPath   the real path of the target of the symlink.
 *
 * @returns true if the link should be followed.
 */
function shouldFollow(mode, sourceRoot, realPath) {
    switch (mode) {
        case options_1.SymlinkFollowMode.ALWAYS:
            return fs.existsSync(realPath);
        case options_1.SymlinkFollowMode.EXTERNAL:
            return !_isInternal() && fs.existsSync(realPath);
        case options_1.SymlinkFollowMode.BLOCK_EXTERNAL:
            return _isInternal() && fs.existsSync(realPath);
        case options_1.SymlinkFollowMode.NEVER:
            return false;
        default:
            throw new Error(`Unsupported FollowMode: ${mode}`);
    }
    function _isInternal() {
        return path.resolve(realPath).startsWith(path.resolve(sourceRoot));
    }
}
exports.shouldFollow = shouldFollow;
//# sourceMappingURL=data:application/json;base64,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