"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetStaging = void 0;
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const crypto = require("crypto");
const fs = require("fs-extra");
const os = require("os");
const path = require("path");
const assets_1 = require("./assets");
const construct_compat_1 = require("./construct-compat");
const fs_1 = require("./fs");
const STAGING_TMP = '.cdk.staging';
/**
 * Stages a file or directory from a location on the file system into a staging
 * directory.
 *
 * This is controlled by the context key 'aws:cdk:asset-staging' and enabled
 * by the CLI by default in order to ensure that when the CDK app exists, all
 * assets are available for deployment. Otherwise, if an app references assets
 * in temporary locations, those will not be available when it exists (see
 * https://github.com/aws/aws-cdk/issues/1716).
 *
 * The `stagedPath` property is a stringified token that represents the location
 * of the file or directory after staging. It will be resolved only during the
 * "prepare" stage and may be either the original path or the staged path
 * depending on the context setting.
 *
 * The file/directory are staged based on their content hash (fingerprint). This
 * means that only if content was changed, copy will happen.
 */
class AssetStaging extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.sourcePath = props.sourcePath;
        this.fingerprintOptions = props;
        if (props.bundling) {
            this.bundleDir = this.bundle(props.bundling);
        }
        this.assetHash = this.calculateHash(props);
        const stagingDisabled = this.node.tryGetContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT);
        if (stagingDisabled) {
            this.stagedPath = (_a = this.bundleDir) !== null && _a !== void 0 ? _a : this.sourcePath;
        }
        else {
            this.relativePath = `asset.${this.assetHash}${path.extname((_b = this.bundleDir) !== null && _b !== void 0 ? _b : this.sourcePath)}`;
            this.stagedPath = this.relativePath;
        }
        this.sourceHash = this.assetHash;
    }
    synthesize(session) {
        // Staging is disabled
        if (!this.relativePath) {
            return;
        }
        const targetPath = path.join(session.assembly.outdir, this.relativePath);
        // Already staged
        if (fs.existsSync(targetPath)) {
            return;
        }
        // Asset has been bundled
        if (this.bundleDir) {
            // Move bundling directory to staging directory
            fs.moveSync(this.bundleDir, targetPath);
            return;
        }
        // Copy file/directory to staging directory
        const stat = fs.statSync(this.sourcePath);
        if (stat.isFile()) {
            fs.copyFileSync(this.sourcePath, targetPath);
        }
        else if (stat.isDirectory()) {
            fs.mkdirSync(targetPath);
            fs_1.FileSystem.copyDirectory(this.sourcePath, targetPath, this.fingerprintOptions);
        }
        else {
            throw new Error(`Unknown file type: ${this.sourcePath}`);
        }
    }
    bundle(options) {
        var _a, _b;
        // Temp staging directory in the working directory
        const stagingTmp = path.join('.', STAGING_TMP);
        fs.ensureDirSync(stagingTmp);
        // Create temp directory for bundling inside the temp staging directory
        const bundleDir = path.resolve(fs.mkdtempSync(path.join(stagingTmp, 'asset-bundle-')));
        let user;
        if (options.user) {
            user = options.user;
        }
        else { // Default to current user
            const userInfo = os.userInfo();
            user = userInfo.uid !== -1 // uid is -1 on Windows
                ? `${userInfo.uid}:${userInfo.gid}`
                : '1000:1000';
        }
        // Always mount input and output dir
        const volumes = [
            {
                hostPath: this.sourcePath,
                containerPath: AssetStaging.BUNDLING_INPUT_DIR,
            },
            {
                hostPath: bundleDir,
                containerPath: AssetStaging.BUNDLING_OUTPUT_DIR,
            },
            ...(_a = options.volumes) !== null && _a !== void 0 ? _a : [],
        ];
        try {
            options.image._run({
                command: options.command,
                user,
                volumes,
                environment: options.environment,
                workingDirectory: (_b = options.workingDirectory) !== null && _b !== void 0 ? _b : AssetStaging.BUNDLING_INPUT_DIR,
            });
        }
        catch (err) {
            throw new Error(`Failed to run bundling Docker image for asset ${this.node.path}: ${err}`);
        }
        if (fs_1.FileSystem.isEmpty(bundleDir)) {
            throw new Error(`Bundling did not produce any output. Check that your container writes content to ${AssetStaging.BUNDLING_OUTPUT_DIR}.`);
        }
        return bundleDir;
    }
    calculateHash(props) {
        let hashType;
        if (props.assetHash) {
            if (props.assetHashType && props.assetHashType !== assets_1.AssetHashType.CUSTOM) {
                throw new Error(`Cannot specify \`${props.assetHashType}\` for \`assetHashType\` when \`assetHash\` is specified. Use \`CUSTOM\` or leave \`undefined\`.`);
            }
            hashType = assets_1.AssetHashType.CUSTOM;
        }
        else if (props.assetHashType) {
            hashType = props.assetHashType;
        }
        else {
            hashType = assets_1.AssetHashType.SOURCE;
        }
        switch (hashType) {
            case assets_1.AssetHashType.SOURCE:
                return fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions);
            case assets_1.AssetHashType.BUNDLE:
                if (!this.bundleDir) {
                    throw new Error('Cannot use `AssetHashType.BUNDLE` when `bundling` is not specified.');
                }
                return fs_1.FileSystem.fingerprint(this.bundleDir, this.fingerprintOptions);
            case assets_1.AssetHashType.CUSTOM:
                if (!props.assetHash) {
                    throw new Error('`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`.');
                }
                // Hash the hash to make sure we can use it in a file/directory name.
                // The resulting hash will also have the same length as for the other hash types.
                return crypto.createHash('sha256').update(props.assetHash).digest('hex');
            default:
                throw new Error('Unknown asset hash type.');
        }
    }
}
exports.AssetStaging = AssetStaging;
/**
 * The directory inside the bundling container into which the asset sources will be mounted.
 * @experimental
 */
AssetStaging.BUNDLING_INPUT_DIR = '/asset-input';
/**
 * The directory inside the bundling container into which the bundled output should be written.
 * @experimental
 */
AssetStaging.BUNDLING_OUTPUT_DIR = '/asset-output';
//# sourceMappingURL=data:application/json;base64,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