"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntegrationPattern = exports.TaskStateBase = void 0;
const cloudwatch = require("../../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const chain_1 = require("../chain");
const state_1 = require("./state");
/**
 * Define a Task state in the state machine
 *
 * Reaching a Task state causes some work to be executed, represented by the
 * Task's resource property. Task constructs represent a generic Amazon
 * States Language Task.
 *
 * For some resource types, more specific subclasses of Task may be available
 * which are more convenient to use.
 */
class TaskStateBase extends state_1.State {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.endStates = [this];
        this.timeout = props.timeout;
        this.heartbeat = props.heartbeat;
    }
    /**
     * Add retry configuration for this state
     *
     * This controls if and how the execution will be retried if a particular
     * error occurs.
     */
    addRetry(props = {}) {
        super._addRetry(props);
        return this;
    }
    /**
     * Add a recovery handler for this state
     *
     * When a particular error occurs, execution will continue at the error
     * handler instead of failing the state machine execution.
     */
    addCatch(handler, props = {}) {
        super._addCatch(handler.startState, props);
        return this;
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            ...this.renderNextEnd(),
            ...this.renderRetryCatch(),
            ...this.renderTaskBase(),
            ...this.renderTask(),
        };
    }
    /**
     * Return the given named metric for this Task
     *
     * @default - sum over 5 minutes
     */
    metric(metricName, props) {
        var _a;
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensions: (_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricDimensions,
            statistic: 'sum',
            ...props,
        }).attachTo(this);
    }
    /**
     * The interval, in milliseconds, between the time the Task starts and the time it closes.
     *
     * @default - average over 5 minutes
     */
    metricRunTime(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixSingular, 'RunTime', { statistic: 'avg', ...props });
    }
    /**
     * The interval, in milliseconds, for which the activity stays in the schedule state.
     *
     * @default - average over 5 minutes
     */
    metricScheduleTime(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixSingular, 'ScheduleTime', { statistic: 'avg', ...props });
    }
    /**
     * The interval, in milliseconds, between the time the activity is scheduled and the time it closes.
     *
     * @default - average over 5 minutes
     */
    metricTime(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixSingular, 'Time', { statistic: 'avg', ...props });
    }
    /**
     * Metric for the number of times this activity is scheduled
     *
     * @default - sum over 5 minutes
     */
    metricScheduled(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixPlural, 'Scheduled', props);
    }
    /**
     * Metric for the number of times this activity times out
     *
     * @default - sum over 5 minutes
     */
    metricTimedOut(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixPlural, 'TimedOut', props);
    }
    /**
     * Metric for the number of times this activity is started
     *
     * @default - sum over 5 minutes
     */
    metricStarted(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixPlural, 'Started', props);
    }
    /**
     * Metric for the number of times this activity succeeds
     *
     * @default - sum over 5 minutes
     */
    metricSucceeded(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixPlural, 'Succeeded', props);
    }
    /**
     * Metric for the number of times this activity fails
     *
     * @default - sum over 5 minutes
     */
    metricFailed(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixPlural, 'Failed', props);
    }
    /**
     * Metric for the number of times the heartbeat times out for this activity
     *
     * @default - sum over 5 minutes
     */
    metricHeartbeatTimedOut(props) {
        var _a;
        return this.taskMetric((_a = this.taskMetrics) === null || _a === void 0 ? void 0 : _a.metricPrefixPlural, 'HeartbeatTimedOut', props);
    }
    whenBoundToGraph(graph) {
        super.whenBoundToGraph(graph);
        for (const policyStatement of this.taskPolicies || []) {
            graph.registerPolicyStatement(policyStatement);
        }
    }
    taskMetric(prefix, suffix, props) {
        if (prefix === undefined) {
            throw new Error('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        }
        return this.metric(prefix + suffix, props);
    }
    renderTaskBase() {
        var _a, _b;
        return {
            Type: 'Task',
            Comment: this.comment,
            TimeoutSeconds: (_a = this.timeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            HeartbeatSeconds: (_b = this.heartbeat) === null || _b === void 0 ? void 0 : _b.toSeconds(),
            InputPath: state_1.renderJsonPath(this.inputPath),
            OutputPath: state_1.renderJsonPath(this.outputPath),
            ResultPath: state_1.renderJsonPath(this.resultPath),
        };
    }
}
exports.TaskStateBase = TaskStateBase;
/**
 *
 * AWS Step Functions integrates with services directly in the Amazon States Language.
 * You can control these AWS services using service integration patterns:
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 *
 */
var IntegrationPattern;
(function (IntegrationPattern) {
    /**
     * Step Functions will wait for an HTTP response and then progress to the next state.
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-default
     */
    IntegrationPattern["REQUEST_RESPONSE"] = "REQUEST_RESPONSE";
    /**
     * Step Functions can wait for a request to complete before progressing to the next state.
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-sync
     */
    IntegrationPattern["RUN_JOB"] = "RUN_JOB";
    /**
     * Callback tasks provide a way to pause a workflow until a task token is returned.
     * You must set a task token when using the callback pattern
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
     */
    IntegrationPattern["WAIT_FOR_TASK_TOKEN"] = "WAIT_FOR_TASK_TOKEN";
})(IntegrationPattern = exports.IntegrationPattern || (exports.IntegrationPattern = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGFzay1iYXNlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGFzay1iYXNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHNEQUFzRCxDQUFDLDBEQUEwRDtBQUVqSCxvQ0FBaUM7QUFHakMsbUNBQWdEO0FBK0RoRDs7Ozs7Ozs7O0dBU0c7QUFDSCxNQUFzQixhQUFjLFNBQVEsYUFBSztJQU03QyxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQXlCO1FBQ25FLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3hCLElBQUksQ0FBQyxTQUFTLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN4QixJQUFJLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDN0IsSUFBSSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO0lBQ3JDLENBQUM7SUFDRDs7Ozs7T0FLRztJQUNJLFFBQVEsQ0FBQyxRQUFvQixFQUFFO1FBQ2xDLEtBQUssQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDdkIsT0FBTyxJQUFJLENBQUM7SUFDaEIsQ0FBQztJQUNEOzs7OztPQUtHO0lBQ0ksUUFBUSxDQUFDLE9BQW1CLEVBQUUsUUFBb0IsRUFBRTtRQUN2RCxLQUFLLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxVQUFVLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDaEIsQ0FBQztJQUNEOztPQUVHO0lBQ0ksSUFBSSxDQUFDLElBQWdCO1FBQ3hCLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sYUFBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUNEOztPQUVHO0lBQ0ksV0FBVztRQUNkLE9BQU87WUFDSCxHQUFHLElBQUksQ0FBQyxhQUFhLEVBQUU7WUFDdkIsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLEVBQUU7WUFDMUIsR0FBRyxJQUFJLENBQUMsY0FBYyxFQUFFO1lBQ3hCLEdBQUcsSUFBSSxDQUFDLFVBQVUsRUFBRTtTQUN2QixDQUFDO0lBQ04sQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsVUFBa0IsRUFBRSxLQUFnQzs7UUFDOUQsT0FBTyxJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUM7WUFDekIsU0FBUyxFQUFFLFlBQVk7WUFDdkIsVUFBVTtZQUNWLFVBQVUsUUFBRSxJQUFJLENBQUMsV0FBVywwQ0FBRSxnQkFBZ0I7WUFDOUMsU0FBUyxFQUFFLEtBQUs7WUFDaEIsR0FBRyxLQUFLO1NBQ1gsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLGFBQWEsQ0FBQyxLQUFnQzs7UUFDakQsT0FBTyxJQUFJLENBQUMsVUFBVSxPQUFDLElBQUksQ0FBQyxXQUFXLDBDQUFFLG9CQUFvQixFQUFFLFNBQVMsRUFBRSxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsR0FBRyxLQUFLLEVBQUUsQ0FBQyxDQUFDO0lBQzlHLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksa0JBQWtCLENBQUMsS0FBZ0M7O1FBQ3RELE9BQU8sSUFBSSxDQUFDLFVBQVUsT0FBQyxJQUFJLENBQUMsV0FBVywwQ0FBRSxvQkFBb0IsRUFBRSxjQUFjLEVBQUUsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLEdBQUcsS0FBSyxFQUFFLENBQUMsQ0FBQztJQUNuSCxDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLFVBQVUsQ0FBQyxLQUFnQzs7UUFDOUMsT0FBTyxJQUFJLENBQUMsVUFBVSxPQUFDLElBQUksQ0FBQyxXQUFXLDBDQUFFLG9CQUFvQixFQUFFLE1BQU0sRUFBRSxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsR0FBRyxLQUFLLEVBQUUsQ0FBQyxDQUFDO0lBQzNHLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksZUFBZSxDQUFDLEtBQWdDOztRQUNuRCxPQUFPLElBQUksQ0FBQyxVQUFVLE9BQUMsSUFBSSxDQUFDLFdBQVcsMENBQUUsa0JBQWtCLEVBQUUsV0FBVyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3JGLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksY0FBYyxDQUFDLEtBQWdDOztRQUNsRCxPQUFPLElBQUksQ0FBQyxVQUFVLE9BQUMsSUFBSSxDQUFDLFdBQVcsMENBQUUsa0JBQWtCLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3BGLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksYUFBYSxDQUFDLEtBQWdDOztRQUNqRCxPQUFPLElBQUksQ0FBQyxVQUFVLE9BQUMsSUFBSSxDQUFDLFdBQVcsMENBQUUsa0JBQWtCLEVBQUUsU0FBUyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ25GLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksZUFBZSxDQUFDLEtBQWdDOztRQUNuRCxPQUFPLElBQUksQ0FBQyxVQUFVLE9BQUMsSUFBSSxDQUFDLFdBQVcsMENBQUUsa0JBQWtCLEVBQUUsV0FBVyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3JGLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksWUFBWSxDQUFDLEtBQWdDOztRQUNoRCxPQUFPLElBQUksQ0FBQyxVQUFVLE9BQUMsSUFBSSxDQUFDLFdBQVcsMENBQUUsa0JBQWtCLEVBQUUsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ2xGLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksdUJBQXVCLENBQUMsS0FBZ0M7O1FBQzNELE9BQU8sSUFBSSxDQUFDLFVBQVUsT0FBQyxJQUFJLENBQUMsV0FBVywwQ0FBRSxrQkFBa0IsRUFBRSxtQkFBbUIsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUM3RixDQUFDO0lBQ1MsZ0JBQWdCLENBQUMsS0FBaUI7UUFDeEMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzlCLEtBQUssTUFBTSxlQUFlLElBQUksSUFBSSxDQUFDLFlBQVksSUFBSSxFQUFFLEVBQUU7WUFDbkQsS0FBSyxDQUFDLHVCQUF1QixDQUFDLGVBQWUsQ0FBQyxDQUFDO1NBQ2xEO0lBQ0wsQ0FBQztJQUVPLFVBQVUsQ0FBQyxNQUEwQixFQUFFLE1BQWMsRUFBRSxLQUFnQztRQUMzRixJQUFJLE1BQU0sS0FBSyxTQUFTLEVBQUU7WUFDdEIsTUFBTSxJQUFJLEtBQUssQ0FBQyw2RUFBNkUsQ0FBQyxDQUFDO1NBQ2xHO1FBQ0QsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sR0FBRyxNQUFNLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUNPLGNBQWM7O1FBQ2xCLE9BQU87WUFDSCxJQUFJLEVBQUUsTUFBTTtZQUNaLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTztZQUNyQixjQUFjLFFBQUUsSUFBSSxDQUFDLE9BQU8sMENBQUUsU0FBUyxFQUFFO1lBQ3pDLGdCQUFnQixRQUFFLElBQUksQ0FBQyxTQUFTLDBDQUFFLFNBQVMsRUFBRTtZQUM3QyxTQUFTLEVBQUUsc0JBQWMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ3pDLFVBQVUsRUFBRSxzQkFBYyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUM7WUFDM0MsVUFBVSxFQUFFLHNCQUFjLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQztTQUM5QyxDQUFDO0lBQ04sQ0FBQztDQUNKO0FBaEtELHNDQWdLQztBQXdCRDs7Ozs7OztHQU9HO0FBQ0gsSUFBWSxrQkFvQlg7QUFwQkQsV0FBWSxrQkFBa0I7SUFDMUI7Ozs7T0FJRztJQUNILDJEQUFxQyxDQUFBO0lBQ3JDOzs7O09BSUc7SUFDSCx5Q0FBbUIsQ0FBQTtJQUNuQjs7Ozs7T0FLRztJQUNILGlFQUEyQyxDQUFBO0FBQy9DLENBQUMsRUFwQlcsa0JBQWtCLEdBQWxCLDBCQUFrQixLQUFsQiwwQkFBa0IsUUFvQjdCIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2xvdWR3YXRjaCBmcm9tIFwiLi4vLi4vLi4vYXdzLWNsb3Vkd2F0Y2hcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJ1xuaW1wb3J0ICogYXMgaWFtIGZyb20gXCIuLi8uLi8uLi9hd3MtaWFtXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtaWFtJ1xuaW1wb3J0IHsgQ2hhaW4gfSBmcm9tICcuLi9jaGFpbic7XG5pbXBvcnQgeyBTdGF0ZUdyYXBoIH0gZnJvbSAnLi4vc3RhdGUtZ3JhcGgnO1xuaW1wb3J0IHsgQ2F0Y2hQcm9wcywgSUNoYWluYWJsZSwgSU5leHRhYmxlLCBSZXRyeVByb3BzIH0gZnJvbSAnLi4vdHlwZXMnO1xuaW1wb3J0IHsgcmVuZGVySnNvblBhdGgsIFN0YXRlIH0gZnJvbSAnLi9zdGF0ZSc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSBcIi4uLy4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG4vKipcbiAqIFByb3BzIHRoYXQgYXJlIGNvbW1vbiB0byBhbGwgdGFza3NcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBUYXNrU3RhdGVCYXNlUHJvcHMge1xuICAgIC8qKlxuICAgICAqIEFuIG9wdGlvbmFsIGRlc2NyaXB0aW9uIGZvciB0aGlzIHN0YXRlXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGNvbW1lbnRcbiAgICAgKi9cbiAgICByZWFkb25seSBjb21tZW50Pzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIEpTT05QYXRoIGV4cHJlc3Npb24gdG8gc2VsZWN0IHBhcnQgb2YgdGhlIHN0YXRlIHRvIGJlIHRoZSBpbnB1dCB0byB0aGlzIHN0YXRlLlxuICAgICAqXG4gICAgICogTWF5IGFsc28gYmUgdGhlIHNwZWNpYWwgdmFsdWUgRElTQ0FSRCwgd2hpY2ggd2lsbCBjYXVzZSB0aGUgZWZmZWN0aXZlXG4gICAgICogaW5wdXQgdG8gYmUgdGhlIGVtcHR5IG9iamVjdCB7fS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gVGhlIGVudGlyZSB0YXNrIGlucHV0IChKU09OIHBhdGggJyQnKVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGlucHV0UGF0aD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBKU09OUGF0aCBleHByZXNzaW9uIHRvIHNlbGVjdCBzZWxlY3QgYSBwb3J0aW9uIG9mIHRoZSBzdGF0ZSBvdXRwdXQgdG8gcGFzc1xuICAgICAqIHRvIHRoZSBuZXh0IHN0YXRlLlxuICAgICAqXG4gICAgICogTWF5IGFsc28gYmUgdGhlIHNwZWNpYWwgdmFsdWUgRElTQ0FSRCwgd2hpY2ggd2lsbCBjYXVzZSB0aGUgZWZmZWN0aXZlXG4gICAgICogb3V0cHV0IHRvIGJlIHRoZSBlbXB0eSBvYmplY3Qge30uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIFRoZSBlbnRpcmUgSlNPTiBub2RlIGRldGVybWluZWQgYnkgdGhlIHN0YXRlIGlucHV0LCB0aGUgdGFzayByZXN1bHQsXG4gICAgICogICBhbmQgcmVzdWx0UGF0aCBpcyBwYXNzZWQgdG8gdGhlIG5leHQgc3RhdGUgKEpTT04gcGF0aCAnJCcpXG4gICAgICovXG4gICAgcmVhZG9ubHkgb3V0cHV0UGF0aD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBKU09OUGF0aCBleHByZXNzaW9uIHRvIGluZGljYXRlIHdoZXJlIHRvIGluamVjdCB0aGUgc3RhdGUncyBvdXRwdXRcbiAgICAgKlxuICAgICAqIE1heSBhbHNvIGJlIHRoZSBzcGVjaWFsIHZhbHVlIERJU0NBUkQsIHdoaWNoIHdpbGwgY2F1c2UgdGhlIHN0YXRlJ3NcbiAgICAgKiBpbnB1dCB0byBiZWNvbWUgaXRzIG91dHB1dC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gUmVwbGFjZXMgdGhlIGVudGlyZSBpbnB1dCB3aXRoIHRoZSByZXN1bHQgKEpTT04gcGF0aCAnJCcpXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmVzdWx0UGF0aD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaW1lb3V0IGZvciB0aGUgc3RhdGUgbWFjaGluZVxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgdGltZW91dD86IGNkay5EdXJhdGlvbjtcbiAgICAvKipcbiAgICAgKiBUaW1lb3V0IGZvciB0aGUgaGVhcnRiZWF0XG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBoZWFydGJlYXQ/OiBjZGsuRHVyYXRpb247XG4gICAgLyoqXG4gICAgICogQVdTIFN0ZXAgRnVuY3Rpb25zIGludGVncmF0ZXMgd2l0aCBzZXJ2aWNlcyBkaXJlY3RseSBpbiB0aGUgQW1hem9uIFN0YXRlcyBMYW5ndWFnZS5cbiAgICAgKiBZb3UgY2FuIGNvbnRyb2wgdGhlc2UgQVdTIHNlcnZpY2VzIHVzaW5nIHNlcnZpY2UgaW50ZWdyYXRpb24gcGF0dGVybnNcbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N0ZXAtZnVuY3Rpb25zL2xhdGVzdC9kZy9jb25uZWN0LXRvLXJlc291cmNlLmh0bWwjY29ubmVjdC13YWl0LXRva2VuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBJbnRlZ3JhdGlvblBhdHRlcm4uUkVRVUVTVF9SRVNQT05TRVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGludGVncmF0aW9uUGF0dGVybj86IEludGVncmF0aW9uUGF0dGVybjtcbn1cbi8qKlxuICogRGVmaW5lIGEgVGFzayBzdGF0ZSBpbiB0aGUgc3RhdGUgbWFjaGluZVxuICpcbiAqIFJlYWNoaW5nIGEgVGFzayBzdGF0ZSBjYXVzZXMgc29tZSB3b3JrIHRvIGJlIGV4ZWN1dGVkLCByZXByZXNlbnRlZCBieSB0aGVcbiAqIFRhc2sncyByZXNvdXJjZSBwcm9wZXJ0eS4gVGFzayBjb25zdHJ1Y3RzIHJlcHJlc2VudCBhIGdlbmVyaWMgQW1hem9uXG4gKiBTdGF0ZXMgTGFuZ3VhZ2UgVGFzay5cbiAqXG4gKiBGb3Igc29tZSByZXNvdXJjZSB0eXBlcywgbW9yZSBzcGVjaWZpYyBzdWJjbGFzc2VzIG9mIFRhc2sgbWF5IGJlIGF2YWlsYWJsZVxuICogd2hpY2ggYXJlIG1vcmUgY29udmVuaWVudCB0byB1c2UuXG4gKi9cbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBUYXNrU3RhdGVCYXNlIGV4dGVuZHMgU3RhdGUgaW1wbGVtZW50cyBJTmV4dGFibGUge1xuICAgIHB1YmxpYyByZWFkb25seSBlbmRTdGF0ZXM6IElOZXh0YWJsZVtdO1xuICAgIHByb3RlY3RlZCBhYnN0cmFjdCByZWFkb25seSB0YXNrTWV0cmljcz86IFRhc2tNZXRyaWNzQ29uZmlnO1xuICAgIHByb3RlY3RlZCBhYnN0cmFjdCByZWFkb25seSB0YXNrUG9saWNpZXM/OiBpYW0uUG9saWN5U3RhdGVtZW50W107XG4gICAgcHJpdmF0ZSByZWFkb25seSB0aW1lb3V0PzogY2RrLkR1cmF0aW9uO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgaGVhcnRiZWF0PzogY2RrLkR1cmF0aW9uO1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogVGFza1N0YXRlQmFzZVByb3BzKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuICAgICAgICB0aGlzLmVuZFN0YXRlcyA9IFt0aGlzXTtcbiAgICAgICAgdGhpcy50aW1lb3V0ID0gcHJvcHMudGltZW91dDtcbiAgICAgICAgdGhpcy5oZWFydGJlYXQgPSBwcm9wcy5oZWFydGJlYXQ7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZCByZXRyeSBjb25maWd1cmF0aW9uIGZvciB0aGlzIHN0YXRlXG4gICAgICpcbiAgICAgKiBUaGlzIGNvbnRyb2xzIGlmIGFuZCBob3cgdGhlIGV4ZWN1dGlvbiB3aWxsIGJlIHJldHJpZWQgaWYgYSBwYXJ0aWN1bGFyXG4gICAgICogZXJyb3Igb2NjdXJzLlxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRSZXRyeShwcm9wczogUmV0cnlQcm9wcyA9IHt9KTogVGFza1N0YXRlQmFzZSB7XG4gICAgICAgIHN1cGVyLl9hZGRSZXRyeShwcm9wcyk7XG4gICAgICAgIHJldHVybiB0aGlzO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGQgYSByZWNvdmVyeSBoYW5kbGVyIGZvciB0aGlzIHN0YXRlXG4gICAgICpcbiAgICAgKiBXaGVuIGEgcGFydGljdWxhciBlcnJvciBvY2N1cnMsIGV4ZWN1dGlvbiB3aWxsIGNvbnRpbnVlIGF0IHRoZSBlcnJvclxuICAgICAqIGhhbmRsZXIgaW5zdGVhZCBvZiBmYWlsaW5nIHRoZSBzdGF0ZSBtYWNoaW5lIGV4ZWN1dGlvbi5cbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkQ2F0Y2goaGFuZGxlcjogSUNoYWluYWJsZSwgcHJvcHM6IENhdGNoUHJvcHMgPSB7fSk6IFRhc2tTdGF0ZUJhc2Uge1xuICAgICAgICBzdXBlci5fYWRkQ2F0Y2goaGFuZGxlci5zdGFydFN0YXRlLCBwcm9wcyk7XG4gICAgICAgIHJldHVybiB0aGlzO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBDb250aW51ZSBub3JtYWwgZXhlY3V0aW9uIHdpdGggdGhlIGdpdmVuIHN0YXRlXG4gICAgICovXG4gICAgcHVibGljIG5leHQobmV4dDogSUNoYWluYWJsZSk6IENoYWluIHtcbiAgICAgICAgc3VwZXIubWFrZU5leHQobmV4dC5zdGFydFN0YXRlKTtcbiAgICAgICAgcmV0dXJuIENoYWluLnNlcXVlbmNlKHRoaXMsIG5leHQpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZXR1cm4gdGhlIEFtYXpvbiBTdGF0ZXMgTGFuZ3VhZ2Ugb2JqZWN0IGZvciB0aGlzIHN0YXRlXG4gICAgICovXG4gICAgcHVibGljIHRvU3RhdGVKc29uKCk6IG9iamVjdCB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAuLi50aGlzLnJlbmRlck5leHRFbmQoKSxcbiAgICAgICAgICAgIC4uLnRoaXMucmVuZGVyUmV0cnlDYXRjaCgpLFxuICAgICAgICAgICAgLi4udGhpcy5yZW5kZXJUYXNrQmFzZSgpLFxuICAgICAgICAgICAgLi4udGhpcy5yZW5kZXJUYXNrKCksXG4gICAgICAgIH07XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJldHVybiB0aGUgZ2l2ZW4gbmFtZWQgbWV0cmljIGZvciB0aGlzIFRhc2tcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gc3VtIG92ZXIgNSBtaW51dGVzXG4gICAgICovXG4gICAgcHVibGljIG1ldHJpYyhtZXRyaWNOYW1lOiBzdHJpbmcsIHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgICAgICByZXR1cm4gbmV3IGNsb3Vkd2F0Y2guTWV0cmljKHtcbiAgICAgICAgICAgIG5hbWVzcGFjZTogJ0FXUy9TdGF0ZXMnLFxuICAgICAgICAgICAgbWV0cmljTmFtZSxcbiAgICAgICAgICAgIGRpbWVuc2lvbnM6IHRoaXMudGFza01ldHJpY3M/Lm1ldHJpY0RpbWVuc2lvbnMsXG4gICAgICAgICAgICBzdGF0aXN0aWM6ICdzdW0nLFxuICAgICAgICAgICAgLi4ucHJvcHMsXG4gICAgICAgIH0pLmF0dGFjaFRvKHRoaXMpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBUaGUgaW50ZXJ2YWwsIGluIG1pbGxpc2Vjb25kcywgYmV0d2VlbiB0aGUgdGltZSB0aGUgVGFzayBzdGFydHMgYW5kIHRoZSB0aW1lIGl0IGNsb3Nlcy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gYXZlcmFnZSBvdmVyIDUgbWludXRlc1xuICAgICAqL1xuICAgIHB1YmxpYyBtZXRyaWNSdW5UaW1lKHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgICAgICByZXR1cm4gdGhpcy50YXNrTWV0cmljKHRoaXMudGFza01ldHJpY3M/Lm1ldHJpY1ByZWZpeFNpbmd1bGFyLCAnUnVuVGltZScsIHsgc3RhdGlzdGljOiAnYXZnJywgLi4ucHJvcHMgfSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFRoZSBpbnRlcnZhbCwgaW4gbWlsbGlzZWNvbmRzLCBmb3Igd2hpY2ggdGhlIGFjdGl2aXR5IHN0YXlzIGluIHRoZSBzY2hlZHVsZSBzdGF0ZS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gYXZlcmFnZSBvdmVyIDUgbWludXRlc1xuICAgICAqL1xuICAgIHB1YmxpYyBtZXRyaWNTY2hlZHVsZVRpbWUocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgICAgIHJldHVybiB0aGlzLnRhc2tNZXRyaWModGhpcy50YXNrTWV0cmljcz8ubWV0cmljUHJlZml4U2luZ3VsYXIsICdTY2hlZHVsZVRpbWUnLCB7IHN0YXRpc3RpYzogJ2F2ZycsIC4uLnByb3BzIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBUaGUgaW50ZXJ2YWwsIGluIG1pbGxpc2Vjb25kcywgYmV0d2VlbiB0aGUgdGltZSB0aGUgYWN0aXZpdHkgaXMgc2NoZWR1bGVkIGFuZCB0aGUgdGltZSBpdCBjbG9zZXMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIGF2ZXJhZ2Ugb3ZlciA1IG1pbnV0ZXNcbiAgICAgKi9cbiAgICBwdWJsaWMgbWV0cmljVGltZShwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICAgICAgcmV0dXJuIHRoaXMudGFza01ldHJpYyh0aGlzLnRhc2tNZXRyaWNzPy5tZXRyaWNQcmVmaXhTaW5ndWxhciwgJ1RpbWUnLCB7IHN0YXRpc3RpYzogJ2F2ZycsIC4uLnByb3BzIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBNZXRyaWMgZm9yIHRoZSBudW1iZXIgb2YgdGltZXMgdGhpcyBhY3Rpdml0eSBpcyBzY2hlZHVsZWRcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gc3VtIG92ZXIgNSBtaW51dGVzXG4gICAgICovXG4gICAgcHVibGljIG1ldHJpY1NjaGVkdWxlZChwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICAgICAgcmV0dXJuIHRoaXMudGFza01ldHJpYyh0aGlzLnRhc2tNZXRyaWNzPy5tZXRyaWNQcmVmaXhQbHVyYWwsICdTY2hlZHVsZWQnLCBwcm9wcyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIE1ldHJpYyBmb3IgdGhlIG51bWJlciBvZiB0aW1lcyB0aGlzIGFjdGl2aXR5IHRpbWVzIG91dFxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBzdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICAgKi9cbiAgICBwdWJsaWMgbWV0cmljVGltZWRPdXQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgICAgIHJldHVybiB0aGlzLnRhc2tNZXRyaWModGhpcy50YXNrTWV0cmljcz8ubWV0cmljUHJlZml4UGx1cmFsLCAnVGltZWRPdXQnLCBwcm9wcyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIE1ldHJpYyBmb3IgdGhlIG51bWJlciBvZiB0aW1lcyB0aGlzIGFjdGl2aXR5IGlzIHN0YXJ0ZWRcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gc3VtIG92ZXIgNSBtaW51dGVzXG4gICAgICovXG4gICAgcHVibGljIG1ldHJpY1N0YXJ0ZWQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgICAgIHJldHVybiB0aGlzLnRhc2tNZXRyaWModGhpcy50YXNrTWV0cmljcz8ubWV0cmljUHJlZml4UGx1cmFsLCAnU3RhcnRlZCcsIHByb3BzKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogTWV0cmljIGZvciB0aGUgbnVtYmVyIG9mIHRpbWVzIHRoaXMgYWN0aXZpdHkgc3VjY2VlZHNcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gc3VtIG92ZXIgNSBtaW51dGVzXG4gICAgICovXG4gICAgcHVibGljIG1ldHJpY1N1Y2NlZWRlZChwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICAgICAgcmV0dXJuIHRoaXMudGFza01ldHJpYyh0aGlzLnRhc2tNZXRyaWNzPy5tZXRyaWNQcmVmaXhQbHVyYWwsICdTdWNjZWVkZWQnLCBwcm9wcyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIE1ldHJpYyBmb3IgdGhlIG51bWJlciBvZiB0aW1lcyB0aGlzIGFjdGl2aXR5IGZhaWxzXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIHN1bSBvdmVyIDUgbWludXRlc1xuICAgICAqL1xuICAgIHB1YmxpYyBtZXRyaWNGYWlsZWQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgICAgIHJldHVybiB0aGlzLnRhc2tNZXRyaWModGhpcy50YXNrTWV0cmljcz8ubWV0cmljUHJlZml4UGx1cmFsLCAnRmFpbGVkJywgcHJvcHMpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBNZXRyaWMgZm9yIHRoZSBudW1iZXIgb2YgdGltZXMgdGhlIGhlYXJ0YmVhdCB0aW1lcyBvdXQgZm9yIHRoaXMgYWN0aXZpdHlcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gc3VtIG92ZXIgNSBtaW51dGVzXG4gICAgICovXG4gICAgcHVibGljIG1ldHJpY0hlYXJ0YmVhdFRpbWVkT3V0KHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgICAgICByZXR1cm4gdGhpcy50YXNrTWV0cmljKHRoaXMudGFza01ldHJpY3M/Lm1ldHJpY1ByZWZpeFBsdXJhbCwgJ0hlYXJ0YmVhdFRpbWVkT3V0JywgcHJvcHMpO1xuICAgIH1cbiAgICBwcm90ZWN0ZWQgd2hlbkJvdW5kVG9HcmFwaChncmFwaDogU3RhdGVHcmFwaCkge1xuICAgICAgICBzdXBlci53aGVuQm91bmRUb0dyYXBoKGdyYXBoKTtcbiAgICAgICAgZm9yIChjb25zdCBwb2xpY3lTdGF0ZW1lbnQgb2YgdGhpcy50YXNrUG9saWNpZXMgfHwgW10pIHtcbiAgICAgICAgICAgIGdyYXBoLnJlZ2lzdGVyUG9saWN5U3RhdGVtZW50KHBvbGljeVN0YXRlbWVudCk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgcHJvdGVjdGVkIGFic3RyYWN0IHJlbmRlclRhc2soKTogYW55O1xuICAgIHByaXZhdGUgdGFza01ldHJpYyhwcmVmaXg6IHN0cmluZyB8IHVuZGVmaW5lZCwgc3VmZml4OiBzdHJpbmcsIHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgICAgICBpZiAocHJlZml4ID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignVGFzayBkb2VzIG5vdCBleHBvc2UgbWV0cmljcy4gVXNlIHRoZSBcXCdtZXRyaWMoKVxcJyBmdW5jdGlvbiB0byBhZGQgbWV0cmljcy4nKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gdGhpcy5tZXRyaWMocHJlZml4ICsgc3VmZml4LCBwcm9wcyk7XG4gICAgfVxuICAgIHByaXZhdGUgcmVuZGVyVGFza0Jhc2UoKSB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBUeXBlOiAnVGFzaycsXG4gICAgICAgICAgICBDb21tZW50OiB0aGlzLmNvbW1lbnQsXG4gICAgICAgICAgICBUaW1lb3V0U2Vjb25kczogdGhpcy50aW1lb3V0Py50b1NlY29uZHMoKSxcbiAgICAgICAgICAgIEhlYXJ0YmVhdFNlY29uZHM6IHRoaXMuaGVhcnRiZWF0Py50b1NlY29uZHMoKSxcbiAgICAgICAgICAgIElucHV0UGF0aDogcmVuZGVySnNvblBhdGgodGhpcy5pbnB1dFBhdGgpLFxuICAgICAgICAgICAgT3V0cHV0UGF0aDogcmVuZGVySnNvblBhdGgodGhpcy5vdXRwdXRQYXRoKSxcbiAgICAgICAgICAgIFJlc3VsdFBhdGg6IHJlbmRlckpzb25QYXRoKHRoaXMucmVzdWx0UGF0aCksXG4gICAgICAgIH07XG4gICAgfVxufVxuLyoqXG4gKiBUYXNrIE1ldHJpY3NcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBUYXNrTWV0cmljc0NvbmZpZyB7XG4gICAgLyoqXG4gICAgICogUHJlZml4IGZvciBzaW5ndWxhciBtZXRyaWMgbmFtZXMgb2YgYWN0aXZpdHkgYWN0aW9uc1xuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBzdWNoIG1ldHJpY3NcbiAgICAgKi9cbiAgICByZWFkb25seSBtZXRyaWNQcmVmaXhTaW5ndWxhcj86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBQcmVmaXggZm9yIHBsdXJhbCBtZXRyaWMgbmFtZXMgb2YgYWN0aXZpdHkgYWN0aW9uc1xuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBzdWNoIG1ldHJpY3NcbiAgICAgKi9cbiAgICByZWFkb25seSBtZXRyaWNQcmVmaXhQbHVyYWw/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIGRpbWVuc2lvbnMgdG8gYXR0YWNoIHRvIG1ldHJpY3NcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gbWV0cmljc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IG1ldHJpY0RpbWVuc2lvbnM/OiBjbG91ZHdhdGNoLkRpbWVuc2lvbkhhc2g7XG59XG4vKipcbiAqXG4gKiBBV1MgU3RlcCBGdW5jdGlvbnMgaW50ZWdyYXRlcyB3aXRoIHNlcnZpY2VzIGRpcmVjdGx5IGluIHRoZSBBbWF6b24gU3RhdGVzIExhbmd1YWdlLlxuICogWW91IGNhbiBjb250cm9sIHRoZXNlIEFXUyBzZXJ2aWNlcyB1c2luZyBzZXJ2aWNlIGludGVncmF0aW9uIHBhdHRlcm5zOlxuICpcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N0ZXAtZnVuY3Rpb25zL2xhdGVzdC9kZy9jb25uZWN0LXRvLXJlc291cmNlLmh0bWxcbiAqXG4gKi9cbmV4cG9ydCBlbnVtIEludGVncmF0aW9uUGF0dGVybiB7XG4gICAgLyoqXG4gICAgICogU3RlcCBGdW5jdGlvbnMgd2lsbCB3YWl0IGZvciBhbiBIVFRQIHJlc3BvbnNlIGFuZCB0aGVuIHByb2dyZXNzIHRvIHRoZSBuZXh0IHN0YXRlLlxuICAgICAqXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc3RlcC1mdW5jdGlvbnMvbGF0ZXN0L2RnL2Nvbm5lY3QtdG8tcmVzb3VyY2UuaHRtbCNjb25uZWN0LWRlZmF1bHRcbiAgICAgKi9cbiAgICBSRVFVRVNUX1JFU1BPTlNFID0gJ1JFUVVFU1RfUkVTUE9OU0UnLFxuICAgIC8qKlxuICAgICAqIFN0ZXAgRnVuY3Rpb25zIGNhbiB3YWl0IGZvciBhIHJlcXVlc3QgdG8gY29tcGxldGUgYmVmb3JlIHByb2dyZXNzaW5nIHRvIHRoZSBuZXh0IHN0YXRlLlxuICAgICAqXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc3RlcC1mdW5jdGlvbnMvbGF0ZXN0L2RnL2Nvbm5lY3QtdG8tcmVzb3VyY2UuaHRtbCNjb25uZWN0LXN5bmNcbiAgICAgKi9cbiAgICBSVU5fSk9CID0gJ1JVTl9KT0InLFxuICAgIC8qKlxuICAgICAqIENhbGxiYWNrIHRhc2tzIHByb3ZpZGUgYSB3YXkgdG8gcGF1c2UgYSB3b3JrZmxvdyB1bnRpbCBhIHRhc2sgdG9rZW4gaXMgcmV0dXJuZWQuXG4gICAgICogWW91IG11c3Qgc2V0IGEgdGFzayB0b2tlbiB3aGVuIHVzaW5nIHRoZSBjYWxsYmFjayBwYXR0ZXJuXG4gICAgICpcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zdGVwLWZ1bmN0aW9ucy9sYXRlc3QvZGcvY29ubmVjdC10by1yZXNvdXJjZS5odG1sI2Nvbm5lY3Qtd2FpdC10b2tlblxuICAgICAqL1xuICAgIFdBSVRfRk9SX1RBU0tfVE9LRU4gPSAnV0FJVF9GT1JfVEFTS19UT0tFTidcbn1cbiJdfQ==