"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StartExecution = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const resource_arn_suffix_1 = require("./resource-arn-suffix");
/**
 * A Step Functions Task to call StartExecution on another state machine.
 *
 * It supports three service integration patterns: FIRE_AND_FORGET, SYNC and WAIT_FOR_TASK_TOKEN.
 *
 * @deprecated - use 'StepFunctionsStartExecution'
 */
class StartExecution {
    constructor(stateMachine, props = {}) {
        this.stateMachine = stateMachine;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call Step Functions.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is missing in input (pass Context.taskToken somewhere in input)');
        }
    }
    bind(task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('states', 'startExecution', this.integrationPattern),
            policyStatements: this.createScopedAccessPolicy(task),
            parameters: {
                Input: this.props.input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            },
        };
    }
    /**
     * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
     *
     * the scoped access policy should be generated accordingly.
     *
     * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
     */
    createScopedAccessPolicy(task) {
        const stack = core_1.Stack.of(task);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['states:DescribeExecution', 'states:StopExecution'],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        sep: ':',
                        resourceName: `${stack.parseArn(this.stateMachine.stateMachineArn, ':').resourceName}*`,
                    })],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    })],
            }));
        }
        return policyStatements;
    }
}
exports.StartExecution = StartExecution;
//# sourceMappingURL=data:application/json;base64,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