"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaInvocationType = exports.LambdaInvoke = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const task_utils_1 = require("../private/task-utils");
/**
 * Invoke a Lambda function as a Task
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
 */
class LambdaInvoke extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, LambdaInvoke.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.payload)) {
            throw new Error('Task Token is required in `payload` for callback. Use Context.taskToken to set the token.');
        }
        this.taskMetrics = {
            metricPrefixSingular: 'LambdaFunction',
            metricPrefixPlural: 'LambdaFunctions',
            metricDimensions: {
                LambdaFunctionArn: this.props.lambdaFunction.functionArn,
                ...(this.props.qualifier && { Qualifier: this.props.qualifier }),
            },
        };
        this.taskPolicies = [
            new iam.PolicyStatement({
                resources: [this.props.lambdaFunction.functionArn],
                actions: ['lambda:InvokeFunction'],
            }),
        ];
    }
    /**
     * Provides the Lambda Invoke service integration task configuration
     */
    renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('lambda', 'invoke', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                FunctionName: this.props.lambdaFunction.functionArn,
                Payload: this.props.payload ? this.props.payload.value : sfn.TaskInput.fromDataAt('$').value,
                InvocationType: this.props.invocationType,
                ClientContext: this.props.clientContext,
                Qualifier: this.props.qualifier,
            }),
        };
    }
}
exports.LambdaInvoke = LambdaInvoke;
LambdaInvoke.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
/**
 * Invocation type of a Lambda
 */
var LambdaInvocationType;
(function (LambdaInvocationType) {
    /**
     * Invoke the function synchronously.
     *
     * Keep the connection open until the function returns a response or times out.
     * The API response includes the function response and additional data.
     */
    LambdaInvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    /**
     * Invoke the function asynchronously.
     *
     * Send events that fail multiple times to the function's dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    LambdaInvocationType["EVENT"] = "Event";
    /**
     * Validate parameter values and verify that the user or role has permission to invoke the function.
     */
    LambdaInvocationType["DRY_RUN"] = "DryRun";
})(LambdaInvocationType = exports.LambdaInvocationType || (exports.LambdaInvocationType = {}));
//# sourceMappingURL=data:application/json;base64,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