"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsRunTaskBase = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * A StepFunctions Task to run a Task on ECS or Fargate
 */
class EcsRunTaskBase {
    constructor(props) {
        this.props = props;
        /**
         * Manage allowed network traffic for this service
         */
        this.connections = new ec2.Connections();
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            sfn.ServiceIntegrationPattern.SYNC,
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call ECS.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.containerOverrides)) {
            throw new Error('Task Token is missing in containerOverrides (pass Context.taskToken somewhere in containerOverrides)');
        }
        for (const override of this.props.containerOverrides || []) {
            const name = override.containerName;
            if (!cdk.Token.isUnresolved(name)) {
                const cont = this.props.taskDefinition.node.tryFindChild(name);
                if (!cont) {
                    throw new Error(`Overrides mention container with name '${name}', but no such container in task definition`);
                }
            }
        }
    }
    bind(task) {
        if (this.networkConfiguration !== undefined) {
            // Make sure we have a security group if we're using AWSVPC networking
            if (this.securityGroup === undefined) {
                this.securityGroup = new ec2.SecurityGroup(task, 'SecurityGroup', { vpc: this.props.cluster.vpc });
            }
            this.connections.addSecurityGroup(this.securityGroup);
        }
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('ecs', 'runTask', this.integrationPattern),
            parameters: {
                Cluster: this.props.cluster.clusterArn,
                TaskDefinition: this.props.taskDefinition.taskDefinitionArn,
                NetworkConfiguration: this.networkConfiguration,
                Overrides: renderOverrides(this.props.containerOverrides),
                ...this.props.parameters,
            },
            policyStatements: this.makePolicyStatements(task),
        };
    }
    configureAwsVpcNetworking(vpc, assignPublicIp, subnetSelection, securityGroup) {
        if (subnetSelection === undefined) {
            subnetSelection = { subnetType: assignPublicIp ? ec2.SubnetType.PUBLIC : ec2.SubnetType.PRIVATE };
        }
        // If none is given here, one will be created later on during bind()
        this.securityGroup = securityGroup;
        this.networkConfiguration = {
            AwsvpcConfiguration: {
                AssignPublicIp: assignPublicIp !== undefined ? (assignPublicIp ? 'ENABLED' : 'DISABLED') : undefined,
                Subnets: vpc.selectSubnets(subnetSelection).subnetIds,
                SecurityGroups: cdk.Lazy.listValue({ produce: () => [this.securityGroup.securityGroupId] }),
            },
        };
    }
    makePolicyStatements(task) {
        const stack = cdk.Stack.of(task);
        // https://docs.aws.amazon.com/step-functions/latest/dg/ecs-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.props.taskDefinition.taskDefinitionArn],
            }),
            new iam.PolicyStatement({
                actions: ['ecs:StopTask', 'ecs:DescribeTasks'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: cdk.Lazy.listValue({ produce: () => this.taskExecutionRoles().map(r => r.roleArn) }),
            }),
        ];
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForECSTaskRule',
                    })],
            }));
        }
        return policyStatements;
    }
    taskExecutionRoles() {
        // Need to be able to pass both Task and Execution role, apparently
        const ret = new Array();
        ret.push(this.props.taskDefinition.taskRole);
        if (this.props.taskDefinition.executionRole) {
            ret.push(this.props.taskDefinition.executionRole);
        }
        return ret;
    }
}
exports.EcsRunTaskBase = EcsRunTaskBase;
function renderOverrides(containerOverrides) {
    if (!containerOverrides) {
        return undefined;
    }
    const ret = new Array();
    for (const override of containerOverrides) {
        ret.push({
            Name: override.containerName,
            Command: override.command,
            Cpu: override.cpu,
            Memory: override.memoryLimit,
            MemoryReservation: override.memoryReservation,
            Environment: override.environment && override.environment.map(e => ({
                Name: e.name,
                Value: e.value,
            })),
        });
    }
    return { ContainerOverrides: ret };
}
//# sourceMappingURL=data:application/json;base64,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