"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheckType = exports.RoutingPolicy = exports.DnsRecordType = exports.Service = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const alias_target_instance_1 = require("./alias-target-instance");
const cname_instance_1 = require("./cname-instance");
const ip_instance_1 = require("./ip-instance");
const namespace_1 = require("./namespace");
const non_ip_instance_1 = require("./non-ip-instance");
const servicediscovery_generated_1 = require("./servicediscovery.generated");
class ServiceBase extends core_1.Resource {
}
/**
 * Define a CloudMap Service
 */
class Service extends ServiceBase {
    constructor(scope, id, props) {
        super(scope, id);
        const namespaceType = props.namespace.type;
        // Validations
        if (namespaceType === namespace_1.NamespaceType.HTTP && (props.routingPolicy || props.dnsRecordType)) {
            throw new Error('Cannot specify `routingPolicy` or `dnsRecord` when using an HTTP namespace.');
        }
        if (props.healthCheck && props.customHealthCheck) {
            throw new Error('Cannot specify both `healthCheckConfig` and `healthCheckCustomConfig`.');
        }
        if (namespaceType === namespace_1.NamespaceType.DNS_PRIVATE && props.healthCheck) {
            throw new Error('Cannot specify `healthCheckConfig` for a Private DNS namespace.');
        }
        if (props.routingPolicy === RoutingPolicy.MULTIVALUE
            && props.dnsRecordType === DnsRecordType.CNAME) {
            throw new Error('Cannot use `CNAME` record when routing policy is `Multivalue`.');
        }
        // Additional validation for eventual attachment of LBs
        // The same validation happens later on during the actual attachment
        // of LBs, but we need the property for the correct configuration of
        // routingPolicy anyway, so might as well do the validation as well.
        if (props.routingPolicy === RoutingPolicy.MULTIVALUE
            && props.loadBalancer) {
            throw new Error('Cannot register loadbalancers when routing policy is `Multivalue`.');
        }
        if (props.healthCheck
            && props.healthCheck.type === HealthCheckType.TCP
            && props.healthCheck.resourcePath) {
            throw new Error('Cannot specify `resourcePath` when using a `TCP` health check.');
        }
        // Set defaults where necessary
        const routingPolicy = (props.dnsRecordType === DnsRecordType.CNAME) || props.loadBalancer
            ? RoutingPolicy.WEIGHTED
            : RoutingPolicy.MULTIVALUE;
        const dnsRecordType = props.dnsRecordType || DnsRecordType.A;
        if (props.loadBalancer
            && (!(dnsRecordType === DnsRecordType.A
                || dnsRecordType === DnsRecordType.AAAA
                || dnsRecordType === DnsRecordType.A_AAAA))) {
            throw new Error('Must support `A` or `AAAA` records to register loadbalancers.');
        }
        const dnsConfig = props.namespace.type === namespace_1.NamespaceType.HTTP
            ? undefined
            : {
                dnsRecords: renderDnsRecords(dnsRecordType, props.dnsTtl),
                namespaceId: props.namespace.namespaceId,
                routingPolicy,
            };
        const healthCheckConfigDefaults = {
            type: HealthCheckType.HTTP,
            failureThreshold: 1,
            resourcePath: props.healthCheck && props.healthCheck.type !== HealthCheckType.TCP
                ? '/'
                : undefined,
        };
        const healthCheckConfig = props.healthCheck && { ...healthCheckConfigDefaults, ...props.healthCheck };
        const healthCheckCustomConfig = props.customHealthCheck;
        // Create service
        const service = new servicediscovery_generated_1.CfnService(this, 'Resource', {
            name: props.name,
            description: props.description,
            dnsConfig,
            healthCheckConfig,
            healthCheckCustomConfig,
            namespaceId: props.namespace.namespaceId,
        });
        this.serviceName = service.attrName;
        this.serviceArn = service.attrArn;
        this.serviceId = service.attrId;
        this.namespace = props.namespace;
        this.dnsRecordType = dnsRecordType;
        this.routingPolicy = routingPolicy;
    }
    static fromServiceAttributes(scope, id, attrs) {
        class Import extends ServiceBase {
            constructor() {
                super(...arguments);
                this.namespace = attrs.namespace;
                this.serviceId = attrs.serviceId;
                this.serviceArn = attrs.serviceArn;
                this.dnsRecordType = attrs.dnsRecordType;
                this.routingPolicy = attrs.routingPolicy;
                this.serviceName = attrs.serviceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Registers an ELB as a new instance with unique name instanceId in this service.
     */
    registerLoadBalancer(id, loadBalancer, customAttributes) {
        return new alias_target_instance_1.AliasTargetInstance(this, id, {
            service: this,
            dnsName: loadBalancer.loadBalancerDnsName,
            customAttributes,
        });
    }
    /**
     * Registers a resource that is accessible using values other than an IP address or a domain name (CNAME).
     */
    registerNonIpInstance(id, props) {
        return new non_ip_instance_1.NonIpInstance(this, id, {
            service: this,
            ...props,
        });
    }
    /**
     * Registers a resource that is accessible using an IP address.
     */
    registerIpInstance(id, props) {
        return new ip_instance_1.IpInstance(this, id, {
            service: this,
            ...props,
        });
    }
    /**
     * Registers a resource that is accessible using a CNAME.
     */
    registerCnameInstance(id, props) {
        return new cname_instance_1.CnameInstance(this, id, {
            service: this,
            ...props,
        });
    }
}
exports.Service = Service;
function renderDnsRecords(dnsRecordType, dnsTtl = core_1.Duration.minutes(1)) {
    const ttl = dnsTtl.toSeconds();
    if (dnsRecordType === DnsRecordType.A_AAAA) {
        return [{
                type: DnsRecordType.A,
                ttl,
            }, {
                type: DnsRecordType.AAAA,
                ttl,
            }];
    }
    else {
        return [{ type: dnsRecordType, ttl }];
    }
}
var DnsRecordType;
(function (DnsRecordType) {
    /**
     * An A record
     */
    DnsRecordType["A"] = "A";
    /**
     * An AAAA record
     */
    DnsRecordType["AAAA"] = "AAAA";
    /**
     * Both an A and AAAA record
     */
    DnsRecordType["A_AAAA"] = "A, AAAA";
    /**
     * A Srv record
     */
    DnsRecordType["SRV"] = "SRV";
    /**
     * A CNAME record
     */
    DnsRecordType["CNAME"] = "CNAME";
})(DnsRecordType = exports.DnsRecordType || (exports.DnsRecordType = {}));
var RoutingPolicy;
(function (RoutingPolicy) {
    /**
     * Route 53 returns the applicable value from one randomly selected instance from among the instances that you
     * registered using the same service.
     */
    RoutingPolicy["WEIGHTED"] = "WEIGHTED";
    /**
     * If you define a health check for the service and the health check is healthy, Route 53 returns the applicable value
     * for up to eight instances.
     */
    RoutingPolicy["MULTIVALUE"] = "MULTIVALUE";
})(RoutingPolicy = exports.RoutingPolicy || (exports.RoutingPolicy = {}));
var HealthCheckType;
(function (HealthCheckType) {
    /**
     * Route 53 tries to establish a TCP connection. If successful, Route 53 submits an HTTP request and waits for an HTTP
     * status code of 200 or greater and less than 400.
     */
    HealthCheckType["HTTP"] = "HTTP";
    /**
     * Route 53 tries to establish a TCP connection. If successful, Route 53 submits an HTTPS request and waits for an
     * HTTP status code of 200 or greater and less than 400.  If you specify HTTPS for the value of Type, the endpoint
     * must support TLS v1.0 or later.
     */
    HealthCheckType["HTTPS"] = "HTTPS";
    /**
     * Route 53 tries to establish a TCP connection.
     * If you specify TCP for Type, don't specify a value for ResourcePath.
     */
    HealthCheckType["TCP"] = "TCP";
})(HealthCheckType = exports.HealthCheckType || (exports.HealthCheckType = {}));
//# sourceMappingURL=data:application/json;base64,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