"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateHostedZone = exports.PublicHostedZone = exports.HostedZone = void 0;
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const record_set_1 = require("./record-set");
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
/**
 * Container for records, and records contain information about how to route traffic for a
 * specific domain, such as example.com and its subdomains (acme.example.com, zenith.example.com)
 */
class HostedZone extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * VPCs to which this hosted zone will be added
         */
        this.vpcs = new Array();
        util_1.validateZoneName(props.zoneName);
        const resource = new route53_generated_1.CfnHostedZone(this, 'Resource', {
            name: props.zoneName + '.',
            hostedZoneConfig: props.comment ? { comment: props.comment } : undefined,
            queryLoggingConfig: props.queryLogsLogGroupArn ? { cloudWatchLogsLogGroupArn: props.queryLogsLogGroupArn } : undefined,
            vpcs: core_1.Lazy.anyValue({ produce: () => this.vpcs.length === 0 ? undefined : this.vpcs }),
        });
        this.hostedZoneId = resource.ref;
        this.hostedZoneNameServers = resource.attrNameServers;
        this.zoneName = props.zoneName;
        for (const vpc of props.vpcs || []) {
            this.addVpc(vpc);
        }
    }
    get hostedZoneArn() {
        return util_1.makeHostedZoneArn(this, this.hostedZoneId);
    }
    /**
     * Import a Route 53 hosted zone defined either outside the CDK, or in a different CDK stack
     *
     * @param scope the parent Construct for this Construct
     * @param id  the logical name of this Construct
     * @param hostedZoneId the ID of the hosted zone to import
     */
    static fromHostedZoneId(scope, id, hostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = hostedZoneId;
            }
            get zoneName() {
                throw new Error('HostedZone.fromHostedZoneId doesn\'t support "zoneName"');
            }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports a hosted zone from another stack.
     */
    static fromHostedZoneAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = attrs.hostedZoneId;
                this.zoneName = attrs.zoneName;
            }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Lookup a hosted zone in the current account/region based on query parameters.
     */
    static fromLookup(scope, id, query) {
        const DEFAULT_HOSTED_ZONE = {
            Id: 'DUMMY',
            Name: query.domainName,
        };
        const response = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.HOSTED_ZONE_PROVIDER,
            dummyValue: DEFAULT_HOSTED_ZONE,
            props: query,
        }).value;
        // CDK handles the '.' at the end, so remove it here
        if (response.Name.endsWith('.')) {
            response.Name = response.Name.substring(0, response.Name.length - 1);
        }
        response.Id = response.Id.replace('/hostedzone/', '');
        return HostedZone.fromHostedZoneAttributes(scope, id, {
            hostedZoneId: response.Id,
            zoneName: response.Name,
        });
    }
    /**
     * Add another VPC to this private hosted zone.
     *
     * @param vpc the other VPC to add.
     */
    addVpc(vpc) {
        this.vpcs.push({ vpcId: vpc.vpcId, vpcRegion: core_1.Stack.of(vpc).region });
    }
}
exports.HostedZone = HostedZone;
/**
 * Create a Route53 public hosted zone.
 *
 * @resource AWS::Route53::HostedZone
 */
class PublicHostedZone extends HostedZone {
    /**
     * Import a Route 53 public hosted zone defined either outside the CDK, or in a different CDK stack
     *
     * @param scope the parent Construct for this Construct
     * @param id the logical name of this Construct
     * @param publicHostedZoneId the ID of the public hosted zone to import
     */
    static fromPublicHostedZoneId(scope, id, publicHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = publicHostedZoneId;
            }
            get zoneName() { throw new Error('cannot retrieve "zoneName" from an an imported hosted zone'); }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        if (props.caaAmazon) {
            new record_set_1.CaaAmazonRecord(this, 'CaaAmazon', {
                zone: this,
            });
        }
    }
    addVpc(_vpc) {
        throw new Error('Cannot associate public hosted zones with a VPC');
    }
    /**
     * Adds a delegation from this zone to a designated zone.
     *
     * @param delegate the zone being delegated to.
     * @param opts     options for creating the DNS record, if any.
     */
    addDelegation(delegate, opts = {}) {
        new record_set_1.ZoneDelegationRecord(this, `${this.zoneName} -> ${delegate.zoneName}`, {
            zone: this,
            recordName: delegate.zoneName,
            nameServers: delegate.hostedZoneNameServers,
            comment: opts.comment,
            ttl: opts.ttl,
        });
    }
}
exports.PublicHostedZone = PublicHostedZone;
/**
 * Create a Route53 private hosted zone for use in one or more VPCs.
 *
 * Note that `enableDnsHostnames` and `enableDnsSupport` must have been enabled
 * for the VPC you're configuring for private hosted zones.
 *
 * @resource AWS::Route53::HostedZone
 */
class PrivateHostedZone extends HostedZone {
    /**
     * Import a Route 53 private hosted zone defined either outside the CDK, or in a different CDK stack
     *
     * @param scope the parent Construct for this Construct
     * @param id the logical name of this Construct
     * @param privateHostedZoneId the ID of the private hosted zone to import
     */
    static fromPrivateHostedZoneId(scope, id, privateHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = privateHostedZoneId;
            }
            get zoneName() { throw new Error('cannot retrieve "zoneName" from an an imported hosted zone'); }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        this.addVpc(props.vpc);
    }
}
exports.PrivateHostedZone = PrivateHostedZone;
//# sourceMappingURL=data:application/json;base64,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