"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpsRedirect = void 0;
const aws_certificatemanager_1 = require("../../aws-certificatemanager"); // Automatically re-written from '@aws-cdk/aws-certificatemanager'
const aws_cloudfront_1 = require("../../aws-cloudfront"); // Automatically re-written from '@aws-cdk/aws-cloudfront'
const aws_route53_1 = require("../../aws-route53"); // Automatically re-written from '@aws-cdk/aws-route53'
const aws_route53_targets_1 = require("../../aws-route53-targets"); // Automatically re-written from '@aws-cdk/aws-route53-targets'
const aws_s3_1 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const crypto = require("crypto");
/**
 * Allows creating a domainA -> domainB redirect using CloudFront and S3.
 * You can specify multiple domains to be redirected.
 */
class HttpsRedirect extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const domainNames = props.recordNames || [props.zone.zoneName];
        const redirectCertArn = props.certificate ? props.certificate.certificateArn : new aws_certificatemanager_1.DnsValidatedCertificate(this, 'RedirectCertificate', {
            domainName: domainNames[0],
            subjectAlternativeNames: domainNames,
            hostedZone: props.zone,
            region: 'us-east-1',
        }).certificateArn;
        const redirectBucket = new aws_s3_1.Bucket(this, 'RedirectBucket', {
            websiteRedirect: {
                hostName: props.targetDomain,
                protocol: aws_s3_1.RedirectProtocol.HTTPS,
            },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        const redirectDist = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'RedirectDistribution', {
            defaultRootObject: '',
            originConfigs: [{
                    behaviors: [{ isDefaultBehavior: true }],
                    customOriginSource: {
                        domainName: redirectBucket.bucketWebsiteDomainName,
                        originProtocolPolicy: aws_cloudfront_1.OriginProtocolPolicy.HTTP_ONLY,
                    },
                }],
            aliasConfiguration: {
                acmCertRef: redirectCertArn,
                names: domainNames,
            },
            comment: `Redirect to ${props.targetDomain} from ${domainNames.join(', ')}`,
            priceClass: aws_cloudfront_1.PriceClass.PRICE_CLASS_ALL,
            viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        });
        domainNames.forEach((domainName) => {
            const hash = crypto.createHash('md5').update(domainName).digest('hex').substr(0, 6);
            new aws_route53_1.ARecord(this, `RedirectAliasRecord${hash}`, {
                recordName: domainName,
                zone: props.zone,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(redirectDist)),
            });
        });
    }
}
exports.HttpsRedirect = HttpsRedirect;
//# sourceMappingURL=data:application/json;base64,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