"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifyCodeConfig = exports.Function = exports.Tracing = void 0;
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const sqs = require("../../aws-sqs"); // Automatically re-written from '@aws-cdk/aws-sqs'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const function_base_1 = require("./function-base");
const function_hash_1 = require("./function-hash");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const log_retention_1 = require("./log-retention");
/**
 * X-Ray Tracing Modes (https://docs.aws.amazon.com/lambda/latest/dg/API_TracingConfig.html)
 */
var Tracing;
(function (Tracing) {
    /**
     * Lambda will respect any tracing header it receives from an upstream service.
     * If no tracing header is received, Lambda will call X-Ray for a tracing decision.
     */
    Tracing["ACTIVE"] = "Active";
    /**
     * Lambda will only trace the request from an upstream service
     * if it contains a tracing header with "sampled=1"
     */
    Tracing["PASS_THROUGH"] = "PassThrough";
    /**
     * Lambda will not trace any request.
     */
    Tracing["DISABLED"] = "Disabled";
})(Tracing = exports.Tracing || (exports.Tracing = {}));
/**
 * Deploys a file from from inside the construct library as a function.
 *
 * The supplied file is subject to the 4096 bytes limit of being embedded in a
 * CloudFormation template.
 *
 * The construct includes an associated role with the lambda.
 *
 * This construct does not yet reproduce all features from the underlying resource
 * library.
 */
class Function extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.functionName,
        });
        this.permissionsNode = this.node;
        this.canCreatePermissions = true;
        this.layers = [];
        this.environment = props.environment || {};
        const managedPolicies = new Array();
        // the arn is in the form of - arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
        managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies,
        });
        this.grantPrincipal = this.role;
        for (const statement of (props.initialPolicy || [])) {
            this.role.addToPolicy(statement);
        }
        const code = props.code.bind(this);
        verifyCodeConfig(code, props.runtime);
        this.deadLetterQueue = this.buildDeadLetterQueue(props);
        const resource = new lambda_generated_1.CfnFunction(this, 'Resource', {
            functionName: this.physicalName,
            description: props.description,
            code: {
                s3Bucket: code.s3Location && code.s3Location.bucketName,
                s3Key: code.s3Location && code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location && code.s3Location.objectVersion,
                zipFile: code.inlineCode,
            },
            layers: core_1.Lazy.listValue({ produce: () => this.layers.map(layer => layer.layerVersionArn) }, { omitEmpty: true }),
            handler: props.handler,
            timeout: props.timeout && props.timeout.toSeconds(),
            runtime: props.runtime.name,
            role: this.role.roleArn,
            environment: core_1.Lazy.anyValue({ produce: () => this.renderEnvironment() }),
            memorySize: props.memorySize,
            vpcConfig: this.configureVpc(props),
            deadLetterConfig: this.buildDeadLetterConfig(this.deadLetterQueue),
            tracingConfig: this.buildTracingConfig(props),
            reservedConcurrentExecutions: props.reservedConcurrentExecutions,
        });
        resource.node.addDependency(this.role);
        this.functionName = this.getResourceNameAttribute(resource.ref);
        this.functionArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'lambda',
            resource: 'function',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.runtime = props.runtime;
        if (props.layers) {
            this.addLayers(...props.layers);
        }
        for (const event of props.events || []) {
            this.addEventSource(event);
        }
        // Log retention
        if (props.logRetention) {
            const logretention = new log_retention_1.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: props.logRetention,
                role: props.logRetentionRole,
                logRetentionRetryOptions: props.logRetentionRetryOptions,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, 'LogGroup', logretention.logGroupArn);
        }
        props.code.bindToResource(resource);
        // Event Invoke Config
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        this.currentVersionOptions = props.currentVersionOptions;
    }
    /**
     * Returns a `lambda.Version` which represents the current version of this
     * Lambda function. A new version will be created every time the function's
     * configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.Function`.
     */
    get currentVersion() {
        if (this._currentVersion) {
            return this._currentVersion;
        }
        this._currentVersion = new lambda_version_1.Version(this, 'CurrentVersion', {
            lambda: this,
            ...this.currentVersionOptions,
        });
        return this._currentVersion;
    }
    static fromFunctionArn(scope, id, functionArn) {
        return Function.fromFunctionAttributes(scope, id, { functionArn });
    }
    /**
     * Creates a Lambda function object which represents a function not defined
     * within this stack.
     *
     * @param scope The parent construct
     * @param id The name of the lambda construct
     * @param attrs the attributes of the function to import
     */
    static fromFunctionAttributes(scope, id, attrs) {
        const functionArn = attrs.functionArn;
        const functionName = extractNameFromArn(attrs.functionArn);
        const role = attrs.role;
        class Import extends function_base_1.FunctionBase {
            constructor(s, i) {
                super(s, i);
                this.functionName = functionName;
                this.functionArn = functionArn;
                this.role = role;
                this.permissionsNode = this.node;
                this.canCreatePermissions = false;
                this.grantPrincipal = role || new iam.UnknownPrincipal({ resource: this });
                if (attrs.securityGroup) {
                    this._connections = new ec2.Connections({
                        securityGroups: [attrs.securityGroup],
                    });
                }
                else if (attrs.securityGroupId) {
                    this._connections = new ec2.Connections({
                        securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)],
                    });
                }
            }
        }
        return new Import(scope, id);
    }
    /**
     * Return the given named metric for this Lambda
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Lambda',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the number of Errors executing all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllErrors(props) {
        return this.metricAll('Errors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the Duration executing all Lambdas
     *
     * @default average over 5 minutes
     */
    static metricAllDuration(props) {
        return this.metricAll('Duration', props);
    }
    /**
     * Metric for the number of invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllInvocations(props) {
        return this.metricAll('Invocations', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of throttled invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllThrottles(props) {
        return this.metricAll('Throttles', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllConcurrentExecutions(props) {
        // Mini-FAQ: why max? This metric is a gauge that is emitted every
        // minute, so either max or avg or a percentile make sense (but sum
        // doesn't). Max is more sensitive to spiky load changes which is
        // probably what you're interested in if you're looking at this metric
        // (Load spikes may lead to concurrent execution errors that would
        // otherwise not be visible in the avg)
        return this.metricAll('ConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Metric for the number of unreserved concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllUnreservedConcurrentExecutions(props) {
        return this.metricAll('UnreservedConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Adds an environment variable to this Lambda function.
     * If this is a ref to a Lambda function, this operation results in a no-op.
     * @param key The environment variable key.
     * @param value The environment variable's value.
     */
    addEnvironment(key, value) {
        this.environment[key] = value;
        return this;
    }
    /**
     * Adds one or more Lambda Layers to this Lambda function.
     *
     * @param layers the layers to be added.
     *
     * @throws if there are already 5 layers on this function, or the layer is incompatible with this function's runtime.
     */
    addLayers(...layers) {
        for (const layer of layers) {
            if (this.layers.length === 5) {
                throw new Error('Unable to add layer: this lambda function already uses 5 layers.');
            }
            if (layer.compatibleRuntimes && !layer.compatibleRuntimes.find(runtime => runtime.runtimeEquals(this.runtime))) {
                const runtimes = layer.compatibleRuntimes.map(runtime => runtime.name).join(', ');
                throw new Error(`This lambda function uses a runtime that is incompatible with this layer (${this.runtime.name} is not in [${runtimes}])`);
            }
            this.layers.push(layer);
        }
    }
    /**
     * Add a new version for this Lambda
     *
     * If you want to deploy through CloudFormation and use aliases, you need to
     * add a new version (with a new name) to your Lambda every time you want to
     * deploy an update. An alias can then refer to the newly created Version.
     *
     * All versions should have distinct names, and you should not delete versions
     * as long as your Alias needs to refer to them.
     *
     * @param name A unique name for this version.
     * @param codeSha256 The SHA-256 hash of the most recently deployed Lambda
     *  source code, or omit to skip validation.
     * @param description A description for this version.
     * @param provisionedExecutions A provisioned concurrency configuration for a
     * function's version.
     * @param asyncInvokeConfig configuration for this version when it is invoked
     * asynchronously.
     * @returns A new Version object.
     *
     * @deprecated This method will create an AWS::Lambda::Version resource which
     * snapshots the AWS Lambda function *at the time of its creation* and it
     * won't get updated when the function changes. Instead, use
     * `this.currentVersion` to obtain a reference to a version resource that gets
     * automatically recreated when the function configuration (or code) changes.
     */
    addVersion(name, codeSha256, description, provisionedExecutions, asyncInvokeConfig = {}) {
        return new lambda_version_1.Version(this, 'Version' + name, {
            lambda: this,
            codeSha256,
            description,
            provisionedConcurrentExecutions: provisionedExecutions,
            ...asyncInvokeConfig,
        });
    }
    /**
     * The LogGroup where the Lambda function's logs are made available.
     *
     * If either `logRetention` is set or this property is called, a CloudFormation custom resource is added to the stack that
     * pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the correct log retention
     * period (never expire, by default).
     *
     * Further, if the log group already exists and the `logRetention` is not set, the custom resource will reset the log retention
     * to never expire even if it was configured with a different value.
     */
    get logGroup() {
        if (!this._logGroup) {
            const logretention = new log_retention_1.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: logs.RetentionDays.INFINITE,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, `${this.node.id}-LogGroup`, logretention.logGroupArn);
        }
        return this._logGroup;
    }
    prepare() {
        super.prepare();
        // if we have a current version resource, override it's logical id
        // so that it includes the hash of the function code and it's configuration.
        if (this._currentVersion) {
            const stack = core_1.Stack.of(this);
            const cfn = this._currentVersion.node.defaultChild;
            const originalLogicalId = stack.resolve(cfn.logicalId);
            const hash = function_hash_1.calculateFunctionHash(this);
            const logicalId = function_hash_1.trimFromStart(originalLogicalId, 255 - 32);
            cfn.overrideLogicalId(`${logicalId}${hash}`);
        }
    }
    renderEnvironment() {
        if (!this.environment || Object.keys(this.environment).length === 0) {
            return undefined;
        }
        // for backwards compatibility we do not sort environment variables in case
        // _currentVersion is not defined. otherwise, this would have invalidated
        // the template, and for example, may cause unneeded updates for nested
        // stacks.
        if (!this._currentVersion) {
            return {
                variables: this.environment,
            };
        }
        // sort environment so the hash of the function used to create
        // `currentVersion` is not affected by key order (this is how lambda does
        // it).
        const variables = {};
        for (const key of Object.keys(this.environment).sort()) {
            variables[key] = this.environment[key];
        }
        return { variables };
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * Lambda creation properties.
     */
    configureVpc(props) {
        if ((props.securityGroup || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if (props.securityGroup && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroup && props.securityGroups) {
            throw new Error('Only one of the function props, securityGroup or securityGroups, is allowed');
        }
        if (props.securityGroups) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic security group for Lambda Function ' + this.node.uniqueId,
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        // Pick subnets, make sure they're not Public. Routing through an IGW
        // won't work because the ENIs don't get a Public IP.
        // Why are we not simply forcing vpcSubnets? Because you might still be choosing
        // Isolated networks or selecting among 2 sets of Private subnets by name.
        const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
        const publicSubnetIds = new Set(props.vpc.publicSubnets.map(s => s.subnetId));
        for (const subnetId of subnetIds) {
            if (publicSubnetIds.has(subnetId)) {
                throw new Error('Not possible to place Lambda Functions in a Public subnet');
            }
        }
        // List can't be empty here, if we got this far you intended to put your Lambda
        // in subnets. We're going to guarantee that we get the nice error message by
        // making VpcNetwork do the selection again.
        return {
            subnetIds,
            securityGroupIds: securityGroups.map(sg => sg.securityGroupId),
        };
    }
    buildDeadLetterQueue(props) {
        if (props.deadLetterQueue && props.deadLetterQueueEnabled === false) {
            throw Error('deadLetterQueue defined but deadLetterQueueEnabled explicitly set to false');
        }
        if (!props.deadLetterQueue && !props.deadLetterQueueEnabled) {
            return undefined;
        }
        const deadLetterQueue = props.deadLetterQueue || new sqs.Queue(this, 'DeadLetterQueue', {
            retentionPeriod: core_1.Duration.days(14),
        });
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sqs:SendMessage'],
            resources: [deadLetterQueue.queueArn],
        }));
        return deadLetterQueue;
    }
    buildDeadLetterConfig(deadLetterQueue) {
        if (deadLetterQueue) {
            return {
                targetArn: deadLetterQueue.queueArn,
            };
        }
        else {
            return undefined;
        }
    }
    buildTracingConfig(props) {
        if (props.tracing === undefined || props.tracing === Tracing.DISABLED) {
            return undefined;
        }
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['xray:PutTraceSegments', 'xray:PutTelemetryRecords'],
            resources: ['*'],
        }));
        return {
            mode: props.tracing,
        };
    }
}
exports.Function = Function;
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the function
 * name from the ARN.
 *
 * Function ARNs look like this:
 *
 *   arn:aws:lambda:region:account-id:function:function-name
 *
 * ..which means that in order to extract the `function-name` component from the ARN, we can
 * split the ARN using ":" and select the component in index 6.
 *
 * @returns `FnSelect(6, FnSplit(':', arn))`
 */
function extractNameFromArn(arn) {
    return core_1.Fn.select(6, core_1.Fn.split(':', arn));
}
function verifyCodeConfig(code, runtime) {
    // mutually exclusive
    if ((!code.inlineCode && !code.s3Location) || (code.inlineCode && code.s3Location)) {
        throw new Error('lambda.Code must specify one of "inlineCode" or "s3Location" but not both');
    }
    // if this is inline code, check that the runtime supports
    if (code.inlineCode && !runtime.supportsInlineCode) {
        throw new Error(`Inline source not allowed for ${runtime.name}`);
    }
}
exports.verifyCodeConfig = verifyCodeConfig;
//# sourceMappingURL=data:application/json;base64,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