"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventBus = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const events_generated_1 = require("./events.generated");
/**
 * Define an EventBridge EventBus
 *
 * @resource AWS::Events::EventBus
 */
class EventBus extends core_1.Resource {
    constructor(scope, id, props) {
        const { eventBusName, eventSourceName } = EventBus.eventBusProps(core_1.Lazy.stringValue({ produce: () => this.node.uniqueId }), props);
        super(scope, id, { physicalName: eventBusName });
        const eventBus = new events_generated_1.CfnEventBus(this, 'Resource', {
            name: eventBusName,
            eventSourceName,
        });
        this.eventBusArn = this.getResourceArnAttribute(eventBus.attrArn, {
            service: 'events',
            resource: 'event-bus',
            resourceName: eventBus.name,
        });
        this.eventBusName = this.getResourceNameAttribute(eventBus.ref);
        this.eventBusPolicy = eventBus.attrPolicy;
        this.eventSourceName = eventBus.eventSourceName;
    }
    /**
     * Import an existing event bus resource
     * @param scope Parent construct
     * @param id Construct ID
     * @param eventBusArn ARN of imported event bus
     */
    static fromEventBusArn(scope, id, eventBusArn) {
        const parts = core_1.Stack.of(scope).parseArn(eventBusArn);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.eventBusArn = eventBusArn;
                this.eventBusName = parts.resourceName || '';
                this.eventBusPolicy = '';
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing event bus resource
     * @param scope Parent construct
     * @param id Construct ID
     * @param attrs Imported event bus properties
     */
    static fromEventBusAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.eventBusArn = attrs.eventBusArn;
                this.eventBusName = attrs.eventBusName;
                this.eventBusPolicy = attrs.eventBusPolicy;
                this.eventSourceName = attrs.eventSourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Permits an IAM Principal to send custom events to EventBridge
     * so that they can be matched to rules.
     *
     * @param grantee The principal (no-op if undefined)
     */
    static grantPutEvents(grantee) {
        // It's currently not possible to restrict PutEvents to specific resources.
        // See https://docs.aws.amazon.com/eventbridge/latest/userguide/permissions-reference-eventbridge.html
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['events:PutEvents'],
            resourceArns: ['*'],
        });
    }
    static eventBusProps(defaultEventBusName, props) {
        if (props) {
            const { eventBusName, eventSourceName } = props;
            const eventBusNameRegex = /^[\/\.\-_A-Za-z0-9]{1,256}$/;
            if (eventBusName !== undefined && eventSourceName !== undefined) {
                throw new Error('\'eventBusName\' and \'eventSourceName\' cannot both be provided');
            }
            else if (eventBusName !== undefined) {
                if (eventBusName === 'default') {
                    throw new Error('\'eventBusName\' must not be \'default\'');
                }
                else if (eventBusName.indexOf('/') > -1) {
                    throw new Error('\'eventBusName\' must not contain \'/\'');
                }
                else if (!eventBusNameRegex.test(eventBusName)) {
                    throw new Error(`'eventBusName' must satisfy: ${eventBusNameRegex}`);
                }
                return { eventBusName };
            }
            else if (eventSourceName !== undefined) {
                // Ex: aws.partner/PartnerName/acct1/repo1
                const eventSourceNameRegex = /^aws\.partner(\/[\.\-_A-Za-z0-9]+){2,}$/;
                if (!eventSourceNameRegex.test(eventSourceName)) {
                    throw new Error(`'eventSourceName' must satisfy: ${eventSourceNameRegex}`);
                }
                else if (!eventBusNameRegex.test(eventSourceName)) {
                    throw new Error(`'eventSourceName' must satisfy: ${eventBusNameRegex}`);
                }
                return { eventBusName: eventSourceName, eventSourceName };
            }
        }
        return { eventBusName: defaultEventBusName };
    }
}
exports.EventBus = EventBus;
//# sourceMappingURL=data:application/json;base64,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