"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationMultipleTargetGroupsFargateService = void 0;
const aws_ecs_1 = require("../../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const application_multiple_target_groups_service_base_1 = require("../base/application-multiple-target-groups-service-base");
/**
 * A Fargate service running on an ECS cluster fronted by an application load balancer.
 */
class ApplicationMultipleTargetGroupsFargateService extends application_multiple_target_groups_service_base_1.ApplicationMultipleTargetGroupsServiceBase {
    /**
     * Constructs a new instance of the ApplicationMultipleTargetGroupsFargateService class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.assignPublicIp = props.assignPublicIp !== undefined ? props.assignPublicIp : false;
        if (props.taskDefinition && props.taskImageOptions) {
            throw new Error('You must specify only one of TaskDefinition or TaskImageOptions.');
        }
        else if (props.taskDefinition) {
            this.taskDefinition = props.taskDefinition;
        }
        else if (props.taskImageOptions) {
            const taskImageOptions = props.taskImageOptions;
            this.taskDefinition = new aws_ecs_1.FargateTaskDefinition(this, 'TaskDef', {
                memoryLimitMiB: props.memoryLimitMiB,
                cpu: props.cpu,
                executionRole: taskImageOptions.executionRole,
                taskRole: taskImageOptions.taskRole,
                family: taskImageOptions.family,
            });
            const containerName = taskImageOptions.containerName !== undefined ? taskImageOptions.containerName : 'web';
            const container = this.taskDefinition.addContainer(containerName, {
                image: taskImageOptions.image,
                logging: this.logDriver,
                environment: taskImageOptions.environment,
                secrets: taskImageOptions.secrets,
            });
            if (taskImageOptions.containerPorts) {
                for (const containerPort of taskImageOptions.containerPorts) {
                    container.addPortMappings({
                        containerPort,
                    });
                }
            }
        }
        else {
            throw new Error('You must specify one of: taskDefinition or image');
        }
        if (!this.taskDefinition.defaultContainer) {
            throw new Error('At least one essential container must be specified');
        }
        if (this.taskDefinition.defaultContainer.portMappings.length === 0) {
            this.taskDefinition.defaultContainer.addPortMappings({
                containerPort: 80,
            });
        }
        this.service = this.createFargateService(props);
        if (props.targetGroups) {
            this.addPortMappingForTargets(this.taskDefinition.defaultContainer, props.targetGroups);
            this.targetGroup = this.registerECSTargets(this.service, this.taskDefinition.defaultContainer, props.targetGroups);
        }
        else {
            this.targetGroup = this.listener.addTargets('ECS', {
                targets: [this.service],
                port: this.taskDefinition.defaultContainer.portMappings[0].containerPort,
            });
        }
    }
    createFargateService(props) {
        return new aws_ecs_1.FargateService(this, 'Service', {
            cluster: this.cluster,
            desiredCount: this.desiredCount,
            taskDefinition: this.taskDefinition,
            assignPublicIp: this.assignPublicIp,
            serviceName: props.serviceName,
            healthCheckGracePeriod: props.healthCheckGracePeriod,
            propagateTags: props.propagateTags,
            enableECSManagedTags: props.enableECSManagedTags,
            cloudMapOptions: props.cloudMapOptions,
            platformVersion: props.platformVersion,
        });
    }
}
exports.ApplicationMultipleTargetGroupsFargateService = ApplicationMultipleTargetGroupsFargateService;
//# sourceMappingURL=data:application/json;base64,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