"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationLoadBalancedServiceBase = void 0;
const aws_certificatemanager_1 = require("../../../aws-certificatemanager"); // Automatically re-written from '@aws-cdk/aws-certificatemanager'
const aws_ecs_1 = require("../../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const aws_elasticloadbalancingv2_1 = require("../../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
const aws_route53_1 = require("../../../aws-route53"); // Automatically re-written from '@aws-cdk/aws-route53'
const aws_route53_targets_1 = require("../../../aws-route53-targets"); // Automatically re-written from '@aws-cdk/aws-route53-targets'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * The base class for ApplicationLoadBalancedEc2Service and ApplicationLoadBalancedFargateService services.
 */
class ApplicationLoadBalancedServiceBase extends cdk.Construct {
    /**
     * Constructs a new instance of the ApplicationLoadBalancedServiceBase class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        if (props.desiredCount !== undefined && props.desiredCount < 1) {
            throw new Error('You must specify a desiredCount greater than 0');
        }
        this.desiredCount = props.desiredCount || 1;
        const internetFacing = props.publicLoadBalancer !== undefined ? props.publicLoadBalancer : true;
        const lbProps = {
            vpc: this.cluster.vpc,
            internetFacing,
        };
        const loadBalancer = props.loadBalancer !== undefined ? props.loadBalancer
            : new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, 'LB', lbProps);
        if (props.certificate !== undefined && props.protocol !== undefined && props.protocol !== aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            throw new Error('The HTTPS protocol must be used when a certificate is given');
        }
        const protocol = props.protocol !== undefined ? props.protocol :
            (props.certificate ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP);
        const targetProps = {
            port: 80,
        };
        this.listener = loadBalancer.addListener('PublicListener', {
            protocol,
            port: props.listenerPort,
            open: true,
        });
        this.targetGroup = this.listener.addTargets('ECS', targetProps);
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            if (typeof props.domainName === 'undefined' || typeof props.domainZone === 'undefined') {
                throw new Error('A domain name and zone is required when using the HTTPS protocol');
            }
            if (props.certificate !== undefined) {
                this.certificate = props.certificate;
            }
            else {
                this.certificate = new aws_certificatemanager_1.DnsValidatedCertificate(this, 'Certificate', {
                    domainName: props.domainName,
                    hostedZone: props.domainZone,
                });
            }
        }
        if (this.certificate !== undefined) {
            this.listener.addCertificates('Arns', [aws_elasticloadbalancingv2_1.ListenerCertificate.fromCertificateManager(this.certificate)]);
        }
        let domainName = loadBalancer.loadBalancerDnsName;
        if (typeof props.domainName !== 'undefined') {
            if (typeof props.domainZone === 'undefined') {
                throw new Error('A Route53 hosted domain zone name is required to configure the specified domain name');
            }
            const record = new aws_route53_1.ARecord(this, 'DNS', {
                zone: props.domainZone,
                recordName: props.domainName,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(loadBalancer)),
            });
            domainName = record.domainName;
        }
        if (loadBalancer instanceof aws_elasticloadbalancingv2_1.ApplicationLoadBalancer) {
            this._applicationLoadBalancer = loadBalancer;
        }
        new cdk.CfnOutput(this, 'LoadBalancerDNS', { value: loadBalancer.loadBalancerDnsName });
        new cdk.CfnOutput(this, 'ServiceURL', { value: protocol.toLowerCase() + '://' + domainName });
    }
    /**
     * The Application Load Balancer for the service.
     */
    get loadBalancer() {
        if (!this._applicationLoadBalancer) {
            throw new Error('.loadBalancer can only be accessed if the class was constructed with an owned, not imported, load balancer');
        }
        return this._applicationLoadBalancer;
    }
    /**
     * Returns the default cluster.
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = cdk.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    /**
     * Adds service as a target of the target group.
     */
    addServiceAsTarget(service) {
        this.targetGroup.addTarget(service);
    }
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
}
exports.ApplicationLoadBalancedServiceBase = ApplicationLoadBalancedServiceBase;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwbGljYXRpb24tbG9hZC1iYWxhbmNlZC1zZXJ2aWNlLWJhc2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhcHBsaWNhdGlvbi1sb2FkLWJhbGFuY2VkLXNlcnZpY2UtYmFzZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw0RUFBd0YsQ0FBQyxrRUFBa0U7QUFFM0osOENBQXlKLENBQUMsbURBQW1EO0FBQzdNLG9GQUErTCxDQUFDLHNFQUFzRTtBQUV0USxzREFBMEUsQ0FBQyx1REFBdUQ7QUFDbEksc0VBQWtFLENBQUMsK0RBQStEO0FBQ2xJLHFDQUFxQyxDQUFDLGdEQUFnRDtBQW1OdEY7O0dBRUc7QUFDSCxNQUFzQixrQ0FBbUMsU0FBUSxHQUFHLENBQUMsU0FBUztJQStCMUU7O09BRUc7SUFDSCxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLFFBQWlELEVBQUU7UUFDN0YsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixJQUFJLEtBQUssQ0FBQyxPQUFPLElBQUksS0FBSyxDQUFDLEdBQUcsRUFBRTtZQUM1QixNQUFNLElBQUksS0FBSyxDQUFDLHFGQUFxRixDQUFDLENBQUM7U0FDMUc7UUFDRCxJQUFJLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLElBQUksSUFBSSxDQUFDLGlCQUFpQixDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDeEUsSUFBSSxLQUFLLENBQUMsWUFBWSxLQUFLLFNBQVMsSUFBSSxLQUFLLENBQUMsWUFBWSxHQUFHLENBQUMsRUFBRTtZQUM1RCxNQUFNLElBQUksS0FBSyxDQUFDLGdEQUFnRCxDQUFDLENBQUM7U0FDckU7UUFDRCxJQUFJLENBQUMsWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLElBQUksQ0FBQyxDQUFDO1FBQzVDLE1BQU0sY0FBYyxHQUFHLEtBQUssQ0FBQyxrQkFBa0IsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDO1FBQ2hHLE1BQU0sT0FBTyxHQUFHO1lBQ1osR0FBRyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRztZQUNyQixjQUFjO1NBQ2pCLENBQUM7UUFDRixNQUFNLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFlBQVk7WUFDdEUsQ0FBQyxDQUFDLElBQUksb0RBQXVCLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztRQUN2RCxJQUFJLEtBQUssQ0FBQyxXQUFXLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxRQUFRLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxRQUFRLEtBQUssZ0RBQW1CLENBQUMsS0FBSyxFQUFFO1lBQ2pILE1BQU0sSUFBSSxLQUFLLENBQUMsNkRBQTZELENBQUMsQ0FBQztTQUNsRjtRQUNELE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDNUQsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxnREFBbUIsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLGdEQUFtQixDQUFDLElBQUksQ0FBQyxDQUFDO1FBQy9FLE1BQU0sV0FBVyxHQUFHO1lBQ2hCLElBQUksRUFBRSxFQUFFO1NBQ1gsQ0FBQztRQUNGLElBQUksQ0FBQyxRQUFRLEdBQUcsWUFBWSxDQUFDLFdBQVcsQ0FBQyxnQkFBZ0IsRUFBRTtZQUN2RCxRQUFRO1lBQ1IsSUFBSSxFQUFFLEtBQUssQ0FBQyxZQUFZO1lBQ3hCLElBQUksRUFBRSxJQUFJO1NBQ2IsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDaEUsSUFBSSxRQUFRLEtBQUssZ0RBQW1CLENBQUMsS0FBSyxFQUFFO1lBQ3hDLElBQUksT0FBTyxLQUFLLENBQUMsVUFBVSxLQUFLLFdBQVcsSUFBSSxPQUFPLEtBQUssQ0FBQyxVQUFVLEtBQUssV0FBVyxFQUFFO2dCQUNwRixNQUFNLElBQUksS0FBSyxDQUFDLGtFQUFrRSxDQUFDLENBQUM7YUFDdkY7WUFDRCxJQUFJLEtBQUssQ0FBQyxXQUFXLEtBQUssU0FBUyxFQUFFO2dCQUNqQyxJQUFJLENBQUMsV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7YUFDeEM7aUJBQ0k7Z0JBQ0QsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLGdEQUF1QixDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7b0JBQ2hFLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtvQkFDNUIsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVO2lCQUMvQixDQUFDLENBQUM7YUFDTjtTQUNKO1FBQ0QsSUFBSSxJQUFJLENBQUMsV0FBVyxLQUFLLFNBQVMsRUFBRTtZQUNoQyxJQUFJLENBQUMsUUFBUSxDQUFDLGVBQWUsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxnREFBbUIsQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQ3pHO1FBQ0QsSUFBSSxVQUFVLEdBQUcsWUFBWSxDQUFDLG1CQUFtQixDQUFDO1FBQ2xELElBQUksT0FBTyxLQUFLLENBQUMsVUFBVSxLQUFLLFdBQVcsRUFBRTtZQUN6QyxJQUFJLE9BQU8sS0FBSyxDQUFDLFVBQVUsS0FBSyxXQUFXLEVBQUU7Z0JBQ3pDLE1BQU0sSUFBSSxLQUFLLENBQUMsc0ZBQXNGLENBQUMsQ0FBQzthQUMzRztZQUNELE1BQU0sTUFBTSxHQUFHLElBQUkscUJBQU8sQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFO2dCQUNwQyxJQUFJLEVBQUUsS0FBSyxDQUFDLFVBQVU7Z0JBQ3RCLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtnQkFDNUIsTUFBTSxFQUFFLDBCQUFZLENBQUMsU0FBUyxDQUFDLElBQUksd0NBQWtCLENBQUMsWUFBWSxDQUFDLENBQUM7YUFDdkUsQ0FBQyxDQUFDO1lBQ0gsVUFBVSxHQUFHLE1BQU0sQ0FBQyxVQUFVLENBQUM7U0FDbEM7UUFDRCxJQUFJLFlBQVksWUFBWSxvREFBdUIsRUFBRTtZQUNqRCxJQUFJLENBQUMsd0JBQXdCLEdBQUcsWUFBWSxDQUFDO1NBQ2hEO1FBQ0QsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxpQkFBaUIsRUFBRSxFQUFFLEtBQUssRUFBRSxZQUFZLENBQUMsbUJBQW1CLEVBQUUsQ0FBQyxDQUFDO1FBQ3hGLElBQUksR0FBRyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxXQUFXLEVBQUUsR0FBRyxLQUFLLEdBQUcsVUFBVSxFQUFFLENBQUMsQ0FBQztJQUNsRyxDQUFDO0lBOUZEOztPQUVHO0lBQ0gsSUFBVyxZQUFZO1FBQ25CLElBQUksQ0FBQyxJQUFJLENBQUMsd0JBQXdCLEVBQUU7WUFDaEMsTUFBTSxJQUFJLEtBQUssQ0FBQyw0R0FBNEcsQ0FBQyxDQUFDO1NBQ2pJO1FBQ0QsT0FBTyxJQUFJLENBQUMsd0JBQXdCLENBQUM7SUFDekMsQ0FBQztJQXVGRDs7T0FFRztJQUNPLGlCQUFpQixDQUFDLEtBQW9CLEVBQUUsR0FBVTtRQUN4RCwrREFBK0Q7UUFDL0QsTUFBTSxrQkFBa0IsR0FBRyw2QkFBNkIsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUM7UUFDakYsTUFBTSxLQUFLLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDbEMsT0FBTyxLQUFLLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxrQkFBa0IsQ0FBWSxJQUFJLElBQUksaUJBQU8sQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO0lBQ3JILENBQUM7SUFDRDs7T0FFRztJQUNPLGtCQUFrQixDQUFDLE9BQW9CO1FBQzdDLElBQUksQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFDUyxrQkFBa0IsQ0FBQyxNQUFjO1FBQ3ZDLE9BQU8sSUFBSSxzQkFBWSxDQUFDLEVBQUUsWUFBWSxFQUFFLE1BQU0sRUFBRSxDQUFDLENBQUM7SUFDdEQsQ0FBQztDQUNKO0FBdEhELGdGQXNIQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IERuc1ZhbGlkYXRlZENlcnRpZmljYXRlLCBJQ2VydGlmaWNhdGUgfSBmcm9tIFwiLi4vLi4vLi4vYXdzLWNlcnRpZmljYXRlbWFuYWdlclwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWNlcnRpZmljYXRlbWFuYWdlcidcbmltcG9ydCB7IElWcGMgfSBmcm9tIFwiLi4vLi4vLi4vYXdzLWVjMlwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWVjMidcbmltcG9ydCB7IEF3c0xvZ0RyaXZlciwgQmFzZVNlcnZpY2UsIENsb3VkTWFwT3B0aW9ucywgQ2x1c3RlciwgQ29udGFpbmVySW1hZ2UsIElDbHVzdGVyLCBMb2dEcml2ZXIsIFByb3BhZ2F0ZWRUYWdTb3VyY2UsIFNlY3JldCB9IGZyb20gXCIuLi8uLi8uLi9hd3MtZWNzXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtZWNzJ1xuaW1wb3J0IHsgQXBwbGljYXRpb25MaXN0ZW5lciwgQXBwbGljYXRpb25Mb2FkQmFsYW5jZXIsIEFwcGxpY2F0aW9uUHJvdG9jb2wsIEFwcGxpY2F0aW9uVGFyZ2V0R3JvdXAsIElBcHBsaWNhdGlvbkxvYWRCYWxhbmNlciwgTGlzdGVuZXJDZXJ0aWZpY2F0ZSB9IGZyb20gXCIuLi8uLi8uLi9hd3MtZWxhc3RpY2xvYWRiYWxhbmNpbmd2MlwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWVsYXN0aWNsb2FkYmFsYW5jaW5ndjInXG5pbXBvcnQgeyBJUm9sZSB9IGZyb20gXCIuLi8uLi8uLi9hd3MtaWFtXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtaWFtJ1xuaW1wb3J0IHsgQVJlY29yZCwgSUhvc3RlZFpvbmUsIFJlY29yZFRhcmdldCB9IGZyb20gXCIuLi8uLi8uLi9hd3Mtcm91dGU1M1wiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLXJvdXRlNTMnXG5pbXBvcnQgeyBMb2FkQmFsYW5jZXJUYXJnZXQgfSBmcm9tIFwiLi4vLi4vLi4vYXdzLXJvdXRlNTMtdGFyZ2V0c1wiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLXJvdXRlNTMtdGFyZ2V0cydcbmltcG9ydCAqIGFzIGNkayBmcm9tIFwiLi4vLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbi8qKlxuICogVGhlIHByb3BlcnRpZXMgZm9yIHRoZSBiYXNlIEFwcGxpY2F0aW9uTG9hZEJhbGFuY2VkRWMyU2VydmljZSBvciBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlZEZhcmdhdGVTZXJ2aWNlIHNlcnZpY2UuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXBwbGljYXRpb25Mb2FkQmFsYW5jZWRTZXJ2aWNlQmFzZVByb3BzIHtcbiAgICAvKipcbiAgICAgKiBUaGUgbmFtZSBvZiB0aGUgY2x1c3RlciB0aGF0IGhvc3RzIHRoZSBzZXJ2aWNlLlxuICAgICAqXG4gICAgICogSWYgYSBjbHVzdGVyIGlzIHNwZWNpZmllZCwgdGhlIHZwYyBjb25zdHJ1Y3Qgc2hvdWxkIGJlIG9taXR0ZWQuIEFsdGVybmF0aXZlbHksIHlvdSBjYW4gb21pdCBib3RoIGNsdXN0ZXIgYW5kIHZwYy5cbiAgICAgKiBAZGVmYXVsdCAtIGNyZWF0ZSBhIG5ldyBjbHVzdGVyOyBpZiBib3RoIGNsdXN0ZXIgYW5kIHZwYyBhcmUgb21pdHRlZCwgYSBuZXcgVlBDIHdpbGwgYmUgY3JlYXRlZCBmb3IgeW91LlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNsdXN0ZXI/OiBJQ2x1c3RlcjtcbiAgICAvKipcbiAgICAgKiBUaGUgVlBDIHdoZXJlIHRoZSBjb250YWluZXIgaW5zdGFuY2VzIHdpbGwgYmUgbGF1bmNoZWQgb3IgdGhlIGVsYXN0aWMgbmV0d29yayBpbnRlcmZhY2VzIChFTklzKSB3aWxsIGJlIGRlcGxveWVkLlxuICAgICAqXG4gICAgICogSWYgYSB2cGMgaXMgc3BlY2lmaWVkLCB0aGUgY2x1c3RlciBjb25zdHJ1Y3Qgc2hvdWxkIGJlIG9taXR0ZWQuIEFsdGVybmF0aXZlbHksIHlvdSBjYW4gb21pdCBib3RoIHZwYyBhbmQgY2x1c3Rlci5cbiAgICAgKiBAZGVmYXVsdCAtIHVzZXMgdGhlIFZQQyBkZWZpbmVkIGluIHRoZSBjbHVzdGVyIG9yIGNyZWF0ZXMgYSBuZXcgVlBDLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHZwYz86IElWcGM7XG4gICAgLyoqXG4gICAgICogVGhlIHByb3BlcnRpZXMgcmVxdWlyZWQgdG8gY3JlYXRlIGEgbmV3IHRhc2sgZGVmaW5pdGlvbi4gVGFza0RlZmluaXRpb24gb3IgVGFza0ltYWdlT3B0aW9ucyBtdXN0IGJlIHNwZWNpZmllZCwgYnV0IG5vdCBib3RoLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgbm9uZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRhc2tJbWFnZU9wdGlvbnM/OiBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlZFRhc2tJbWFnZU9wdGlvbnM7XG4gICAgLyoqXG4gICAgICogRGV0ZXJtaW5lcyB3aGV0aGVyIHRoZSBMb2FkIEJhbGFuY2VyIHdpbGwgYmUgaW50ZXJuZXQtZmFjaW5nLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgdHJ1ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHB1YmxpY0xvYWRCYWxhbmNlcj86IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogVGhlIGRlc2lyZWQgbnVtYmVyIG9mIGluc3RhbnRpYXRpb25zIG9mIHRoZSB0YXNrIGRlZmluaXRpb24gdG8ga2VlcCBydW5uaW5nIG9uIHRoZSBzZXJ2aWNlLlxuICAgICAqIFRoZSBtaW5pbXVtIHZhbHVlIGlzIDFcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IDFcbiAgICAgKi9cbiAgICByZWFkb25seSBkZXNpcmVkQ291bnQ/OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogQ2VydGlmaWNhdGUgTWFuYWdlciBjZXJ0aWZpY2F0ZSB0byBhc3NvY2lhdGUgd2l0aCB0aGUgbG9hZCBiYWxhbmNlci5cbiAgICAgKiBTZXR0aW5nIHRoaXMgb3B0aW9uIHdpbGwgc2V0IHRoZSBsb2FkIGJhbGFuY2VyIHByb3RvY29sIHRvIEhUVFBTLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBjZXJ0aWZpY2F0ZSBhc3NvY2lhdGVkIHdpdGggdGhlIGxvYWQgYmFsYW5jZXIsIGlmIHVzaW5nXG4gICAgICogdGhlIEhUVFAgcHJvdG9jb2wuIEZvciBIVFRQUywgYSBETlMtdmFsaWRhdGVkIGNlcnRpZmljYXRlIHdpbGwgYmVcbiAgICAgKiBjcmVhdGVkIGZvciB0aGUgbG9hZCBiYWxhbmNlcidzIHNwZWNpZmllZCBkb21haW4gbmFtZS5cbiAgICAgKi9cbiAgICByZWFkb25seSBjZXJ0aWZpY2F0ZT86IElDZXJ0aWZpY2F0ZTtcbiAgICAvKipcbiAgICAgKiBUaGUgcHJvdG9jb2wgZm9yIGNvbm5lY3Rpb25zIGZyb20gY2xpZW50cyB0byB0aGUgbG9hZCBiYWxhbmNlci5cbiAgICAgKiBUaGUgbG9hZCBiYWxhbmNlciBwb3J0IGlzIGRldGVybWluZWQgZnJvbSB0aGUgcHJvdG9jb2wgKHBvcnQgODAgZm9yXG4gICAgICogSFRUUCwgcG9ydCA0NDMgZm9yIEhUVFBTKS4gIEEgZG9tYWluIG5hbWUgYW5kIHpvbmUgbXVzdCBiZSBhbHNvIGJlXG4gICAgICogc3BlY2lmaWVkIGlmIHVzaW5nIEhUVFBTLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgSFRUUC4gSWYgYSBjZXJ0aWZpY2F0ZSBpcyBzcGVjaWZpZWQsIHRoZSBwcm90b2NvbCB3aWxsIGJlXG4gICAgICogc2V0IGJ5IGRlZmF1bHQgdG8gSFRUUFMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgcHJvdG9jb2w/OiBBcHBsaWNhdGlvblByb3RvY29sO1xuICAgIC8qKlxuICAgICAqIFRoZSBkb21haW4gbmFtZSBmb3IgdGhlIHNlcnZpY2UsIGUuZy4gXCJhcGkuZXhhbXBsZS5jb20uXCJcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gZG9tYWluIG5hbWUuXG4gICAgICovXG4gICAgcmVhZG9ubHkgZG9tYWluTmFtZT86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgUm91dGU1MyBob3N0ZWQgem9uZSBmb3IgdGhlIGRvbWFpbiwgZS5nLiBcImV4YW1wbGUuY29tLlwiXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIFJvdXRlNTMgaG9zdGVkIGRvbWFpbiB6b25lLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGRvbWFpblpvbmU/OiBJSG9zdGVkWm9uZTtcbiAgICAvKipcbiAgICAgKiBUaGUgbmFtZSBvZiB0aGUgc2VydmljZS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gQ2xvdWRGb3JtYXRpb24tZ2VuZXJhdGVkIG5hbWUuXG4gICAgICovXG4gICAgcmVhZG9ubHkgc2VydmljZU5hbWU/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHBlcmlvZCBvZiB0aW1lLCBpbiBzZWNvbmRzLCB0aGF0IHRoZSBBbWF6b24gRUNTIHNlcnZpY2Ugc2NoZWR1bGVyIGlnbm9yZXMgdW5oZWFsdGh5XG4gICAgICogRWxhc3RpYyBMb2FkIEJhbGFuY2luZyB0YXJnZXQgaGVhbHRoIGNoZWNrcyBhZnRlciBhIHRhc2sgaGFzIGZpcnN0IHN0YXJ0ZWQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIGRlZmF1bHRzIHRvIDYwIHNlY29uZHMgaWYgYXQgbGVhc3Qgb25lIGxvYWQgYmFsYW5jZXIgaXMgaW4tdXNlIGFuZCBpdCBpcyBub3QgYWxyZWFkeSBzZXRcbiAgICAgKi9cbiAgICByZWFkb25seSBoZWFsdGhDaGVja0dyYWNlUGVyaW9kPzogY2RrLkR1cmF0aW9uO1xuICAgIC8qKlxuICAgICAqIFRoZSBtYXhpbXVtIG51bWJlciBvZiB0YXNrcywgc3BlY2lmaWVkIGFzIGEgcGVyY2VudGFnZSBvZiB0aGUgQW1hem9uIEVDU1xuICAgICAqIHNlcnZpY2UncyBEZXNpcmVkQ291bnQgdmFsdWUsIHRoYXQgY2FuIHJ1biBpbiBhIHNlcnZpY2UgZHVyaW5nIGFcbiAgICAgKiBkZXBsb3ltZW50LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSAxMDAgaWYgZGFlbW9uLCBvdGhlcndpc2UgMjAwXG4gICAgICovXG4gICAgcmVhZG9ubHkgbWF4SGVhbHRoeVBlcmNlbnQ/OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogVGhlIG1pbmltdW0gbnVtYmVyIG9mIHRhc2tzLCBzcGVjaWZpZWQgYXMgYSBwZXJjZW50YWdlIG9mXG4gICAgICogdGhlIEFtYXpvbiBFQ1Mgc2VydmljZSdzIERlc2lyZWRDb3VudCB2YWx1ZSwgdGhhdCBtdXN0XG4gICAgICogY29udGludWUgdG8gcnVuIGFuZCByZW1haW4gaGVhbHRoeSBkdXJpbmcgYSBkZXBsb3ltZW50LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSAwIGlmIGRhZW1vbiwgb3RoZXJ3aXNlIDUwXG4gICAgICovXG4gICAgcmVhZG9ubHkgbWluSGVhbHRoeVBlcmNlbnQ/OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogVGhlIGFwcGxpY2F0aW9uIGxvYWQgYmFsYW5jZXIgdGhhdCB3aWxsIHNlcnZlIHRyYWZmaWMgdG8gdGhlIHNlcnZpY2UuXG4gICAgICogVGhlIFZQQyBhdHRyaWJ1dGUgb2YgYSBsb2FkIGJhbGFuY2VyIG11c3QgYmUgc3BlY2lmaWVkIGZvciBpdCB0byBiZSB1c2VkXG4gICAgICogdG8gY3JlYXRlIGEgbmV3IHNlcnZpY2Ugd2l0aCB0aGlzIHBhdHRlcm4uXG4gICAgICpcbiAgICAgKiBbZGlzYWJsZS1hd3NsaW50OnJlZi12aWEtaW50ZXJmYWNlXVxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBhIG5ldyBsb2FkIGJhbGFuY2VyIHdpbGwgYmUgY3JlYXRlZC5cbiAgICAgKi9cbiAgICByZWFkb25seSBsb2FkQmFsYW5jZXI/OiBJQXBwbGljYXRpb25Mb2FkQmFsYW5jZXI7XG4gICAgLyoqXG4gICAgICogTGlzdGVuZXIgcG9ydCBvZiB0aGUgYXBwbGljYXRpb24gbG9hZCBiYWxhbmNlciB0aGF0IHdpbGwgc2VydmUgdHJhZmZpYyB0byB0aGUgc2VydmljZS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gVGhlIGRlZmF1bHQgbGlzdGVuZXIgcG9ydCBpcyBkZXRlcm1pbmVkIGZyb20gdGhlIHByb3RvY29sIChwb3J0IDgwIGZvciBIVFRQLFxuICAgICAqIHBvcnQgNDQzIGZvciBIVFRQUykuIEEgZG9tYWluIG5hbWUgYW5kIHpvbmUgbXVzdCBiZSBhbHNvIGJlIHNwZWNpZmllZCBpZiB1c2luZyBIVFRQUy5cbiAgICAgKi9cbiAgICByZWFkb25seSBsaXN0ZW5lclBvcnQ/OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogU3BlY2lmaWVzIHdoZXRoZXIgdG8gcHJvcGFnYXRlIHRoZSB0YWdzIGZyb20gdGhlIHRhc2sgZGVmaW5pdGlvbiBvciB0aGUgc2VydmljZSB0byB0aGUgdGFza3MgaW4gdGhlIHNlcnZpY2UuXG4gICAgICogVGFncyBjYW4gb25seSBiZSBwcm9wYWdhdGVkIHRvIHRoZSB0YXNrcyB3aXRoaW4gdGhlIHNlcnZpY2UgZHVyaW5nIHNlcnZpY2UgY3JlYXRpb24uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIG5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBwcm9wYWdhdGVUYWdzPzogUHJvcGFnYXRlZFRhZ1NvdXJjZTtcbiAgICAvKipcbiAgICAgKiBTcGVjaWZpZXMgd2hldGhlciB0byBlbmFibGUgQW1hem9uIEVDUyBtYW5hZ2VkIHRhZ3MgZm9yIHRoZSB0YXNrcyB3aXRoaW4gdGhlIHNlcnZpY2UuIEZvciBtb3JlIGluZm9ybWF0aW9uLCBzZWVcbiAgICAgKiBbVGFnZ2luZyBZb3VyIEFtYXpvbiBFQ1MgUmVzb3VyY2VzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9lY3MtdXNpbmctdGFncy5odG1sKVxuICAgICAqXG4gICAgICogQGRlZmF1bHQgZmFsc2VcbiAgICAgKi9cbiAgICByZWFkb25seSBlbmFibGVFQ1NNYW5hZ2VkVGFncz86IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogVGhlIG9wdGlvbnMgZm9yIGNvbmZpZ3VyaW5nIGFuIEFtYXpvbiBFQ1Mgc2VydmljZSB0byB1c2Ugc2VydmljZSBkaXNjb3ZlcnkuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIEFXUyBDbG91ZCBNYXAgc2VydmljZSBkaXNjb3ZlcnkgaXMgbm90IGVuYWJsZWQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgY2xvdWRNYXBPcHRpb25zPzogQ2xvdWRNYXBPcHRpb25zO1xufVxuZXhwb3J0IGludGVyZmFjZSBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlZFRhc2tJbWFnZU9wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIFRoZSBpbWFnZSB1c2VkIHRvIHN0YXJ0IGEgY29udGFpbmVyLiBJbWFnZSBvciB0YXNrRGVmaW5pdGlvbiBtdXN0IGJlIHNwZWNpZmllZCwgbm90IGJvdGguXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIG5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBpbWFnZTogQ29udGFpbmVySW1hZ2U7XG4gICAgLyoqXG4gICAgICogVGhlIGVudmlyb25tZW50IHZhcmlhYmxlcyB0byBwYXNzIHRvIHRoZSBjb250YWluZXIuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGVudmlyb25tZW50IHZhcmlhYmxlcy5cbiAgICAgKi9cbiAgICByZWFkb25seSBlbnZpcm9ubWVudD86IHtcbiAgICAgICAgW2tleTogc3RyaW5nXTogc3RyaW5nO1xuICAgIH07XG4gICAgLyoqXG4gICAgICogVGhlIHNlY3JldCB0byBleHBvc2UgdG8gdGhlIGNvbnRhaW5lciBhcyBhbiBlbnZpcm9ubWVudCB2YXJpYWJsZS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gc2VjcmV0IGVudmlyb25tZW50IHZhcmlhYmxlcy5cbiAgICAgKi9cbiAgICByZWFkb25seSBzZWNyZXRzPzoge1xuICAgICAgICBba2V5OiBzdHJpbmddOiBTZWNyZXQ7XG4gICAgfTtcbiAgICAvKipcbiAgICAgKiBGbGFnIHRvIGluZGljYXRlIHdoZXRoZXIgdG8gZW5hYmxlIGxvZ2dpbmcuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCB0cnVlXG4gICAgICovXG4gICAgcmVhZG9ubHkgZW5hYmxlTG9nZ2luZz86IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogVGhlIGxvZyBkcml2ZXIgdG8gdXNlLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBBd3NMb2dEcml2ZXIgaWYgZW5hYmxlTG9nZ2luZyBpcyB0cnVlXG4gICAgICovXG4gICAgcmVhZG9ubHkgbG9nRHJpdmVyPzogTG9nRHJpdmVyO1xuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIG9mIHRoZSB0YXNrIGV4ZWN1dGlvbiBJQU0gcm9sZSB0aGF0IGdyYW50cyB0aGUgQW1hem9uIEVDUyBjb250YWluZXIgYWdlbnQgcGVybWlzc2lvbiB0byBjYWxsIEFXUyBBUElzIG9uIHlvdXIgYmVoYWxmLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyB2YWx1ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGV4ZWN1dGlvblJvbGU/OiBJUm9sZTtcbiAgICAvKipcbiAgICAgKiBUaGUgbmFtZSBvZiB0aGUgdGFzayBJQU0gcm9sZSB0aGF0IGdyYW50cyBjb250YWluZXJzIGluIHRoZSB0YXNrIHBlcm1pc3Npb24gdG8gY2FsbCBBV1MgQVBJcyBvbiB5b3VyIGJlaGFsZi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gQSB0YXNrIHJvbGUgaXMgYXV0b21hdGljYWxseSBjcmVhdGVkIGZvciB5b3UuXG4gICAgICovXG4gICAgcmVhZG9ubHkgdGFza1JvbGU/OiBJUm9sZTtcbiAgICAvKipcbiAgICAgKiBUaGUgY29udGFpbmVyIG5hbWUgdmFsdWUgdG8gYmUgc3BlY2lmaWVkIGluIHRoZSB0YXNrIGRlZmluaXRpb24uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIG5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBjb250YWluZXJOYW1lPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBwb3J0IG51bWJlciBvbiB0aGUgY29udGFpbmVyIHRoYXQgaXMgYm91bmQgdG8gdGhlIHVzZXItc3BlY2lmaWVkIG9yIGF1dG9tYXRpY2FsbHkgYXNzaWduZWQgaG9zdCBwb3J0LlxuICAgICAqXG4gICAgICogSWYgeW91IGFyZSB1c2luZyBjb250YWluZXJzIGluIGEgdGFzayB3aXRoIHRoZSBhd3N2cGMgb3IgaG9zdCBuZXR3b3JrIG1vZGUsIGV4cG9zZWQgcG9ydHMgc2hvdWxkIGJlIHNwZWNpZmllZCB1c2luZyBjb250YWluZXJQb3J0LlxuICAgICAqIElmIHlvdSBhcmUgdXNpbmcgY29udGFpbmVycyBpbiBhIHRhc2sgd2l0aCB0aGUgYnJpZGdlIG5ldHdvcmsgbW9kZSBhbmQgeW91IHNwZWNpZnkgYSBjb250YWluZXIgcG9ydCBhbmQgbm90IGEgaG9zdCBwb3J0LFxuICAgICAqIHlvdXIgY29udGFpbmVyIGF1dG9tYXRpY2FsbHkgcmVjZWl2ZXMgYSBob3N0IHBvcnQgaW4gdGhlIGVwaGVtZXJhbCBwb3J0IHJhbmdlLlxuICAgICAqXG4gICAgICogUG9ydCBtYXBwaW5ncyB0aGF0IGFyZSBhdXRvbWF0aWNhbGx5IGFzc2lnbmVkIGluIHRoaXMgd2F5IGRvIG5vdCBjb3VudCB0b3dhcmQgdGhlIDEwMCByZXNlcnZlZCBwb3J0cyBsaW1pdCBvZiBhIGNvbnRhaW5lciBpbnN0YW5jZS5cbiAgICAgKlxuICAgICAqIEZvciBtb3JlIGluZm9ybWF0aW9uLCBzZWVcbiAgICAgKiBbaG9zdFBvcnRdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUG9ydE1hcHBpbmcuaHRtbCNFQ1MtVHlwZS1Qb3J0TWFwcGluZy1ob3N0UG9ydCkuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCA4MFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNvbnRhaW5lclBvcnQ/OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogVGhlIG5hbWUgb2YgYSBmYW1pbHkgdGhhdCB0aGlzIHRhc2sgZGVmaW5pdGlvbiBpcyByZWdpc3RlcmVkIHRvLiBBIGZhbWlseSBncm91cHMgbXVsdGlwbGUgdmVyc2lvbnMgb2YgYSB0YXNrIGRlZmluaXRpb24uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIEF1dG9tYXRpY2FsbHkgZ2VuZXJhdGVkIG5hbWUuXG4gICAgICovXG4gICAgcmVhZG9ubHkgZmFtaWx5Pzogc3RyaW5nO1xufVxuLyoqXG4gKiBUaGUgYmFzZSBjbGFzcyBmb3IgQXBwbGljYXRpb25Mb2FkQmFsYW5jZWRFYzJTZXJ2aWNlIGFuZCBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlZEZhcmdhdGVTZXJ2aWNlIHNlcnZpY2VzLlxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgQXBwbGljYXRpb25Mb2FkQmFsYW5jZWRTZXJ2aWNlQmFzZSBleHRlbmRzIGNkay5Db25zdHJ1Y3Qge1xuICAgIC8qKlxuICAgICAqIFRoZSBkZXNpcmVkIG51bWJlciBvZiBpbnN0YW50aWF0aW9ucyBvZiB0aGUgdGFzayBkZWZpbml0aW9uIHRvIGtlZXAgcnVubmluZyBvbiB0aGUgc2VydmljZS5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgZGVzaXJlZENvdW50OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogVGhlIEFwcGxpY2F0aW9uIExvYWQgQmFsYW5jZXIgZm9yIHRoZSBzZXJ2aWNlLlxuICAgICAqL1xuICAgIHB1YmxpYyBnZXQgbG9hZEJhbGFuY2VyKCk6IEFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyIHtcbiAgICAgICAgaWYgKCF0aGlzLl9hcHBsaWNhdGlvbkxvYWRCYWxhbmNlcikge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCcubG9hZEJhbGFuY2VyIGNhbiBvbmx5IGJlIGFjY2Vzc2VkIGlmIHRoZSBjbGFzcyB3YXMgY29uc3RydWN0ZWQgd2l0aCBhbiBvd25lZCwgbm90IGltcG9ydGVkLCBsb2FkIGJhbGFuY2VyJyk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHRoaXMuX2FwcGxpY2F0aW9uTG9hZEJhbGFuY2VyO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBUaGUgbGlzdGVuZXIgZm9yIHRoZSBzZXJ2aWNlLlxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBsaXN0ZW5lcjogQXBwbGljYXRpb25MaXN0ZW5lcjtcbiAgICAvKipcbiAgICAgKiBUaGUgdGFyZ2V0IGdyb3VwIGZvciB0aGUgc2VydmljZS5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgdGFyZ2V0R3JvdXA6IEFwcGxpY2F0aW9uVGFyZ2V0R3JvdXA7XG4gICAgLyoqXG4gICAgICogQ2VydGlmaWNhdGUgTWFuYWdlciBjZXJ0aWZpY2F0ZSB0byBhc3NvY2lhdGUgd2l0aCB0aGUgbG9hZCBiYWxhbmNlci5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgY2VydGlmaWNhdGU/OiBJQ2VydGlmaWNhdGU7XG4gICAgLyoqXG4gICAgICogVGhlIGNsdXN0ZXIgdGhhdCBob3N0cyB0aGUgc2VydmljZS5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgY2x1c3RlcjogSUNsdXN0ZXI7XG4gICAgcHJpdmF0ZSByZWFkb25seSBfYXBwbGljYXRpb25Mb2FkQmFsYW5jZXI/OiBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlcjtcbiAgICAvKipcbiAgICAgKiBDb25zdHJ1Y3RzIGEgbmV3IGluc3RhbmNlIG9mIHRoZSBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlZFNlcnZpY2VCYXNlIGNsYXNzLlxuICAgICAqL1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQXBwbGljYXRpb25Mb2FkQmFsYW5jZWRTZXJ2aWNlQmFzZVByb3BzID0ge30pIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgaWYgKHByb3BzLmNsdXN0ZXIgJiYgcHJvcHMudnBjKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1lvdSBjYW4gb25seSBzcGVjaWZ5IGVpdGhlciB2cGMgb3IgY2x1c3Rlci4gQWx0ZXJuYXRpdmVseSwgeW91IGNhbiBsZWF2ZSBib3RoIGJsYW5rJyk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5jbHVzdGVyID0gcHJvcHMuY2x1c3RlciB8fCB0aGlzLmdldERlZmF1bHRDbHVzdGVyKHRoaXMsIHByb3BzLnZwYyk7XG4gICAgICAgIGlmIChwcm9wcy5kZXNpcmVkQ291bnQgIT09IHVuZGVmaW5lZCAmJiBwcm9wcy5kZXNpcmVkQ291bnQgPCAxKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1lvdSBtdXN0IHNwZWNpZnkgYSBkZXNpcmVkQ291bnQgZ3JlYXRlciB0aGFuIDAnKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLmRlc2lyZWRDb3VudCA9IHByb3BzLmRlc2lyZWRDb3VudCB8fCAxO1xuICAgICAgICBjb25zdCBpbnRlcm5ldEZhY2luZyA9IHByb3BzLnB1YmxpY0xvYWRCYWxhbmNlciAhPT0gdW5kZWZpbmVkID8gcHJvcHMucHVibGljTG9hZEJhbGFuY2VyIDogdHJ1ZTtcbiAgICAgICAgY29uc3QgbGJQcm9wcyA9IHtcbiAgICAgICAgICAgIHZwYzogdGhpcy5jbHVzdGVyLnZwYyxcbiAgICAgICAgICAgIGludGVybmV0RmFjaW5nLFxuICAgICAgICB9O1xuICAgICAgICBjb25zdCBsb2FkQmFsYW5jZXIgPSBwcm9wcy5sb2FkQmFsYW5jZXIgIT09IHVuZGVmaW5lZCA/IHByb3BzLmxvYWRCYWxhbmNlclxuICAgICAgICAgICAgOiBuZXcgQXBwbGljYXRpb25Mb2FkQmFsYW5jZXIodGhpcywgJ0xCJywgbGJQcm9wcyk7XG4gICAgICAgIGlmIChwcm9wcy5jZXJ0aWZpY2F0ZSAhPT0gdW5kZWZpbmVkICYmIHByb3BzLnByb3RvY29sICE9PSB1bmRlZmluZWQgJiYgcHJvcHMucHJvdG9jb2wgIT09IEFwcGxpY2F0aW9uUHJvdG9jb2wuSFRUUFMpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignVGhlIEhUVFBTIHByb3RvY29sIG11c3QgYmUgdXNlZCB3aGVuIGEgY2VydGlmaWNhdGUgaXMgZ2l2ZW4nKTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBwcm90b2NvbCA9IHByb3BzLnByb3RvY29sICE9PSB1bmRlZmluZWQgPyBwcm9wcy5wcm90b2NvbCA6XG4gICAgICAgICAgICAocHJvcHMuY2VydGlmaWNhdGUgPyBBcHBsaWNhdGlvblByb3RvY29sLkhUVFBTIDogQXBwbGljYXRpb25Qcm90b2NvbC5IVFRQKTtcbiAgICAgICAgY29uc3QgdGFyZ2V0UHJvcHMgPSB7XG4gICAgICAgICAgICBwb3J0OiA4MCxcbiAgICAgICAgfTtcbiAgICAgICAgdGhpcy5saXN0ZW5lciA9IGxvYWRCYWxhbmNlci5hZGRMaXN0ZW5lcignUHVibGljTGlzdGVuZXInLCB7XG4gICAgICAgICAgICBwcm90b2NvbCxcbiAgICAgICAgICAgIHBvcnQ6IHByb3BzLmxpc3RlbmVyUG9ydCxcbiAgICAgICAgICAgIG9wZW46IHRydWUsXG4gICAgICAgIH0pO1xuICAgICAgICB0aGlzLnRhcmdldEdyb3VwID0gdGhpcy5saXN0ZW5lci5hZGRUYXJnZXRzKCdFQ1MnLCB0YXJnZXRQcm9wcyk7XG4gICAgICAgIGlmIChwcm90b2NvbCA9PT0gQXBwbGljYXRpb25Qcm90b2NvbC5IVFRQUykge1xuICAgICAgICAgICAgaWYgKHR5cGVvZiBwcm9wcy5kb21haW5OYW1lID09PSAndW5kZWZpbmVkJyB8fCB0eXBlb2YgcHJvcHMuZG9tYWluWm9uZSA9PT0gJ3VuZGVmaW5lZCcpIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0EgZG9tYWluIG5hbWUgYW5kIHpvbmUgaXMgcmVxdWlyZWQgd2hlbiB1c2luZyB0aGUgSFRUUFMgcHJvdG9jb2wnKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGlmIChwcm9wcy5jZXJ0aWZpY2F0ZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICAgICAgdGhpcy5jZXJ0aWZpY2F0ZSA9IHByb3BzLmNlcnRpZmljYXRlO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICAgICAgdGhpcy5jZXJ0aWZpY2F0ZSA9IG5ldyBEbnNWYWxpZGF0ZWRDZXJ0aWZpY2F0ZSh0aGlzLCAnQ2VydGlmaWNhdGUnLCB7XG4gICAgICAgICAgICAgICAgICAgIGRvbWFpbk5hbWU6IHByb3BzLmRvbWFpbk5hbWUsXG4gICAgICAgICAgICAgICAgICAgIGhvc3RlZFpvbmU6IHByb3BzLmRvbWFpblpvbmUsXG4gICAgICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHRoaXMuY2VydGlmaWNhdGUgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgdGhpcy5saXN0ZW5lci5hZGRDZXJ0aWZpY2F0ZXMoJ0FybnMnLCBbTGlzdGVuZXJDZXJ0aWZpY2F0ZS5mcm9tQ2VydGlmaWNhdGVNYW5hZ2VyKHRoaXMuY2VydGlmaWNhdGUpXSk7XG4gICAgICAgIH1cbiAgICAgICAgbGV0IGRvbWFpbk5hbWUgPSBsb2FkQmFsYW5jZXIubG9hZEJhbGFuY2VyRG5zTmFtZTtcbiAgICAgICAgaWYgKHR5cGVvZiBwcm9wcy5kb21haW5OYW1lICE9PSAndW5kZWZpbmVkJykge1xuICAgICAgICAgICAgaWYgKHR5cGVvZiBwcm9wcy5kb21haW5ab25lID09PSAndW5kZWZpbmVkJykge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQSBSb3V0ZTUzIGhvc3RlZCBkb21haW4gem9uZSBuYW1lIGlzIHJlcXVpcmVkIHRvIGNvbmZpZ3VyZSB0aGUgc3BlY2lmaWVkIGRvbWFpbiBuYW1lJyk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBjb25zdCByZWNvcmQgPSBuZXcgQVJlY29yZCh0aGlzLCAnRE5TJywge1xuICAgICAgICAgICAgICAgIHpvbmU6IHByb3BzLmRvbWFpblpvbmUsXG4gICAgICAgICAgICAgICAgcmVjb3JkTmFtZTogcHJvcHMuZG9tYWluTmFtZSxcbiAgICAgICAgICAgICAgICB0YXJnZXQ6IFJlY29yZFRhcmdldC5mcm9tQWxpYXMobmV3IExvYWRCYWxhbmNlclRhcmdldChsb2FkQmFsYW5jZXIpKSxcbiAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgZG9tYWluTmFtZSA9IHJlY29yZC5kb21haW5OYW1lO1xuICAgICAgICB9XG4gICAgICAgIGlmIChsb2FkQmFsYW5jZXIgaW5zdGFuY2VvZiBBcHBsaWNhdGlvbkxvYWRCYWxhbmNlcikge1xuICAgICAgICAgICAgdGhpcy5fYXBwbGljYXRpb25Mb2FkQmFsYW5jZXIgPSBsb2FkQmFsYW5jZXI7XG4gICAgICAgIH1cbiAgICAgICAgbmV3IGNkay5DZm5PdXRwdXQodGhpcywgJ0xvYWRCYWxhbmNlckROUycsIHsgdmFsdWU6IGxvYWRCYWxhbmNlci5sb2FkQmFsYW5jZXJEbnNOYW1lIH0pO1xuICAgICAgICBuZXcgY2RrLkNmbk91dHB1dCh0aGlzLCAnU2VydmljZVVSTCcsIHsgdmFsdWU6IHByb3RvY29sLnRvTG93ZXJDYXNlKCkgKyAnOi8vJyArIGRvbWFpbk5hbWUgfSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJldHVybnMgdGhlIGRlZmF1bHQgY2x1c3Rlci5cbiAgICAgKi9cbiAgICBwcm90ZWN0ZWQgZ2V0RGVmYXVsdENsdXN0ZXIoc2NvcGU6IGNkay5Db25zdHJ1Y3QsIHZwYz86IElWcGMpOiBDbHVzdGVyIHtcbiAgICAgICAgLy8gbWFnaWMgc3RyaW5nIHRvIGF2b2lkIGNvbGxpc2lvbiB3aXRoIHVzZXItZGVmaW5lZCBjb25zdHJ1Y3RzXG4gICAgICAgIGNvbnN0IERFRkFVTFRfQ0xVU1RFUl9JRCA9IGBFY3NEZWZhdWx0Q2x1c3Rlck1uTDNtTk5ZTiR7dnBjID8gdnBjLm5vZGUuaWQgOiAnJ31gO1xuICAgICAgICBjb25zdCBzdGFjayA9IGNkay5TdGFjay5vZihzY29wZSk7XG4gICAgICAgIHJldHVybiBzdGFjay5ub2RlLnRyeUZpbmRDaGlsZChERUZBVUxUX0NMVVNURVJfSUQpIGFzIENsdXN0ZXIgfHwgbmV3IENsdXN0ZXIoc3RhY2ssIERFRkFVTFRfQ0xVU1RFUl9JRCwgeyB2cGMgfSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZHMgc2VydmljZSBhcyBhIHRhcmdldCBvZiB0aGUgdGFyZ2V0IGdyb3VwLlxuICAgICAqL1xuICAgIHByb3RlY3RlZCBhZGRTZXJ2aWNlQXNUYXJnZXQoc2VydmljZTogQmFzZVNlcnZpY2UpIHtcbiAgICAgICAgdGhpcy50YXJnZXRHcm91cC5hZGRUYXJnZXQoc2VydmljZSk7XG4gICAgfVxuICAgIHByb3RlY3RlZCBjcmVhdGVBV1NMb2dEcml2ZXIocHJlZml4OiBzdHJpbmcpOiBBd3NMb2dEcml2ZXIge1xuICAgICAgICByZXR1cm4gbmV3IEF3c0xvZ0RyaXZlcih7IHN0cmVhbVByZWZpeDogcHJlZml4IH0pO1xuICAgIH1cbn1cbiJdfQ==