"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupVault = exports.BackupVaultEvents = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const backup_generated_1 = require("./backup.generated");
/**
 * Backup vault events
 */
var BackupVaultEvents;
(function (BackupVaultEvents) {
    /** BACKUP_JOB_STARTED */
    BackupVaultEvents["BACKUP_JOB_STARTED"] = "BACKUP_JOB_STARTED";
    /** BACKUP_JOB_COMPLETED */
    BackupVaultEvents["BACKUP_JOB_COMPLETED"] = "BACKUP_JOB_COMPLETED";
    /** BACKUP_JOB_SUCCESSFUL */
    BackupVaultEvents["BACKUP_JOB_SUCCESSFUL"] = "BACKUP_JOB_SUCCESSFUL";
    /** BACKUP_JOB_FAILED */
    BackupVaultEvents["BACKUP_JOB_FAILED"] = "BACKUP_JOB_FAILED";
    /** BACKUP_JOB_EXPIRED */
    BackupVaultEvents["BACKUP_JOB_EXPIRED"] = "BACKUP_JOB_EXPIRED";
    /** RESTORE_JOB_STARTED */
    BackupVaultEvents["RESTORE_JOB_STARTED"] = "RESTORE_JOB_STARTED";
    /** RESTORE_JOB_COMPLETED */
    BackupVaultEvents["RESTORE_JOB_COMPLETED"] = "RESTORE_JOB_COMPLETED";
    /** RESTORE_JOB_SUCCESSFUL */
    BackupVaultEvents["RESTORE_JOB_SUCCESSFUL"] = "RESTORE_JOB_SUCCESSFUL";
    /** RESTORE_JOB_FAILED */
    BackupVaultEvents["RESTORE_JOB_FAILED"] = "RESTORE_JOB_FAILED";
    /** COPY_JOB_STARTED */
    BackupVaultEvents["COPY_JOB_STARTED"] = "COPY_JOB_STARTED";
    /** COPY_JOB_SUCCESSFUL */
    BackupVaultEvents["COPY_JOB_SUCCESSFUL"] = "COPY_JOB_SUCCESSFUL";
    /** COPY_JOB_FAILED */
    BackupVaultEvents["COPY_JOB_FAILED"] = "COPY_JOB_FAILED";
    /** RECOVERY_POINT_MODIFIED */
    BackupVaultEvents["RECOVERY_POINT_MODIFIED"] = "RECOVERY_POINT_MODIFIED";
    /** BACKUP_PLAN_CREATED */
    BackupVaultEvents["BACKUP_PLAN_CREATED"] = "BACKUP_PLAN_CREATED";
    /** BACKUP_PLAN_MODIFIED */
    BackupVaultEvents["BACKUP_PLAN_MODIFIED"] = "BACKUP_PLAN_MODIFIED";
})(BackupVaultEvents = exports.BackupVaultEvents || (exports.BackupVaultEvents = {}));
/**
 * A backup vault
 */
class BackupVault extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        if (props.backupVaultName && !/^[a-zA-Z0-9\-_\.]{1,50}$/.test(props.backupVaultName)) {
            throw new Error('Expected vault name to match pattern `^[a-zA-Z0-9\-\_\.]{1,50}$`');
        }
        let notifications;
        if (props.notificationTopic) {
            notifications = {
                backupVaultEvents: props.notificationEvents || Object.values(BackupVaultEvents),
                snsTopicArn: props.notificationTopic.topicArn,
            };
            props.notificationTopic.grantPublish(new iam.ServicePrincipal('backup.amazonaws.com'));
        }
        const vault = new backup_generated_1.CfnBackupVault(this, 'Resource', {
            backupVaultName: props.backupVaultName || this.uniqueVaultName(),
            accessPolicy: props.accessPolicy && props.accessPolicy.toJSON(),
            encryptionKeyArn: props.encryptionKey && props.encryptionKey.keyArn,
            notifications,
        });
        vault.applyRemovalPolicy(props.removalPolicy);
        this.backupVaultName = vault.attrBackupVaultName;
        this.backupVaultArn = vault.attrBackupVaultArn;
    }
    /**
     * Import an existing backup vault
     */
    static fromBackupVaultName(scope, id, backupVaultName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.backupVaultName = backupVaultName;
            }
        }
        return new Import(scope, id);
    }
    uniqueVaultName() {
        // Max length of 50 chars, get the last 50 chars
        const id = `${this.node.uniqueId}${core_1.Aws.STACK_NAME}`;
        return id.substring(Math.max(id.length - 50, 0), id.length);
    }
}
exports.BackupVault = BackupVault;
//# sourceMappingURL=data:application/json;base64,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