import * as cloudwatch from "../../aws-cloudwatch";
import * as cdk from "../../core";
import { IAutoScalingGroup } from './auto-scaling-group';
/**
 * Base interface for target tracking props
 *
 * Contains the attributes that are common to target tracking policies,
 * except the ones relating to the metric and to the scalable target.
 *
 * This interface is reused by more specific target tracking props objects.
 */
export interface BaseTargetTrackingProps {
    /**
     * Indicates whether scale in by the target tracking policy is disabled.
     *
     * If the value is true, scale in is disabled and the target tracking policy
     * won't remove capacity from the autoscaling group. Otherwise, scale in is
     * enabled and the target tracking policy can remove capacity from the
     * group.
     *
     * @default false
     */
    readonly disableScaleIn?: boolean;
    /**
     * Period after a scaling completes before another scaling activity can start.
     *
     * @default - The default cooldown configured on the AutoScalingGroup.
     */
    readonly cooldown?: cdk.Duration;
    /**
     * Estimated time until a newly launched instance can send metrics to CloudWatch.
     *
     * @default - Same as the cooldown.
     */
    readonly estimatedInstanceWarmup?: cdk.Duration;
}
/**
 * Properties for a Target Tracking policy that include the metric but exclude the target
 */
export interface BasicTargetTrackingScalingPolicyProps extends BaseTargetTrackingProps {
    /**
     * The target value for the metric.
     */
    readonly targetValue: number;
    /**
     * A predefined metric for application autoscaling
     *
     * The metric must track utilization. Scaling out will happen if the metric is higher than
     * the target value, scaling in will happen in the metric is lower than the target value.
     *
     * Exactly one of customMetric or predefinedMetric must be specified.
     *
     * @default - No predefined metric.
     */
    readonly predefinedMetric?: PredefinedMetric;
    /**
     * A custom metric for application autoscaling
     *
     * The metric must track utilization. Scaling out will happen if the metric is higher than
     * the target value, scaling in will happen in the metric is lower than the target value.
     *
     * Exactly one of customMetric or predefinedMetric must be specified.
     *
     * @default - No custom metric.
     */
    readonly customMetric?: cloudwatch.IMetric;
    /**
     * The resource label associated with the predefined metric
     *
     * Should be supplied if the predefined metric is ALBRequestCountPerTarget, and the
     * format should be:
     *
     * app/<load-balancer-name>/<load-balancer-id>/targetgroup/<target-group-name>/<target-group-id>
     *
     * @default - No resource label.
     */
    readonly resourceLabel?: string;
}
/**
 * Properties for a concrete TargetTrackingPolicy
 *
 * Adds the scalingTarget.
 */
export interface TargetTrackingScalingPolicyProps extends BasicTargetTrackingScalingPolicyProps {
    readonly autoScalingGroup: IAutoScalingGroup;
}
export declare class TargetTrackingScalingPolicy extends cdk.Construct {
    /**
     * ARN of the scaling policy
     */
    readonly scalingPolicyArn: string;
    /**
     * The resource object
     */
    private resource;
    constructor(scope: cdk.Construct, id: string, props: TargetTrackingScalingPolicyProps);
}
/**
 * One of the predefined autoscaling metrics
 */
export declare enum PredefinedMetric {
    /**
     * Average CPU utilization of the Auto Scaling group
     */
    ASG_AVERAGE_CPU_UTILIZATION = "ASGAverageCPUUtilization",
    /**
     * Average number of bytes received on all network interfaces by the Auto Scaling group
     */
    ASG_AVERAGE_NETWORK_IN = "ASGAverageNetworkIn",
    /**
     * Average number of bytes sent out on all network interfaces by the Auto Scaling group
     */
    ASG_AVERAGE_NETWORK_OUT = "ASGAverageNetworkOut",
    /**
     * Number of requests completed per target in an Application Load Balancer target group
     *
     * Specify the ALB to look at in the `resourceLabel` field.
     */
    ALB_REQUEST_COUNT_PER_TARGET = "ALBRequestCountPerTarget"
}
