import { Construct, Duration, IResource, Resource } from "../../../core";
import { IHttpRouteIntegration } from './integration';
import { BatchHttpRouteOptions, HttpMethod, HttpRoute } from './route';
import { HttpStage, HttpStageOptions } from './stage';
/**
 * Represents an HTTP API
 */
export interface IHttpApi extends IResource {
    /**
     * The identifier of this API Gateway HTTP API.
     * @attribute
     */
    readonly httpApiId: string;
}
/**
 * Properties to initialize an instance of `HttpApi`.
 */
export interface HttpApiProps {
    /**
     * Name for the HTTP API resoruce
     * @default - id of the HttpApi construct.
     */
    readonly apiName?: string;
    /**
     * An integration that will be configured on the catch-all route ($default).
     * @default - none
     */
    readonly defaultIntegration?: IHttpRouteIntegration;
    /**
     * Whether a default stage and deployment should be automatically created.
     * @default true
     */
    readonly createDefaultStage?: boolean;
    /**
     * Specifies a CORS configuration for an API.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-cors.html
     * @default - CORS disabled.
     */
    readonly corsPreflight?: CorsPreflightOptions;
}
/**
 * Options for the CORS Configuration
 */
export interface CorsPreflightOptions {
    /**
     * Specifies whether credentials are included in the CORS request.
     * @default false
     */
    readonly allowCredentials?: boolean;
    /**
     * Represents a collection of allowed headers.
     * @default - No Headers are allowed.
     */
    readonly allowHeaders?: string[];
    /**
     * Represents a collection of allowed HTTP methods.
     * @default - No Methods are allowed.
     */
    readonly allowMethods?: HttpMethod[];
    /**
     * Represents a collection of allowed origins.
     * @default - No Origins are allowed.
     */
    readonly allowOrigins?: string[];
    /**
     * Represents a collection of exposed headers.
     * @default - No Expose Headers are allowed.
     */
    readonly exposeHeaders?: string[];
    /**
     * The duration that the browser should cache preflight request results.
     * @default Duration.seconds(0)
     */
    readonly maxAge?: Duration;
}
/**
 * Options for the Route with Integration resoruce
 */
export interface AddRoutesOptions extends BatchHttpRouteOptions {
    /**
     * The path at which all of these routes are configured.
     */
    readonly path: string;
    /**
     * The HTTP methods to be configured
     * @default HttpMethod.ANY
     */
    readonly methods?: HttpMethod[];
}
/**
 * Create a new API Gateway HTTP API endpoint.
 * @resource AWS::ApiGatewayV2::Api
 */
export declare class HttpApi extends Resource implements IHttpApi {
    /**
     * Import an existing HTTP API into this CDK app.
     */
    static fromApiId(scope: Construct, id: string, httpApiId: string): IHttpApi;
    readonly httpApiId: string;
    private readonly defaultStage;
    constructor(scope: Construct, id: string, props?: HttpApiProps);
    /**
     * Get the URL to the default stage of this API.
     * Returns `undefined` if `createDefaultStage` is unset.
     */
    get url(): string | undefined;
    /**
     * Add a new stage.
     */
    addStage(id: string, options: HttpStageOptions): HttpStage;
    /**
     * Add multiple routes that uses the same configuration. The routes all go to the same path, but for different
     * methods.
     */
    addRoutes(options: AddRoutesOptions): HttpRoute[];
}
