"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deployment = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const crypto = require("crypto");
const apigateway_generated_1 = require("./apigateway.generated");
const restapi_1 = require("./restapi");
/**
 * A Deployment of a REST API.
 *
 * An immutable representation of a RestApi resource that can be called by users
 * using Stages. A deployment must be associated with a Stage for it to be
 * callable over the Internet.
 *
 * Normally, you don't need to define deployments manually. The RestApi
 * construct manages a Deployment resource that represents the latest model. It
 * can be accessed through `restApi.latestDeployment` (unless `deploy: false` is
 * set when defining the `RestApi`).
 *
 * If you manually define this resource, you will need to know that since
 * deployments are immutable, as long as the resource's logical ID doesn't
 * change, the deployment will represent the snapshot in time in which the
 * resource was created. This means that if you modify the RestApi model (i.e.
 * add methods or resources), these changes will not be reflected unless a new
 * deployment resource is created.
 *
 * To achieve this behavior, the method `addToLogicalId(data)` can be used to
 * augment the logical ID generated for the deployment resource such that it
 * will include arbitrary data. This is done automatically for the
 * `restApi.latestDeployment` deployment.
 *
 * Furthermore, since a deployment does not reference any of the REST API
 * resources and methods, CloudFormation will likely provision it before these
 * resources are created, which means that it will represent a "half-baked"
 * model. Use the `node.addDependency(dep)` method to circumvent that. This is done
 * automatically for the `restApi.latestDeployment` deployment.
 */
class Deployment extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.resource = new LatestDeploymentResource(this, 'Resource', {
            description: props.description,
            restApi: props.api,
        });
        if (props.retainDeployments) {
            this.resource.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        }
        this.api = props.api;
        this.deploymentId = core_1.Lazy.stringValue({ produce: () => this.resource.ref });
    }
    /**
     * Adds a component to the hash that determines this Deployment resource's
     * logical ID.
     *
     * This should be called by constructs of the API Gateway model that want to
     * invalidate the deployment when their settings change. The component will
     * be resolve()ed during synthesis so tokens are welcome.
     */
    addToLogicalId(data) {
        this.resource.addToLogicalId(data);
    }
    /**
     * Hook into synthesis before it occurs and make any final adjustments.
     */
    prepare() {
        if (this.api instanceof restapi_1.RestApi) {
            // Ignore IRestApi that are imported
            /*
             * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html
             * Quoting from CloudFormation's docs - "If you create an AWS::ApiGateway::RestApi resource and its methods (using AWS::ApiGateway::Method) in
             * the same template as your deployment, the deployment must depend on the RestApi's methods. To create a dependency, add a DependsOn attribute
             * to the deployment. If you don't, AWS CloudFormation creates the deployment right after it creates the RestApi resource that doesn't contain
             * any methods, and AWS CloudFormation encounters the following error: The REST API doesn't contain any methods."
             */
            /*
             * Adding a dependency between LatestDeployment and Method construct, using ConstructNode.addDependencies(), creates additional dependencies
             * between AWS::ApiGateway::Deployment and the AWS::Lambda::Permission nodes (children under Method), causing cyclic dependency errors. Hence,
             * falling back to declaring dependencies between the underlying CfnResources.
             */
            this.api.methods.map(m => m.node.defaultChild).forEach(m => this.resource.addDependsOn(m));
        }
    }
}
exports.Deployment = Deployment;
class LatestDeploymentResource extends apigateway_generated_1.CfnDeployment {
    constructor(scope, id, props) {
        super(scope, id, {
            description: props.description,
            restApiId: props.restApi.restApiId,
        });
        this.hashComponents = new Array();
        this.api = props.restApi;
        this.originalLogicalId = core_1.Stack.of(this).getLogicalId(this);
    }
    /**
     * Allows adding arbitrary data to the hashed logical ID of this deployment.
     * This can be used to couple the deployment to the API Gateway model.
     */
    addToLogicalId(data) {
        // if the construct is locked, it means we are already synthesizing and then
        // we can't modify the hash because we might have already calculated it.
        if (this.node.locked) {
            throw new Error('Cannot modify the logical ID when the construct is locked');
        }
        this.hashComponents.push(data);
    }
    /**
     * Hooks into synthesis to calculate a logical ID that hashes all the components
     * add via `addToLogicalId`.
     */
    prepare() {
        if (this.api instanceof restapi_1.RestApi || this.api instanceof restapi_1.SpecRestApi) { // Ignore IRestApi that are imported
            // Add CfnRestApi to the logical id so a new deployment is triggered when any of its properties change.
            const cfnRestApiCF = this.api.node.defaultChild._toCloudFormation();
            this.addToLogicalId(core_1.Stack.of(this).resolve(cfnRestApiCF));
        }
        const stack = core_1.Stack.of(this);
        // if hash components were added to the deployment, we use them to calculate
        // a logical ID for the deployment resource.
        if (this.hashComponents.length > 0) {
            const md5 = crypto.createHash('md5');
            this.hashComponents.map(x => stack.resolve(x)).forEach(c => md5.update(JSON.stringify(c)));
            this.overrideLogicalId(this.originalLogicalId + md5.digest('hex'));
        }
        super.prepare();
    }
}
//# sourceMappingURL=data:application/json;base64,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