"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * InvalidCidrRangeError is thrown when attempting to perform operations on a CIDR
 * range that is either not valid, or outside of the VPC size limits.
 */
class InvalidCidrRangeError extends Error {
    constructor(cidr) {
        super(cidr + ' is not a valid VPC CIDR range (must be between /16 and /28)');
        // The following line is required for type checking of custom errors, and must be called right after super()
        // https://stackoverflow.com/questions/31626231/custom-error-class-in-typescript
        Object.setPrototypeOf(this, InvalidCidrRangeError.prototype);
    }
}
exports.InvalidCidrRangeError = InvalidCidrRangeError;
/**
 * NetworkUtils contains helpers to work with network constructs (subnets/ranges)
 */
class NetworkUtils {
    /**
     * Validates an IPv4 string
     *
     * returns true of the string contains 4 numbers between 0-255 delimited by
     * a `.` character
     */
    static validIp(ipAddress) {
        const octets = ipAddress.split('.');
        if (octets.length !== 4) {
            return false;
        }
        return octets.map((octet) => parseInt(octet, 10)).
            filter((octet) => octet >= 0 && octet <= 255).length === 4;
    }
    /**
     * Converts a string IPv4 to a number
     *
     * takes an IP Address (e.g. 174.66.173.168) and converts to a number
     * (e.g 2923605416); currently only supports IPv4
     *
     * Uses the formula:
     * (first octet * 256³) + (second octet * 256²) + (third octet * 256) +
     * (fourth octet)
     *
     * @param  {string} the IP address (e.g. 174.66.173.168)
     * @returns {number} the integer value of the IP address (e.g 2923605416)
     */
    static ipToNum(ipAddress) {
        if (!this.validIp(ipAddress)) {
            throw new Error(`${ipAddress} is not valid`);
        }
        return ipAddress
            .split('.')
            .reduce((p, c, i) => p + parseInt(c, 10) * 256 ** (3 - i), 0);
    }
    /**
     * Takes number and converts it to IPv4 address string
     *
     * Takes a number (e.g 2923605416) and converts it to an IPv4 address string
     * currently only supports IPv4
     *
     * @param  {number} the integer value of the IP address (e.g 2923605416)
     * @returns {string} the IPv4 address (e.g. 174.66.173.168)
     */
    static numToIp(ipNum) {
        // this all because bitwise math is signed
        let remaining = ipNum;
        const address = new Array();
        for (let i = 0; i < 4; i++) {
            if (remaining !== 0) {
                address.push(Math.floor(remaining / 256 ** (3 - i)));
                remaining = remaining % 256 ** (3 - i);
            }
            else {
                address.push(0);
            }
        }
        const ipAddress = address.join('.');
        if (!this.validIp(ipAddress)) {
            throw new Error(`${ipAddress} is not a valid IP Address`);
        }
        return ipAddress;
    }
}
exports.NetworkUtils = NetworkUtils;
/**
 * Creates a network based on a CIDR Block to build contained subnets
 */
class NetworkBuilder {
    /**
     * Create a network using the provided CIDR block
     *
     * No subnets are allocated in the constructor, the maxIpConsumed is set one
     * less than the first IP in the network
     *
     */
    constructor(cidr) {
        /**
         * The list of CIDR blocks for subnets within this network
         */
        this.subnetCidrs = [];
        this.networkCidr = new CidrBlock(cidr);
        this.subnetCidrs = [];
        this.nextAvailableIp = this.networkCidr.minAddress();
    }
    /**
     * Add a subnet to the network and update the maxIpConsumed
     */
    addSubnet(mask) {
        return this.addSubnets(mask, 1)[0];
    }
    /**
     * Add {count} number of subnets to the network and update the maxIpConsumed
     */
    addSubnets(mask, count = 1) {
        if (mask < 16 || mask > 28) {
            throw new InvalidCidrRangeError(`x.x.x.x/${mask}`);
        }
        const maxIp = this.nextAvailableIp + (CidrBlock.calculateNetsize(mask) * count);
        if (this.networkCidr.maxAddress() < maxIp - 1) {
            throw new Error(`${count} of /${mask} exceeds remaining space of ${this.networkCidr.cidr}`);
        }
        const subnets = [];
        for (let i = 0; i < count; i++) {
            const subnet = new CidrBlock(this.nextAvailableIp, mask);
            this.nextAvailableIp = subnet.nextBlock().minAddress();
            this.subnetCidrs.push(subnet);
            subnets.push(subnet);
        }
        return subnets.map((subnet) => (subnet.cidr));
    }
    /**
     * return the CIDR notation strings for all subnets in the network
     */
    get cidrStrings() {
        return this.subnetCidrs.map((subnet) => (subnet.cidr));
    }
    /**
     * Calculates the largest subnet to create of the given count from the
     * remaining IP space
     */
    maskForRemainingSubnets(subnetCount) {
        const remaining = this.networkCidr.maxAddress() - this.nextAvailableIp + 1;
        const ipsPerSubnet = Math.floor(remaining / subnetCount);
        return 32 - Math.floor(Math.log2(ipsPerSubnet));
    }
}
exports.NetworkBuilder = NetworkBuilder;
/**
 * A block of IP address space with a given bit prefix
 */
class CidrBlock {
    constructor(ipAddressOrCidr, mask) {
        if (typeof ipAddressOrCidr === 'string') {
            this.mask = parseInt(ipAddressOrCidr.split('/')[1], 10);
            this.networkAddress = NetworkUtils.ipToNum(ipAddressOrCidr.split('/')[0]) +
                CidrBlock.calculateNetsize(this.mask) - 1;
        }
        else {
            if (typeof mask === 'number') {
                this.mask = mask;
            }
            else {
                // this should be impossible
                this.mask = 16;
            }
            this.networkAddress = ipAddressOrCidr + CidrBlock.calculateNetsize(this.mask) - 1;
            this.networkSize = 2 ** (32 - this.mask);
        }
        this.networkSize = 2 ** (32 - this.mask);
        this.cidr = `${this.minIp()}/${this.mask}`;
    }
    /**
     * Calculates the netmask for a given CIDR mask
     *
     * For example:
     * CidrBlock.calculateNetmask(24) returns '255.255.255.0'
     */
    static calculateNetmask(mask) {
        return NetworkUtils.numToIp(2 ** 32 - 2 ** (32 - mask));
    }
    /**
     * Calculates the number IP addresses in a CIDR Mask
     *
     * For example:
     * CidrBlock.calculateNetsize(24) returns 256
     */
    static calculateNetsize(mask) {
        return 2 ** (32 - mask);
    }
    /*
     * The maximum IP in the CIDR Blcok e.g. '10.0.8.255'
     */
    maxIp() {
        // min + (2^(32-mask)) - 1 [zero needs to count]
        return NetworkUtils.numToIp(this.maxAddress());
    }
    /*
     * The minimum IP in the CIDR Blcok e.g. '10.0.0.0'
     */
    minIp() {
        return NetworkUtils.numToIp(this.minAddress());
    }
    /*
     * Returns the number representation for the minimum IPv4 address
     */
    minAddress() {
        const div = this.networkAddress % this.networkSize;
        return this.networkAddress - div;
    }
    /*
     * Returns the number representation for the maximum IPv4 address
     */
    maxAddress() {
        // min + (2^(32-mask)) - 1 [zero needs to count]
        return this.minAddress() + this.networkSize - 1;
    }
    /*
     * Returns the next CIDR Block of the same mask size
     */
    nextBlock() {
        return new CidrBlock(this.maxAddress() + 1, this.mask);
    }
    /*
     * Returns true if this CidrBlock fully contains the provided CidrBlock
     */
    containsCidr(other) {
        return (this.maxAddress() >= other.maxAddress()) &&
            (this.minAddress() <= other.minAddress());
    }
}
exports.CidrBlock = CidrBlock;
//# sourceMappingURL=data:application/json;base64,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