"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Severity = exports.ImageRepository = void 0;
const path = require("path");
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const aws_ecr_1 = require("@aws-cdk/aws-ecr");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_iam_1 = require("@aws-cdk/aws-iam");
class ImageRepository extends aws_ecr_1.Repository {
    constructor(scope, id, props = {}) {
        const { forceDelete = false, ...rest } = props;
        super(scope, id, {
            removalPolicy: forceDelete ? core_1.RemovalPolicy.DESTROY : undefined,
            ...rest,
        });
        if (forceDelete) {
            new custom_resources_1.AwsCustomResource(this, 'ForceImageRepositoryDeletion', {
                resourceType: 'Custom::ECRForceImageRepositoryDeletion',
                onDelete: {
                    service: 'ECR',
                    action: 'deleteRepository',
                    parameters: {
                        repositoryName: this.repositoryName,
                        force: true,
                    },
                    physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.repositoryArn),
                },
                policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.repositoryArn],
                }),
            });
        }
    }
    onFinding(id, options) {
        const { alarmTopic, severity, imageTags } = options;
        const rule = this.onImageScanCompleted(id, { imageTags });
        const severityFilter = new aws_lambda_1.Function(this, 'SevierityFilter', {
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'lambdas', 'severity-filter')),
            handler: 'index.handler',
        });
        severityFilter.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['sns:Publish'],
            resources: [alarmTopic.topicArn],
        }));
        rule.addTarget(new aws_events_targets_1.LambdaFunction(severityFilter, {
            event: aws_events_1.RuleTargetInput.fromObject({
                account: aws_events_1.EventField.account,
                region: aws_events_1.EventField.region,
                time: aws_events_1.EventField.time,
                repositoryName: aws_events_1.EventField.fromPath('$.detail.repository-name'),
                imageDigest: aws_events_1.EventField.fromPath('$.detail.image-digest'),
                imageTags: aws_events_1.EventField.fromPath('$.detail.image-tags'),
                findingSeveriyCounts: aws_events_1.EventField.fromPath('$.detail.finding-severity-counts'),
                severity: severity,
                alarmTopicArn: alarmTopic.topicArn,
            }),
        }));
        return rule;
    }
}
exports.ImageRepository = ImageRepository;
var Severity;
(function (Severity) {
    Severity["CRITICAL"] = "CRITICAL";
    Severity["HIGH"] = "HIGH";
    Severity["MEDIUM"] = "MEDIUM";
    Severity["LOW"] = "LOW";
    Severity["INFORMATIONAL"] = "INFORMATIONAL";
    Severity["UNDEFINED"] = "UNDEFINED";
})(Severity = exports.Severity || (exports.Severity = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW1hZ2UtcmVwb3NpdG9yeS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9pbWFnZS1yZXBvc2l0b3J5LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLDZCQUE2QjtBQUM3Qix3Q0FBeUQ7QUFDekQsZ0VBSW1DO0FBQ25DLDhDQUErRDtBQUMvRCxvREFBd0U7QUFDeEUsb0VBQTZEO0FBQzdELG9EQUE4RDtBQUU5RCw4Q0FBMkQ7QUF5QjNELE1BQWEsZUFBZ0IsU0FBUSxvQkFBVTtJQUM3QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQThCLEVBQUU7UUFDeEUsTUFBTSxFQUFFLFdBQVcsR0FBRyxLQUFLLEVBQUUsR0FBRyxJQUFJLEVBQUUsR0FBRyxLQUFLLENBQUM7UUFFL0MsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixhQUFhLEVBQUUsV0FBVyxDQUFDLENBQUMsQ0FBQyxvQkFBYSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsU0FBUztZQUM5RCxHQUFHLElBQUk7U0FDUixDQUFDLENBQUM7UUFFSCxJQUFJLFdBQVcsRUFBRTtZQUNmLElBQUksb0NBQWlCLENBQUMsSUFBSSxFQUFFLDhCQUE4QixFQUFFO2dCQUMxRCxZQUFZLEVBQUUseUNBQXlDO2dCQUN2RCxRQUFRLEVBQUU7b0JBQ1IsT0FBTyxFQUFFLEtBQUs7b0JBQ2QsTUFBTSxFQUFFLGtCQUFrQjtvQkFDMUIsVUFBVSxFQUFFO3dCQUNWLGNBQWMsRUFBRSxJQUFJLENBQUMsY0FBYzt3QkFDbkMsS0FBSyxFQUFFLElBQUk7cUJBQ1o7b0JBQ0Qsa0JBQWtCLEVBQUUscUNBQWtCLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUM7aUJBQzlEO2dCQUNELE1BQU0sRUFBRSwwQ0FBdUIsQ0FBQyxZQUFZLENBQUM7b0JBQzNDLFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUM7aUJBQ2hDLENBQUM7YUFDSCxDQUFDLENBQUM7U0FDSjtJQUNILENBQUM7SUFFTSxTQUFTLENBQUMsRUFBVSxFQUFFLE9BQXlCO1FBQ3BELE1BQU0sRUFBRSxVQUFVLEVBQUUsUUFBUSxFQUFFLFNBQVMsRUFBRSxHQUFHLE9BQU8sQ0FBQztRQUVwRCxNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsb0JBQW9CLENBQUMsRUFBRSxFQUFFLEVBQUUsU0FBUyxFQUFFLENBQUMsQ0FBQztRQUUxRCxNQUFNLGNBQWMsR0FBRyxJQUFJLHFCQUFRLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO1lBQzNELE9BQU8sRUFBRSxvQkFBTyxDQUFDLFdBQVc7WUFDNUIsSUFBSSxFQUFFLGlCQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1lBQ3hFLE9BQU8sRUFBRSxlQUFlO1NBQ3pCLENBQUMsQ0FBQztRQUVILGNBQWMsQ0FBQyxlQUFlLENBQzVCLElBQUkseUJBQWUsQ0FBQztZQUNsQixNQUFNLEVBQUUsZ0JBQU0sQ0FBQyxLQUFLO1lBQ3BCLE9BQU8sRUFBRSxDQUFDLGFBQWEsQ0FBQztZQUN4QixTQUFTLEVBQUUsQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDO1NBQ2pDLENBQUMsQ0FDSCxDQUFDO1FBRUYsSUFBSSxDQUFDLFNBQVMsQ0FDWixJQUFJLG1DQUFjLENBQUMsY0FBYyxFQUFFO1lBQ2pDLEtBQUssRUFBRSw0QkFBZSxDQUFDLFVBQVUsQ0FBQztnQkFDaEMsT0FBTyxFQUFFLHVCQUFVLENBQUMsT0FBTztnQkFDM0IsTUFBTSxFQUFFLHVCQUFVLENBQUMsTUFBTTtnQkFDekIsSUFBSSxFQUFFLHVCQUFVLENBQUMsSUFBSTtnQkFDckIsY0FBYyxFQUFFLHVCQUFVLENBQUMsUUFBUSxDQUFDLDBCQUEwQixDQUFDO2dCQUMvRCxXQUFXLEVBQUUsdUJBQVUsQ0FBQyxRQUFRLENBQUMsdUJBQXVCLENBQUM7Z0JBQ3pELFNBQVMsRUFBRSx1QkFBVSxDQUFDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQztnQkFDckQsb0JBQW9CLEVBQUUsdUJBQVUsQ0FBQyxRQUFRLENBQ3ZDLGtDQUFrQyxDQUNuQztnQkFDRCxRQUFRLEVBQUUsUUFBUTtnQkFDbEIsYUFBYSxFQUFFLFVBQVUsQ0FBQyxRQUFRO2FBQ25DLENBQUM7U0FDSCxDQUFDLENBQ0gsQ0FBQztRQUVGLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztDQUNGO0FBbkVELDBDQW1FQztBQUVELElBQVksUUFPWDtBQVBELFdBQVksUUFBUTtJQUNsQixpQ0FBcUIsQ0FBQTtJQUNyQix5QkFBYSxDQUFBO0lBQ2IsNkJBQWlCLENBQUE7SUFDakIsdUJBQVcsQ0FBQTtJQUNYLDJDQUErQixDQUFBO0lBQy9CLG1DQUF1QixDQUFBO0FBQ3pCLENBQUMsRUFQVyxRQUFRLEdBQVIsZ0JBQVEsS0FBUixnQkFBUSxRQU9uQiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QsIFJlbW92YWxQb2xpY3kgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7XG4gIEF3c0N1c3RvbVJlc291cmNlLFxuICBBd3NDdXN0b21SZXNvdXJjZVBvbGljeSxcbiAgUGh5c2ljYWxSZXNvdXJjZUlkLFxufSBmcm9tICdAYXdzLWNkay9jdXN0b20tcmVzb3VyY2VzJztcbmltcG9ydCB7IFJlcG9zaXRvcnksIFJlcG9zaXRvcnlQcm9wcyB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1lY3InO1xuaW1wb3J0IHsgUnVsZSwgRXZlbnRGaWVsZCwgUnVsZVRhcmdldElucHV0IH0gZnJvbSAnQGF3cy1jZGsvYXdzLWV2ZW50cyc7XG5pbXBvcnQgeyBMYW1iZGFGdW5jdGlvbiB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1ldmVudHMtdGFyZ2V0cyc7XG5pbXBvcnQgeyBDb2RlLCBGdW5jdGlvbiwgUnVudGltZSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0IHsgSVRvcGljIH0gZnJvbSAnQGF3cy1jZGsvYXdzLXNucyc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIEVmZmVjdCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuXG5leHBvcnQgaW50ZXJmYWNlIE9uRmluZGluZ09wdGlvbnMge1xuICByZWFkb25seSBhbGFybVRvcGljOiBJVG9waWM7XG5cbiAgcmVhZG9ubHkgc2V2ZXJpdHk6IFNldmVyaXR5O1xuXG4gIC8qKlxuICAgKiBPbmx5IHdhdGNoIGNoYW5nZXMgdG8gdGhlIGltYWdlIHRhZ3Mgc3BlY2lmaWVkLlxuICAgKiBMZWF2ZSBpdCB1bmRlZmluZWQgdG8gd2F0Y2ggdGhlIGZ1bGwgcmVwb3NpdG9yeS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBXYXRjaCB0aGUgY2hhbmdlcyB0byB0aGUgcmVwb3NpdG9yeSB3aXRoIGFsbCBpbWFnZSB0YWdzXG4gICAqL1xuICByZWFkb25seSBpbWFnZVRhZ3M/OiBzdHJpbmdbXTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBJbWFnZVJlcG9zaXRvcnlQcm9wcyBleHRlbmRzIFJlcG9zaXRvcnlQcm9wcyB7XG4gIC8qKlxuICAgKiBJZiBhIHJlcG9zaXRvcnkgY29udGFpbnMgaW1hZ2VzLCBmb3JjZXMgdGhlIGRlbGV0aW9uIGR1cmluZyBzdGFjayBkZWxldGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGZvcmNlRGVsZXRlPzogYm9vbGVhbjtcbn1cblxuZXhwb3J0IGNsYXNzIEltYWdlUmVwb3NpdG9yeSBleHRlbmRzIFJlcG9zaXRvcnkge1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogSW1hZ2VSZXBvc2l0b3J5UHJvcHMgPSB7fSkge1xuICAgIGNvbnN0IHsgZm9yY2VEZWxldGUgPSBmYWxzZSwgLi4ucmVzdCB9ID0gcHJvcHM7XG5cbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHJlbW92YWxQb2xpY3k6IGZvcmNlRGVsZXRlID8gUmVtb3ZhbFBvbGljeS5ERVNUUk9ZIDogdW5kZWZpbmVkLFxuICAgICAgLi4ucmVzdCxcbiAgICB9KTtcblxuICAgIGlmIChmb3JjZURlbGV0ZSkge1xuICAgICAgbmV3IEF3c0N1c3RvbVJlc291cmNlKHRoaXMsICdGb3JjZUltYWdlUmVwb3NpdG9yeURlbGV0aW9uJywge1xuICAgICAgICByZXNvdXJjZVR5cGU6ICdDdXN0b206OkVDUkZvcmNlSW1hZ2VSZXBvc2l0b3J5RGVsZXRpb24nLFxuICAgICAgICBvbkRlbGV0ZToge1xuICAgICAgICAgIHNlcnZpY2U6ICdFQ1InLFxuICAgICAgICAgIGFjdGlvbjogJ2RlbGV0ZVJlcG9zaXRvcnknLFxuICAgICAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgICAgIHJlcG9zaXRvcnlOYW1lOiB0aGlzLnJlcG9zaXRvcnlOYW1lLFxuICAgICAgICAgICAgZm9yY2U6IHRydWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBwaHlzaWNhbFJlc291cmNlSWQ6IFBoeXNpY2FsUmVzb3VyY2VJZC5vZih0aGlzLnJlcG9zaXRvcnlBcm4pLFxuICAgICAgICB9LFxuICAgICAgICBwb2xpY3k6IEF3c0N1c3RvbVJlc291cmNlUG9saWN5LmZyb21TZGtDYWxscyh7XG4gICAgICAgICAgcmVzb3VyY2VzOiBbdGhpcy5yZXBvc2l0b3J5QXJuXSxcbiAgICAgICAgfSksXG4gICAgICB9KTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgb25GaW5kaW5nKGlkOiBzdHJpbmcsIG9wdGlvbnM6IE9uRmluZGluZ09wdGlvbnMpOiBSdWxlIHtcbiAgICBjb25zdCB7IGFsYXJtVG9waWMsIHNldmVyaXR5LCBpbWFnZVRhZ3MgfSA9IG9wdGlvbnM7XG5cbiAgICBjb25zdCBydWxlID0gdGhpcy5vbkltYWdlU2NhbkNvbXBsZXRlZChpZCwgeyBpbWFnZVRhZ3MgfSk7XG5cbiAgICBjb25zdCBzZXZlcml0eUZpbHRlciA9IG5ldyBGdW5jdGlvbih0aGlzLCAnU2V2aWVyaXR5RmlsdGVyJywge1xuICAgICAgcnVudGltZTogUnVudGltZS5OT0RFSlNfMTJfWCxcbiAgICAgIGNvZGU6IENvZGUuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdsYW1iZGFzJywgJ3NldmVyaXR5LWZpbHRlcicpKSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICB9KTtcblxuICAgIHNldmVyaXR5RmlsdGVyLmFkZFRvUm9sZVBvbGljeShcbiAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWydzbnM6UHVibGlzaCddLFxuICAgICAgICByZXNvdXJjZXM6IFthbGFybVRvcGljLnRvcGljQXJuXSxcbiAgICAgIH0pLFxuICAgICk7XG5cbiAgICBydWxlLmFkZFRhcmdldChcbiAgICAgIG5ldyBMYW1iZGFGdW5jdGlvbihzZXZlcml0eUZpbHRlciwge1xuICAgICAgICBldmVudDogUnVsZVRhcmdldElucHV0LmZyb21PYmplY3Qoe1xuICAgICAgICAgIGFjY291bnQ6IEV2ZW50RmllbGQuYWNjb3VudCxcbiAgICAgICAgICByZWdpb246IEV2ZW50RmllbGQucmVnaW9uLFxuICAgICAgICAgIHRpbWU6IEV2ZW50RmllbGQudGltZSxcbiAgICAgICAgICByZXBvc2l0b3J5TmFtZTogRXZlbnRGaWVsZC5mcm9tUGF0aCgnJC5kZXRhaWwucmVwb3NpdG9yeS1uYW1lJyksXG4gICAgICAgICAgaW1hZ2VEaWdlc3Q6IEV2ZW50RmllbGQuZnJvbVBhdGgoJyQuZGV0YWlsLmltYWdlLWRpZ2VzdCcpLFxuICAgICAgICAgIGltYWdlVGFnczogRXZlbnRGaWVsZC5mcm9tUGF0aCgnJC5kZXRhaWwuaW1hZ2UtdGFncycpLFxuICAgICAgICAgIGZpbmRpbmdTZXZlcml5Q291bnRzOiBFdmVudEZpZWxkLmZyb21QYXRoKFxuICAgICAgICAgICAgJyQuZGV0YWlsLmZpbmRpbmctc2V2ZXJpdHktY291bnRzJyxcbiAgICAgICAgICApLFxuICAgICAgICAgIHNldmVyaXR5OiBzZXZlcml0eSxcbiAgICAgICAgICBhbGFybVRvcGljQXJuOiBhbGFybVRvcGljLnRvcGljQXJuLFxuICAgICAgICB9KSxcbiAgICAgIH0pLFxuICAgICk7XG5cbiAgICByZXR1cm4gcnVsZTtcbiAgfVxufVxuXG5leHBvcnQgZW51bSBTZXZlcml0eSB7XG4gIENSSVRJQ0FMID0gJ0NSSVRJQ0FMJyxcbiAgSElHSCA9ICdISUdIJyxcbiAgTUVESVVNID0gJ01FRElVTScsXG4gIExPVyA9ICdMT1cnLFxuICBJTkZPUk1BVElPTkFMID0gJ0lORk9STUFUSU9OQUwnLFxuICBVTkRFRklORUQgPSAnVU5ERUZJTkVEJyxcbn1cbiJdfQ==