# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pgq', 'pgq.migrations']

package_data = \
{'': ['*']}

install_requires = \
['django>=3.2', 'dunamai>=1.13.2,<2.0.0']

setup_kwargs = {
    'name': 'mishka-queue',
    'version': '0.0.0',
    'description': 'Mishka Queue - A django task queue',
    'long_description': '# Mishka Queue - A django task queue\n\nMishka queue is a task queue system for Django backed by postgres.\n\nIt was forked from:\n- https://github.com/SweetProcess/django-pg-queue, which in turn was forked from\n- https://github.com/gavinwahl/django-postgres-queue by Gavin Wahl.\n\n# Why postgres?\n\nI thought you were never supposed to use an RDBMS as a queue? Well,\npostgres has some features that make it not as bad as you might think,\nit has some compelling advantages.\n\n-   Transactional behavior and reliability.\n\n    Adding tasks is atomic with respect to other database work. There is\n    no need to use `transaction.on_commit` hooks and there is no risk of\n    a transaction being committed but the tasks it queued being lost.\n\n    Processing tasks is atomic with respect to other database work.\n    Database work done by a task will either be committed, or the task\n    will not be marked as processed, no exceptions. If the task only\n    does database work, you achieve true exactly-once message\n    processing.\n\n-   Operational simplicity\n\n    By reusing the durable, transactional storage that we\'re already\n    using anyway, there\'s no need to configure, monitor, and backup\n    another stateful service. For small teams and light workloads, this\n    is the right trade-off.\n\n-   Easy introspection\n\n    Since tasks are stored in a database table, it\'s easy to query and\n    monitor the state of the queue.\n\n-   Safety\n\n    By using postgres transactions, there is no possibility of jobs\n    being left in a locked or ambiguous state if a worker dies. Tasks\n    immediately become available for another worker to pick up. You can\n    even `kill -9` a worker and be sure your database and queue will be\n    left in a consistent state.\n\n-   Priority queues\n\n    Since ordering is specified explicitly when selecting the next task\n    to work on, it\'s easy to ensure high-priority tasks are processed\n    first.\n\n-   Queues\n\n    Simply implemented by allowing filtering by a queue name in the\n    query.\n\n# Disadvantages\n\n-   Lower throughput than a dedicated queue server.\n-   Harder to scale a relational database than a dedicated queue server.\n-   Thundering herd. Postgres will notify all workers who LISTEN for the\n    same name.\n-   With at-least-once delivery, a postgres transaction has to be held\n    open for the duration of the task. For long running tasks, this can\n    cause table bloat and performance problems.\n-   When a task crashes or raises an exception under at-least-once\n    delivery, it immediately becomes eligible to be retried. If you want\n    to implement a retry delay, you must catch exceptions and requeue\n    the task with a delay. If your task crashes without throwing an\n    exception (eg SIGKILL), you could end up in an endless retry loop\n    that prevents other tasks from being processed.\n\n# How it works\n\ndjango-pg-queue is able to claim, process, and remove a task in a single\n(simplified) query.\n\n``` sql\nDELETE FROM pgq_job\nWHERE id = (\n    SELECT id\n    FROM pgq_job\n    WHERE execute_at <= now()\n    ORDER BY priority DESC, created_at\n    FOR UPDATE SKIP LOCKED\n    LIMIT 1\n)\nRETURNING *;\n```\n\nAs soon as this query runs, the task is unable to be claimed by other\nworkers. When the transaction commits, the task will be deleted. If the\ntransaction rolls back or the worker crashes, the task will immediately\nbecome available for another worker.\n\nTo achieve at-least-once delivery, we begin a transaction, process the\ntask, then commit the transaction. For at-most-once, we claim the task\nand immediately commit the transaction, then process the task. For tasks\nthat don\'t have any external effects and only do database work, the\nat-least-once behavior is actually exactly-once (because both the\nclaiming of the job and the database work will commit or rollback\ntogether).\n\n# Comparison to Celery\n\nmishka queue fills the same role as Celery. You must use postgres as the\nbackend and the library is small enough that you can read and understand\nall the code.\n\n# A note on the use of `AtLeastOnceQueue` and Django\'s `transaction.on_commit()`\n\nA failure in an `on_commit()` callback will not cause that job to be\nretried when using an `AtLeastOnceQueue` (usually a job in an\n`AtLeastOnceQueue` queue will remain in the queue if the job fails).\nThis is because `on_commit()` callbacks are executed after the\ntransaction has been committed and, for django-pg-queue, the job is\nremoved from the queue when the transaction commits.\n\nIf you require more certainty that the code in an `on_commit()` callback\nis executed successfully, you may need to ensure it is idempotent and\ncall it from within the job rather than using `on_commit()`.\n\n## Usage\n\n# Requirements\n\ndjango-pg-queue requires Python 3, at least postgres 9.5 and at least\nDjango 2.1.\n\n# Installation\n\nInstall with pip:\n\n    pip install django-pg-queue\n\nThen add `\'pgq\'` to your `INSTALLED_APPS`. Run `manage.py migrate` to\ncreate the jobs table.\n\nInstantiate a queue object. This can go wherever you like and be named\nwhatever you like. For example, `someapp/queue.py`:\n\n``` python\nfrom pgq.queue import AtLeastOnceQueue\n\nqueue = AtLeastOnceQueue(\n    tasks={\n        # ...\n    },\n    queue=\'my-queue\',\n    notify_channel=\'my-queue\',\n)\n```\n\nYou will need to import this queue instance to queue or process tasks.\nUse `AtLeastOnceQueue` for at-least-once delivery, or `AtMostOnceQueue`\nfor at-most-once delivery.\n\ndjango-pg-queue comes with a management command base class that you can\nuse to consume your tasks. It can be called whatever you like, for\nexample in a `someapp/managment/commands/worker.py`:\n\n``` python\nfrom pgq.commands import Worker\n\nfrom someapp.queue import queue\n\nclass Command(Worker):\n    queue = queue\n```\n\nThen you can run `manage.py worker` to start your worker.\n\nA task function takes two arguments -- the queue instance in use, and\nthe Job instance for this task. The function can be defined anywhere and\ncalled whatever you like. Here\'s an example:\n\n``` python\nfrom pgq.decorators import task\n\nfrom .queues import queue\n\n@task(queue)\ndef debug_task(queue, job):\n    print(job.args)\n```\n\nInstead of using the task decorator, you can manually register it as a\ntask. Add it to your queue instance when it is being created:\n\n``` python\nqueue = AtLeastOnceQueue(tasks={\n    \'debug_task\': debug_task,\n}, queue=\'my-queue\')\n```\n\nThe key is the task name, used to queue the task. It doesn\'t have to\nmatch the function name.\n\nTo queue the task, if you used the task decorator you may:\n\n``` python\ndebug_task.enqueue({\'some_args\': 0})\n```\n\nTo manually queue the task, use the `enqueue` method on your queue\ninstance:\n\n``` python\nqueue.enqueue(\'debug_task\', {\'some_args\': 0})\n```\n\nAssuming you have a worker running for this queue, the task will be run\nimmediately. The second argument must be a single json-serializeable\nvalue and will be available to the task as `job.args`.\n\nTasks registered using the `@task` decorator will only be available on\nthe queue if the file in which the task is defined has been imported. If\nyour worker doesn\'t import the file containing the `@task` decorators\nsomewhere, the tasks will not be available for dispatch. Importing files\nin the `apps.py` `AppConfig.ready()` method will ensure that the tasks\nare always available on the queue without having to import them in your\nworker just for the import side effects.\n\n``` python\n# Contents of someapp/apps.py\nfrom django.apps import AppConfig\n\nclass SomeAppAppConfig(AppConfig):\n    def ready(self):\n        # Tasks registered with @task are defined in this import\n        import someapp.tasks\n```\n\n# Multiple Queues\n\nYou may run multiple queues and workers may each listen to a queue. You\ncan have multiple workers listening to the same queue too. A queue is\nimplemented as a CharField in the database. The queue would simply\nfilter for jobs matching its queue name.\n\n# Bulk Enqueue\n\nMany jobs can be efficiently created using `bulk_enqueue()` which\naccepts one task name for all the jobs being created and a list of\ndictionaries containing `args` for the task to execute with and,\noptionally, `priority` and `execute_at` for that particular job.\n\n``` python\nqueue.bulk_enqueue(\n    \'debug_task\',\n    [\n        {\'args\': {\'some_args\': 0}},\n        {\n            \'args\': {\'some_args\': 10}\n            \'priority\': 10,\n            \'execute_at\': timezone.now() + timedelta(days=1),\n        },\n    ]\n)\n```\n\n# Monitoring\n\nTasks are just database rows stored in the `pgq_job` table, so you can\nmonitor the system with SQL.\n\nTo get a count of current tasks:\n\n``` sql\nSELECT queue, count(*) FROM pgq_job WHERE execute_at <= now() GROUP BY queue\n```\n\nThis will include both tasks ready to process and tasks currently being\nprocessed. To see tasks currently being processed, we need visibility\ninto postgres row locks. This can be provided by the [pgrowlocks\nextension](https://www.postgresql.org/docs/9.6/static/pgrowlocks.html).\nOnce installed, this query will count currently-running tasks:\n\n``` sql\nSELECT queue, count(*)\nFROM pgrowlocks(\'pgq_job\')\nWHERE \'For Update\' = ANY(modes)\nGROUP BY queue;\n```\n\nYou could join the results of `pgrowlocks` with `pgq_job` to get the\nfull list of tasks in progress if you want.\n\n# Logging\n\ndjango-pg-queue logs through Python\'s logging framework, so can be\nconfigured with the `LOGGING` dict in your Django settings. It will not\nlog anything under the default config, so be sure to configure some form\nof logging. Everything is logged under the `pgq` namespace. Here is an\nexample configuration that will log INFO level messages to stdout:\n\n``` python\nLOGGING = {\n    \'version\': 1,\n    \'root\': {\n        \'level\': \'DEBUG\',\n        \'handlers\': [\'console\'],\n    },\n    \'formatters\': {\n        \'verbose\': {\n            \'format\': \'%(levelname)s %(asctime)s %(module)s %(process)d %(thread)d %(message)s\',\n        },\n    },\n    \'handlers\': {\n        \'console\': {\n            \'level\': \'INFO\',\n            \'class\': \'logging.StreamHandler\',\n            \'formatter\': \'verbose\',\n        },\n    },\n    \'loggers\': {\n        \'pgq\': {\n            \'handlers\': [\'console\'],\n            \'level\': \'INFO\',\n            \'propagate\': False,\n        },\n    }\n}\n```\n\nIt would also be sensible to log WARNING and higher messages to\nsomething like Sentry:\n\n``` python\nLOGGING = {\n    \'version\': 1,\n    \'root\': {\n        \'level\': \'INFO\',\n        \'handlers\': [\'sentry\', \'console\'],\n    },\n    \'formatters\': {\n        \'verbose\': {\n            \'format\': \'%(levelname)s %(asctime)s %(module)s %(process)d %(thread)d %(message)s\',\n        },\n    },\n    \'handlers\': {\n        \'console\': {\n            \'level\': \'INFO\',\n            \'class\': \'logging.StreamHandler\',\n            \'formatter\': \'verbose\',\n        },\n        \'sentry\': {\n            \'level\': \'WARNING\',\n            \'class\': \'raven.contrib.django.handlers.SentryHandler\',\n        },\n    },\n    \'loggers\': {\n        \'pgq\': {\n            \'level\': \'INFO\',\n            \'handlers\': [\'console\', \'sentry\'],\n            \'propagate\': False,\n        },\n    },\n}\n```\n\nYou could also log to a file by using the built-in\n`logging.FileHandler`.\n\n## Useful Recipes\n\nThese recipes aren\'t officially supported features of <span\nclass="title-ref">django-pg-queue</span>. We provide them so that you\ncan mimick some of the common features in other task queues.\n\n# <span class="title-ref">CELERY_ALWAYS_EAGER</span>\n\nCelery uses the <span class="title-ref">CELERY_ALWAYS_EAGER</span>\nsetting to run a task immediately, without queueing it for a worker. It\ncould be used during tests, and while debugging in a development\nenvironment with any workers turned off.\n\n``` python\nclass EagerAtLeastOnceQueue(AtLeastOnceQueue):\n    def enqueue(self, *args, **kwargs):\n        job = super().enqueue(*args, **kwargs)\n        if settings.QUEUE_ALWAYS_EAGER:\n            self.run_job(job)\n        return job\n```\n',
    'author': 'Your Name',
    'author_email': 'b.howes@neuronostics.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
