from functools import partial
import logging

import msgpack
import msgpack_numpy as mpn

from confluent_kafka.cimpl import KafkaException
import numpy as np
import pickle
import pytest

from bluesky_kafka import Publisher, BlueskyConsumer
from bluesky_kafka.tests.conftest import get_all_documents_from_queue
from bluesky.plans import count
from event_model import sanitize_doc

# mpn.patch() is recommended by msgpack-numpy as a way
# to patch msgpack but it caused a utf-8 decode error
mpn.patch()

logging.getLogger("bluesky.kafka").setLevel("DEBUG")


# the Kafka test broker should be configured with
# KAFKA_CFG_AUTO_CREATE_TOPICS_ENABLE=true


def test_producer_config():
    test_topic = "test.producer.config"
    kafka_publisher = Publisher(
        topic=test_topic,
        bootstrap_servers="1.2.3.4:9092",
        key="kafka-unit-test-key",
        # work with a single broker
        producer_config={
            "bootstrap.servers": "5.6.7.8:9092",
            "acks": 1,
            "enable.idempotence": False,
            "request.timeout.ms": 5000,
        },
    )

    assert (
        kafka_publisher._producer_config["bootstrap.servers"]
        == "1.2.3.4:9092,5.6.7.8:9092"
    )


def test_get_cluster_metadata(publisher_factory):
    # the topic test.get.cluster.metadata will be created
    # by the call to publisher.get_cluster_metadata
    # if automatic topic creation is enabled
    # otherwise this test will fail
    publisher = publisher_factory(topic="test.get.cluster.metadata")
    cluster_metadata = publisher.get_cluster_metadata()
    assert "test.get.cluster.metadata" in cluster_metadata.topics


def test_get_cluster_metadata_failure(publisher_factory):
    publisher = publisher_factory(
        topic="test.get.cluster.metadata.failure",
        bootstrap_servers="5.6.7.8:9092"
    )
    with pytest.raises(KafkaException):
        publisher.get_cluster_metadata()


def test_consumer_config():
    test_topic = "test.consumer.config"
    bluesky_consumer = BlueskyConsumer(
        topics=[test_topic],
        bootstrap_servers="1.2.3.4:9092",
        group_id="abc",
        consumer_config={
            "bootstrap.servers": "5.6.7.8:9092",
            "auto.offset.reset": "latest",
        },
    )

    assert (
        bluesky_consumer._consumer_config["bootstrap.servers"]
        == "1.2.3.4:9092,5.6.7.8:9092"
    )


def test_bad_consumer_config():
    test_topic = "test.bad.consumer.config"
    with pytest.raises(ValueError) as excinfo:
        BlueskyConsumer(
            topics=[test_topic],
            bootstrap_servers="1.2.3.4:9092",
            group_id="abc",
            consumer_config={
                "bootstrap.servers": "5.6.7.8:9092",
                "auto.offset.reset": "latest",
                "group.id": "raise an exception!",
            },
        )
        assert (
            "do not specify 'group.id' in consumer_config, use only the 'group_id' argument"
            in excinfo.value
        )


@pytest.mark.parametrize(
    "serializer, deserializer",
    [(pickle.dumps, pickle.loads), (msgpack.dumps, msgpack.loads)],
)
def test_kafka_remote_dispatcher(
    RE,
    hw,
    serializer,
    deserializer,
    publisher_factory,
    remote_dispatcher_process_factory,
    external_process_document_queue,
):
    # COMPONENT 1
    # a Kafka broker must be running
    # in addition the topic "test.remote.dispatcher" must exist
    # or the broker must be configured to create topics on demand (recommended)

    # COMPONENT 2
    # Run a RemoteDispatcher on a separate process. Pass the documents
    # it receives over a Queue to this process so we can count them for
    # our test.
    test_topic = "test.remote.dispatcher"
    with external_process_document_queue(
        topics=[test_topic],
        deserializer=deserializer,
        process_factory=remote_dispatcher_process_factory,
    ) as document_queue:
        # COMPONENT 3
        # Set up a RunEngine in this process that will
        # send all documents to a bluesky_kafka.Publisher
        # and accumulate all documents in the local_documents list
        kafka_publisher = publisher_factory(
            topic=test_topic, serializer=serializer, flush_on_stop_doc=True
        )
        RE.subscribe(kafka_publisher)

        local_documents = []
        RE.subscribe(
            lambda local_name, local_doc: local_documents.append(
                (local_name, local_doc)
            )
        )

        # test that numpy data is transmitted correctly
        md = {
            "numpy_data": {"nested": np.array([1, 2, 3])},
            "numpy_scalar": np.float64(3),
            "numpy_array": np.ones((3, 3)),
        }

        # documents will be generated by this plan
        # and published by the Kafka Publisher
        RE(count([hw.det]), md=md)

        # retrieve the documents published by the Kafka broker
        remote_documents = get_all_documents_from_queue(document_queue=document_queue)

        # sanitize_doc normalizes some document data, such as numpy arrays, that are
        # problematic for direct comparison of documents by "assert"
        sanitized_local_documents = [sanitize_doc(doc) for doc in local_documents]
        sanitized_remote_documents = [sanitize_doc(doc) for doc in remote_documents]

        assert len(sanitized_remote_documents) == len(sanitized_local_documents)
        assert sanitized_remote_documents == sanitized_local_documents


@pytest.mark.parametrize(
    "serializer, deserializer",
    [(pickle.dumps, pickle.loads), (msgpack.dumps, msgpack.loads)],
)
def test_bluesky_consumer(
    RE,
    hw,
    serializer,
    deserializer,
    publisher_factory,
    consumer_process_factory,
    external_process_document_queue,
):
    # COMPONENT 1
    # a Kafka broker must be running
    # in addition the broker must have topic "test.bluesky.consumer"
    # or be configured to create topics on demand (recommended)

    # COMPONENT 2
    # Run a BlueskyConsumer polling loop in a separate process.
    # Pass the documents it receives over a Queue to this process
    # and compare them against the documents published directly
    # by the RunEngine.
    test_topic = "test.bluesky.consumer"
    with external_process_document_queue(
        topics=[test_topic],
        deserializer=deserializer,
        process_factory=partial(
            consumer_process_factory, consumer_factory=BlueskyConsumer
        ),
    ) as document_queue:
        # COMPONENT 3
        # Set up a RunEngine in this process that will
        # send all documents to a bluesky_kafka.Publisher
        # and accumulate all documents in the local_documents list
        kafka_publisher = publisher_factory(
            topic=test_topic, serializer=serializer, flush_on_stop_doc=True
        )
        RE.subscribe(kafka_publisher)

        local_documents = []
        RE.subscribe(
            lambda local_name, local_doc: local_documents.append(
                (local_name, local_doc)
            )
        )

        # test that numpy data is transmitted correctly
        md = {
            "numpy_data": {"nested": np.array([1, 2, 3])},
            "numpy_scalar": np.float64(3),
            "numpy_array": np.ones((3, 3)),
        }

        # documents will be generated by this plan
        # and published by the Kafka Publisher
        RE(count([hw.det]), md=md)

        # retrieve the documents published by the Kafka broker
        remote_documents = get_all_documents_from_queue(document_queue=document_queue)

        # sanitize_doc normalizes some document data, such as numpy arrays, that are
        # problematic for direct comparison of documents by "assert"
        sanitized_local_documents = [sanitize_doc(doc) for doc in local_documents]
        sanitized_remote_documents = [sanitize_doc(doc) for doc in remote_documents]

        assert len(sanitized_remote_documents) == len(sanitized_local_documents)
        assert sanitized_remote_documents == sanitized_local_documents
