#
# Copyright (c) 2022, NVIDIA CORPORATION.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

from typing import Protocol, runtime_checkable


@runtime_checkable
class MerlinModel(Protocol):
    """Protocol for a merlin model.

    This defines common methods that should be implemented on all model implementations.
    Including those with different frameworks (e.g. Tensorflow, XGBoost, Pytorch)
    """

    def save(self, path) -> None:
        """Save the model to a local path.

        Parameters
        ----------
        path : Union[str, os.PathLike]
        """
        ...

    @classmethod
    def load(cls, path):
        """Load the model from a path-like argument provided where a model was previously saved.

        Parameters
        ----------
        path : Union[str, os.PathLike]
            A path correspoonding to the directory where a model was previously saved.
        """
        ...

    def fit(self, dataset, *args, **kwargs):
        """Fit the model on the provided dataset.

        Parameters
        ----------
        dataset : merlin.io.Dataset
            The training dataset to be used to fit the model.
        """
        ...

    def evaluate(self, dataset, *args, **kwargs):
        """Return evaluation metrics on a dataset.

        Parameters
        ----------
        dataset : merlin.io.Dataset
            The evaluation dataset to be used to compute metrics.
        """
        ...

    def predict(self, dataset, *args, **kwargs):
        """Return predictions generated by the model.

        Parameters
        ----------
        dataset : merlin.io.Dataset
            The dataset to generate predictions from.
        """
        ...
