"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Expires = exports.StorageClass = exports.ServerSideEncryption = exports.CacheControl = exports.BucketDeployment = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const efs = require("@aws-cdk/aws-efs");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const lambda_layer_awscli_1 = require("@aws-cdk/lambda-layer-awscli");
const case_1 = require("case");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
// tag key has a limit of 128 characters
const CUSTOM_RESOURCE_OWNER_TAG = 'aws-cdk:cr-owned';
/**
 * `BucketDeployment` populates an S3 bucket with the contents of .zip files from
 * other S3 buckets or from local disk
 */
class BucketDeployment extends core_1.Construct {
    constructor(scope, id, props) {
        var _d, _e;
        super(scope, id);
        this.requestDestinationArn = false;
        jsiiDeprecationWarnings._aws_cdk_aws_s3_deployment_BucketDeploymentProps(props);
        if (props.distributionPaths) {
            if (!props.distribution) {
                throw new Error('Distribution must be specified if distribution paths are specified');
            }
            if (!cdk.Token.isUnresolved(props.distributionPaths)) {
                if (!props.distributionPaths.every(distributionPath => cdk.Token.isUnresolved(distributionPath) || distributionPath.startsWith('/'))) {
                    throw new Error('Distribution paths must start with "/"');
                }
            }
        }
        if (props.useEfs && !props.vpc) {
            throw new Error('Vpc must be specified if useEfs is set');
        }
        const accessPointPath = '/lambda';
        let accessPoint;
        if (props.useEfs && props.vpc) {
            const accessMode = '0777';
            const fileSystem = this.getOrCreateEfsFileSystem(scope, {
                vpc: props.vpc,
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            accessPoint = fileSystem.addAccessPoint('AccessPoint', {
                path: accessPointPath,
                createAcl: {
                    ownerUid: '1001',
                    ownerGid: '1001',
                    permissions: accessMode,
                },
                posixUser: {
                    uid: '1001',
                    gid: '1001',
                },
            });
            accessPoint.node.addDependency(fileSystem.mountTargetsAvailable);
        }
        // Making VPC dependent on BucketDeployment so that CFN stack deletion is smooth.
        // Refer comments on https://github.com/aws/aws-cdk/pull/15220 for more details.
        if (props.vpc) {
            this.node.addDependency(props.vpc);
        }
        const mountPath = `/mnt${accessPointPath}`;
        const handler = new lambda.SingletonFunction(this, 'CustomResourceHandler', {
            uuid: this.renderSingletonUuid(props.memoryLimit, props.vpc),
            code: lambda.Code.fromAsset(path.join(__dirname, 'lambda')),
            layers: [new lambda_layer_awscli_1.AwsCliLayer(this, 'AwsCliLayer')],
            runtime: lambda.Runtime.PYTHON_3_7,
            environment: props.useEfs ? {
                MOUNT_PATH: mountPath,
            } : undefined,
            handler: 'index.handler',
            lambdaPurpose: 'Custom::CDKBucketDeployment',
            timeout: cdk.Duration.minutes(15),
            role: props.role,
            memorySize: props.memoryLimit,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
            filesystem: accessPoint ? lambda.FileSystem.fromEfsAccessPoint(accessPoint, mountPath) : undefined,
            logRetention: props.logRetention,
        });
        const handlerRole = handler.role;
        if (!handlerRole) {
            throw new Error('lambda.SingletonFunction should have created a Role');
        }
        const sources = props.sources.map((source) => source.bind(this, { handlerRole }));
        props.destinationBucket.grantReadWrite(handler);
        if (props.distribution) {
            handler.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['cloudfront:GetInvalidation', 'cloudfront:CreateInvalidation'],
                resources: ['*'],
            }));
        }
        // to avoid redundant stack updates, only include "SourceMarkers" if one of
        // the sources actually has markers.
        const hasMarkers = sources.some(source => source.markers);
        const crUniqueId = `CustomResource${this.renderUniqueId(props.memoryLimit, props.vpc)}`;
        this.cr = new cdk.CustomResource(this, crUniqueId, {
            serviceToken: handler.functionArn,
            resourceType: 'Custom::CDKBucketDeployment',
            properties: {
                SourceBucketNames: sources.map(source => source.bucket.bucketName),
                SourceObjectKeys: sources.map(source => source.zipObjectKey),
                SourceMarkers: hasMarkers ? sources.map(source => { var _d; return (_d = source.markers) !== null && _d !== void 0 ? _d : {}; }) : undefined,
                DestinationBucketName: props.destinationBucket.bucketName,
                DestinationBucketKeyPrefix: props.destinationKeyPrefix,
                RetainOnDelete: props.retainOnDelete,
                Prune: (_d = props.prune) !== null && _d !== void 0 ? _d : true,
                Exclude: props.exclude,
                Include: props.include,
                UserMetadata: props.metadata ? mapUserMetadata(props.metadata) : undefined,
                SystemMetadata: mapSystemMetadata(props),
                DistributionId: (_e = props.distribution) === null || _e === void 0 ? void 0 : _e.distributionId,
                DistributionPaths: props.distributionPaths,
                // Passing through the ARN sequences dependencees on the deployment
                DestinationBucketArn: cdk.Lazy.string({ produce: () => this.requestDestinationArn ? props.destinationBucket.bucketArn : undefined }),
            },
        });
        let prefix = props.destinationKeyPrefix ?
            `:${props.destinationKeyPrefix}` :
            '';
        prefix += `:${this.cr.node.addr.substr(-8)}`;
        const tagKey = CUSTOM_RESOURCE_OWNER_TAG + prefix;
        // destinationKeyPrefix can be 104 characters before we hit
        // the tag key limit of 128
        // '/this/is/a/random/key/prefix/that/is/a/lot/of/characters/do/we/think/that/it/will/ever/be/this/long?????'
        // better to throw an error here than wait for CloudFormation to fail
        if (tagKey.length > 128) {
            throw new Error('The BucketDeployment construct requires that the "destinationKeyPrefix" be <=104 characters');
        }
        /*
         * This will add a tag to the deployment bucket in the format of
         * `aws-cdk:cr-owned:{keyPrefix}:{uniqueHash}`
         *
         * For example:
         * {
         *   Key: 'aws-cdk:cr-owned:deploy/here/:240D17B3',
         *   Value: 'true',
         * }
         *
         * This will allow for scenarios where there is a single S3 Bucket that has multiple
         * BucketDeployment resources deploying to it. Each bucket + keyPrefix can be "owned" by
         * 1 or more BucketDeployment resources. Since there are some scenarios where multiple BucketDeployment
         * resources can deploy to the same bucket and key prefix (e.g. using include/exclude) we
         * also append part of the id to make the key unique.
         *
         * As long as a bucket + keyPrefix is "owned" by a BucketDeployment resource, another CR
         * cannot delete data. There are a couple of scenarios where this comes into play.
         *
         * 1. If the LogicalResourceId of the CustomResource changes (e.g. the crUniqueId changes)
         * CloudFormation will first issue a 'Create' to create the new CustomResource and will
         * update the Tag on the bucket. CloudFormation will then issue a 'Delete' on the old CustomResource
         * and since the new CR "owns" the Bucket+keyPrefix it will not delete the contents of the bucket
         *
         * 2. If the BucketDeployment resource is deleted _and_ it is the only CR for that bucket+keyPrefix
         * then CloudFormation will first remove the tag from the bucket and then issue a "Delete" to the
         * CR. Since there are no tags indicating that this bucket+keyPrefix is "owned" then it will delete
         * the contents.
         *
         * 3. If the BucketDeployment resource is deleted _and_ it is *not* the only CR for that bucket:keyPrefix
         * then CloudFormation will first remove the tag from the bucket and then issue a "Delete" to the CR.
         * Since there are other CRs that also "own" that bucket+keyPrefix there will still be a tag on the bucket
         * and the contents will not be removed.
         *
         * 4. If the BucketDeployment resource _and_ the S3 Bucket are both removed, then CloudFormation will first
         * issue a "Delete" to the CR and since there is a tag on the bucket the contents will not be removed. If you
         * want the contents of the bucket to be removed on bucket deletion, then `autoDeleteObjects` property should
         * be set to true on the Bucket.
         */
        cdk.Tags.of(props.destinationBucket).add(tagKey, 'true');
    }
    /**
     * The bucket after the deployment
     *
     * If you want to reference the destination bucket in another construct and make sure the
     * bucket deployment has happened before the next operation is started, pass the other construct
     * a reference to `deployment.deployedBucket`.
     *
     * Doing this replaces calling `otherResource.node.addDependency(deployment)`.
     */
    get deployedBucket() {
        var _d;
        this.requestDestinationArn = true;
        this._deployedBucket = (_d = this._deployedBucket) !== null && _d !== void 0 ? _d : s3.Bucket.fromBucketArn(this, 'DestinationBucket', cdk.Token.asString(this.cr.getAtt('DestinationBucketArn')));
        return this._deployedBucket;
    }
    renderUniqueId(memoryLimit, vpc) {
        let uuid = '';
        // if user specify a custom memory limit, define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (memoryLimit) {
            if (cdk.Token.isUnresolved(memoryLimit)) {
                throw new Error('Can\'t use tokens when specifying "memoryLimit" since we use it to identify the singleton custom resource handler');
            }
            uuid += `-${memoryLimit.toString()}MiB`;
        }
        // if user specify to use VPC, define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        // A VPC is a must if EFS storage is used and that's why we are only using VPC in uuid.
        if (vpc) {
            uuid += `-${vpc.node.addr}`;
        }
        return uuid;
    }
    renderSingletonUuid(memoryLimit, vpc) {
        let uuid = '8693BB64-9689-44B6-9AAF-B0CC9EB8756C';
        uuid += this.renderUniqueId(memoryLimit, vpc);
        return uuid;
    }
    /**
     * Function to get/create a stack singleton instance of EFS FileSystem per vpc.
     *
     * @param scope Construct
     * @param fileSystemProps EFS FileSystemProps
     */
    getOrCreateEfsFileSystem(scope, fileSystemProps) {
        var _d;
        const stack = cdk.Stack.of(scope);
        const uuid = `BucketDeploymentEFS-VPC-${fileSystemProps.vpc.node.addr}`;
        return (_d = stack.node.tryFindChild(uuid)) !== null && _d !== void 0 ? _d : new efs.FileSystem(scope, uuid, fileSystemProps);
    }
}
exports.BucketDeployment = BucketDeployment;
_a = JSII_RTTI_SYMBOL_1;
BucketDeployment[_a] = { fqn: "@aws-cdk/aws-s3-deployment.BucketDeployment", version: "1.147.0" };
/**
 * Metadata
 */
function mapUserMetadata(metadata) {
    const mapKey = (key) => key.toLowerCase();
    return Object.keys(metadata).reduce((o, key) => ({ ...o, [mapKey(key)]: metadata[key] }), {});
}
function mapSystemMetadata(metadata) {
    const res = {};
    if (metadata.cacheControl) {
        res['cache-control'] = metadata.cacheControl.map(c => c.value).join(', ');
    }
    if (metadata.expires) {
        res.expires = metadata.expires.date.toUTCString();
    }
    if (metadata.contentDisposition) {
        res['content-disposition'] = metadata.contentDisposition;
    }
    if (metadata.contentEncoding) {
        res['content-encoding'] = metadata.contentEncoding;
    }
    if (metadata.contentLanguage) {
        res['content-language'] = metadata.contentLanguage;
    }
    if (metadata.contentType) {
        res['content-type'] = metadata.contentType;
    }
    if (metadata.serverSideEncryption) {
        res.sse = metadata.serverSideEncryption;
    }
    if (metadata.storageClass) {
        res['storage-class'] = metadata.storageClass;
    }
    if (metadata.websiteRedirectLocation) {
        res['website-redirect'] = metadata.websiteRedirectLocation;
    }
    if (metadata.serverSideEncryptionAwsKmsKeyId) {
        res['sse-kms-key-id'] = metadata.serverSideEncryptionAwsKmsKeyId;
    }
    if (metadata.serverSideEncryptionCustomerAlgorithm) {
        res['sse-c-copy-source'] = metadata.serverSideEncryptionCustomerAlgorithm;
    }
    if (metadata.accessControl) {
        res.acl = case_1.kebab(metadata.accessControl.toString());
    }
    return Object.keys(res).length === 0 ? undefined : res;
}
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
class CacheControl {
    constructor(
    /**
     * The raw cache control setting.
     */
    value) {
        this.value = value;
    }
    /**
     * Sets 'must-revalidate'.
     */
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    /**
     * Sets 'no-cache'.
     */
    static noCache() { return new CacheControl('no-cache'); }
    /**
     * Sets 'no-transform'.
     */
    static noTransform() { return new CacheControl('no-transform'); }
    /**
     * Sets 'public'.
     */
    static setPublic() { return new CacheControl('public'); }
    /**
     * Sets 'private'.
     */
    static setPrivate() { return new CacheControl('private'); }
    /**
     * Sets 'proxy-revalidate'.
     */
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    /**
     * Sets 'max-age=<duration-in-seconds>'.
     */
    static maxAge(t) { return new CacheControl(`max-age=${t.toSeconds()}`); }
    /**
     * Sets 's-maxage=<duration-in-seconds>'.
     */
    static sMaxAge(t) { return new CacheControl(`s-maxage=${t.toSeconds()}`); }
    /**
     * Constructs a custom cache control key from the literal value.
     */
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
_b = JSII_RTTI_SYMBOL_1;
CacheControl[_b] = { fqn: "@aws-cdk/aws-s3-deployment.CacheControl", version: "1.147.0" };
/**
 * Indicates whether server-side encryption is enabled for the object, and whether that encryption is
 * from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var ServerSideEncryption;
(function (ServerSideEncryption) {
    /**
     * 'AES256'
     */
    ServerSideEncryption["AES_256"] = "AES256";
    /**
     * 'aws:kms'
     */
    ServerSideEncryption["AWS_KMS"] = "aws:kms";
})(ServerSideEncryption = exports.ServerSideEncryption || (exports.ServerSideEncryption = {}));
/**
 * Storage class used for storing the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var StorageClass;
(function (StorageClass) {
    /**
     * 'STANDARD'
     */
    StorageClass["STANDARD"] = "STANDARD";
    /**
     * 'REDUCED_REDUNDANCY'
     */
    StorageClass["REDUCED_REDUNDANCY"] = "REDUCED_REDUNDANCY";
    /**
     * 'STANDARD_IA'
     */
    StorageClass["STANDARD_IA"] = "STANDARD_IA";
    /**
     * 'ONEZONE_IA'
     */
    StorageClass["ONEZONE_IA"] = "ONEZONE_IA";
    /**
     * 'INTELLIGENT_TIERING'
     */
    StorageClass["INTELLIGENT_TIERING"] = "INTELLIGENT_TIERING";
    /**
     * 'GLACIER'
     */
    StorageClass["GLACIER"] = "GLACIER";
    /**
     * 'DEEP_ARCHIVE'
     */
    StorageClass["DEEP_ARCHIVE"] = "DEEP_ARCHIVE";
})(StorageClass = exports.StorageClass || (exports.StorageClass = {}));
/**
 * Used for HTTP expires header, which influences downstream caches. Does NOT influence deletion of the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 *
 * @deprecated use core.Expiration
 */
class Expires {
    constructor(
    /**
     * The raw expiration date expression.
     */
    value) {
        this.value = value;
    }
    /**
     * Expire at the specified date
     * @param d date to expire at
     */
    static atDate(d) { jsiiDeprecationWarnings.print("@aws-cdk/aws-s3-deployment.Expires#atDate", "use core.Expiration"); return new Expires(d.toUTCString()); }
    /**
     * Expire at the specified timestamp
     * @param t timestamp in unix milliseconds
     */
    static atTimestamp(t) { jsiiDeprecationWarnings.print("@aws-cdk/aws-s3-deployment.Expires#atTimestamp", "use core.Expiration"); return Expires.atDate(new Date(t)); }
    /**
     * Expire once the specified duration has passed since deployment time
     * @param t the duration to wait before expiring
     */
    static after(t) { jsiiDeprecationWarnings.print("@aws-cdk/aws-s3-deployment.Expires#after", "use core.Expiration"); return Expires.atDate(new Date(Date.now() + t.toMilliseconds())); }
    /**
     * Create an expiration date from a raw date string.
     */
    static fromString(s) { jsiiDeprecationWarnings.print("@aws-cdk/aws-s3-deployment.Expires#fromString", "use core.Expiration"); return new Expires(s); }
}
exports.Expires = Expires;
_c = JSII_RTTI_SYMBOL_1;
Expires[_c] = { fqn: "@aws-cdk/aws-s3-deployment.Expires", version: "1.147.0" };
//# sourceMappingURL=data:application/json;base64,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