"use strict";
const assert_1 = require("@aws-cdk/assert");
const targets = require("@aws-cdk/aws-events-targets");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const config = require("../lib");
module.exports = {
    'create a managed rule'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new config.ManagedRule(stack, 'Rule', {
            description: 'really cool rule',
            identifier: 'AWS_SUPER_COOL',
            inputParameters: {
                key: 'value',
            },
            maximumExecutionFrequency: config.MaximumExecutionFrequency.THREE_HOURS,
            configRuleName: 'cool rule',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Config::ConfigRule', {
            Source: {
                Owner: 'AWS',
                SourceIdentifier: 'AWS_SUPER_COOL',
            },
            ConfigRuleName: 'cool rule',
            Description: 'really cool rule',
            InputParameters: {
                key: 'value',
            },
            MaximumExecutionFrequency: 'Three_Hours',
        }));
        test.done();
    },
    'create a custom rule'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.AssetCode.fromInline('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        new config.CustomRule(stack, 'Rule', {
            configurationChanges: true,
            description: 'really cool rule',
            inputParameters: {
                key: 'value',
            },
            lambdaFunction: fn,
            maximumExecutionFrequency: config.MaximumExecutionFrequency.SIX_HOURS,
            configRuleName: 'cool rule',
            periodic: true,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Config::ConfigRule', {
            Properties: {
                Source: {
                    Owner: 'CUSTOM_LAMBDA',
                    SourceDetails: [
                        {
                            EventSource: 'aws.config',
                            MessageType: 'ConfigurationItemChangeNotification',
                        },
                        {
                            EventSource: 'aws.config',
                            MessageType: 'OversizedConfigurationItemChangeNotification',
                        },
                        {
                            EventSource: 'aws.config',
                            MaximumExecutionFrequency: 'Six_Hours',
                            MessageType: 'ScheduledNotification',
                        },
                    ],
                    SourceIdentifier: {
                        'Fn::GetAtt': [
                            'Function76856677',
                            'Arn',
                        ],
                    },
                },
                ConfigRuleName: 'cool rule',
                Description: 'really cool rule',
                InputParameters: {
                    key: 'value',
                },
                MaximumExecutionFrequency: 'Six_Hours',
            },
            DependsOn: [
                'FunctionPermissionEC8FE997',
                'Function76856677',
                'FunctionServiceRole675BB04A',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Principal: 'config.amazonaws.com',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            ManagedPolicyArns: [
                {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole',
                        ],
                    ],
                },
                {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':iam::aws:policy/service-role/AWSConfigRulesExecutionRole',
                        ],
                    ],
                },
            ],
        }));
        test.done();
    },
    'scope to resource'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const rule = new config.ManagedRule(stack, 'Rule', {
            identifier: 'AWS_SUPER_COOL',
        });
        // WHEN
        rule.scopeToResource('AWS::EC2::Instance', 'i-1234');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Config::ConfigRule', {
            Scope: {
                ComplianceResourceId: 'i-1234',
                ComplianceResourceTypes: [
                    'AWS::EC2::Instance',
                ],
            },
        }));
        test.done();
    },
    'scope to resources'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const rule = new config.ManagedRule(stack, 'Rule', {
            identifier: 'AWS_SUPER_COOL',
        });
        // WHEN
        rule.scopeToResources('AWS::S3::Bucket', 'AWS::CloudFormation::Stack');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Config::ConfigRule', {
            Scope: {
                ComplianceResourceTypes: [
                    'AWS::S3::Bucket',
                    'AWS::CloudFormation::Stack',
                ],
            },
        }));
        test.done();
    },
    'scope to tag'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const rule = new config.ManagedRule(stack, 'Rule', {
            identifier: 'RULE',
        });
        // WHEN
        rule.scopeToTag('key', 'value');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Config::ConfigRule', {
            Scope: {
                TagKey: 'key',
                TagValue: 'value',
            },
        }));
        test.done();
    },
    'allows scoping a custom rule without configurationChanges enabled'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.AssetCode.fromInline('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        const rule = new config.CustomRule(stack, 'Rule', {
            lambdaFunction: fn,
            periodic: true,
        });
        // THEN
        test.doesNotThrow(() => rule.scopeToResource('resource'));
        test.done();
    },
    'throws when both configurationChanges and periodic are falsy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.AssetCode.fromInline('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // THEN
        test.throws(() => new config.CustomRule(stack, 'Rule', {
            lambdaFunction: fn,
        }), /`configurationChanges`.*`periodic`/);
        test.done();
    },
    'on compliance change event'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const rule = new config.ManagedRule(stack, 'Rule', {
            identifier: 'RULE',
        });
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.fromInline('dummy'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        rule.onComplianceChange('ComplianceChange', {
            target: new targets.LambdaFunction(fn),
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            EventPattern: {
                'source': [
                    'aws.config',
                ],
                'detail': {
                    configRuleName: [
                        {
                            Ref: 'Rule4C995B7F',
                        },
                    ],
                },
                'detail-type': [
                    'Config Rules Compliance Change',
                ],
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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