# EdisonScientific Platform API Documentation

Documentation and tutorials for edison-client,
a client for interacting with endpoints of the EdisonScientific platform.

<!--TOC-->

---

**Table of Contents**

- [Installation](#installation)
- [Quickstart](#quickstart)
- [Functionalities](#functionalities)
- [Authentication](#authentication)
- [Simple task running](#simple-task-running)
- [Task Continuation](#task-continuation)
- [Asynchronous tasks](#asynchronous-tasks)

---

<!--TOC-->

## Installation

```bash
uv pip install edison-client
```

## Quickstart

```python
from edison_client import EdisonClient, JobNames

# Set your API key to EDISON_API_KEY environment variable,
# or pass it as a string literal to the api_key parameter
client = EdisonClient()

task_data = {
    "name": JobNames.LITERATURE,
    "query": (
        "Which neglected diseases had a treatment developed"
        " by artificial intelligence?"
    ),
}
(task_response,) = client.run_tasks_until_done(task_data)
```

A quickstart example can be found in the [client_notebook.ipynb][1] file,
where we show how to submit and retrieve a job task,
pass runtime configuration to the agent,
and ask follow-up questions to the previous job.

[1]: https://edisonscientific.gitbook.io/edison-cookbook/edison-client/docs/client_notebook

## Functionalities

EdisonScientific client implements a RestClient (called `EdisonClient`) with the following functionalities:

- [Simple task running](#simple-task-running):
  `run_tasks_until_done(TaskRequest)` or `await arun_tasks_until_done(TaskRequest)`
- [Asynchronous tasks](#asynchronous-tasks):
  `get_task(task_id)` or `aget_task(task_id)`
  and `create_task(TaskRequest)` or `acreate_task(TaskRequest)`

To create a `EdisonClient`,
you need to pass an EdisonScientific platform api key (see [Authentication](#authentication)):

```python
from edison_client import EdisonClient

# Set your API key to EDISON_API_KEY environment variable
client = EdisonClient()
# Or pass it as a string literal to the api_key parameter
client = EdisonClient(api_key="your_api_key")
```

## Authentication

In order to use the `EdisonClient`, you need to authenticate yourself.
Authentication is done by providing an API key,
which can be obtained directly from your [profile page in the EdisonScientific platform][2].

[2]: https://platform.edisonscientific.com/profile

## Simple task running

In the EdisonScientific platform,
we define the deployed combination of an agent and an environment as a `job`.
To invoke a job, we need to submit a `task` (also called a `query`) to it.
`EdisonClient` can be used to submit tasks/queries to available jobs in the EdisonScientific platform.
Using a `EdisonClient` instance,
you can submit tasks to the platform by calling the `create_task` method,
which receives a `TaskRequest` (or a dictionary with `kwargs`) and returns the task id.
Aiming to make the submission of tasks as simple as possible,
we have created a `JobNames` `enum` that contains the available task types.

The available supported jobs are:

| Alias              | Job Name                        | Task type        | Description                                                                                                                                              |
| ------------------ | ------------------------------- | ---------------- | -------------------------------------------------------------------------------------------------------------------------------------------------------- |
| `JobNames.LITERATURE`    | `job-futurehouse-paperqa2`      | Literature Search      | Ask a question of scientific data sources, and receive a high-accuracy, cited response. Built with [PaperQA2](https://github.com/Future-House/paper-qa). |
| `JobNames.ANALYSIS`  | `job-futurehouse-data-analysis-crow-high` | Data Analysis      | Turn biological datasets into detailed analyses answering your research questions.                                                       |
| `JobNames.PRECEDENT`     | `job-futurehouse-paperqa3-precedent`     | Precedent Search | Formerly known as HasAnyone, query if anyone has ever done something in science.                                                                         |
| `JobNames.MOLECULES` | `job-futurehouse-phoenix`       | Chemistry Tasks  | A new iteration of ChemCrow, Phoenix uses cheminformatics tools to do chemistry. Good for planning synthesis and designing new molecules.                |
| `JobNames.DUMMY`   | `job-futurehouse-dummy`         | Dummy Task       | This is a dummy task. Mainly for testing purposes.                                                                                                       |

Using `JobNames`, the task submission looks like this:

```python
from edison_client import EdisonClient, JobNames

client = EdisonClient()

task_data = {
    "name": JobNames.PRECEDENT,
    "query": "Has anyone tested therapeutic exerkines in humans or NHPs?",
}
(task_response,) = client.run_tasks_until_done(task_data)
print(task_response.answer)
```

Or if running async code:

```python
import asyncio
from edison_client import EdisonClient, JobNames


async def main():
    client = EdisonClient(
        api_key="your_api_key",
    )

    task_data = {
        "name": JobNames.PRECEDENT,
        "query": "Has anyone tested therapeutic exerkines in humans or NHPs?",
    }
    (task_response,) = await client.arun_tasks_until_done(task_data)
    print(task_response.answer)
    return task_id


# For Python 3.7+
if __name__ == "__main__":
    task_id = asyncio.run(main())
```

Note that in either the sync or the async code,
collections of tasks can be given to the client to run them in a batch:

```python
import asyncio
from edison_client import EdisonClient, JobNames


async def main():
    client = EdisonClient()

    task_data = [
        {
            "name": JobNames.PRECEDENT,
            "query": "Has anyone tested therapeutic exerkines in humans or NHPs?",
        },
        {
            "name": JobNames.LITERATURE,
            "query": (
                "Are there any clinically validated"
                " therapeutic exerkines for humans?"
            ),
        },
    ]
    task_responses = await client.arun_tasks_until_done(task_data)
    print(task_responses[0].answer)
    print(task_responses[1].answer)
    return task_id


# For Python 3.7+
if __name__ == "__main__":
    task_id = asyncio.run(main())
```

`TaskRequest` can also be used to submit jobs and it has the following fields:

| Field          | Type          | Description                                                                                                         |
| -------------- | ------------- | ------------------------------------------------------------------------------------------------------------------- |
| id             | UUID          | Optional job identifier. A UUID will be generated if not provided                                                   |
| name           | str           | Name of the job to execute eg. `job-futurehouse-paperqa2`, or using the `JobNames` for convenience: `JobNames.LITERATURE` |
| query          | str           | Query or task to be executed by the job                                                                             |
| runtime_config | RuntimeConfig | Optional runtime parameters for the job                                                                             |

`runtime_config` can receive a `AgentConfig` object with the desired kwargs.
Check the available `AgentConfig` fields in the [LDP documentation][3].
Besides the `AgentConfig` object,
we can also pass `timeout` and `max_steps` to limit the execution time and the number of steps the agent can take.

[3]: https://github.com/Future-House/ldp/blob/main/src/ldp/agent/agent.py#L87

```python
from edison_client import EdisonClient, JobNames
from edison_client.models.app import TaskRequest

client = EdisonClient()

(task_response,) = client.run_tasks_until_done(
    TaskRequest(
        name=JobNames.PRECEDENT,
        query="Has anyone tested therapeutic exerkines in humans or NHPs?",
    )
)
print(task_response.answer)
```

A `TaskResponse` will be returned from using our agents.
For Owl, Crow, and Falcon, we default to a subclass, `PQATaskResponse` which has some key attributes:

| Field                 | Type | Description                                                                     |
| --------------------- | ---- | ------------------------------------------------------------------------------- |
| answer                | str  | Answer to your query.                                                           |
| formatted_answer      | str  | Specially formatted answer with references.                                     |
| has_successful_answer | bool | Flag for whether the agent was able to find a good answer to your query or not. |

If using the `verbose` setting, much more data can be pulled down from your `TaskResponse`,
which will exist across all agents (not just Owl, Crow, and Falcon).

```python
from edison_client import EdisonClient, JobNames
from edison_client.models.app import TaskRequest

client = EdisonClient(
    api_key="your_api_key",
)

(task_response,) = client.run_tasks_until_done(
    TaskRequest(
        name=JobNames.PRECEDENT,
        query="Has anyone tested therapeutic exerkines in humans or NHPs?",
    ),
    verbose=True,
)
print(task_response.environment_frame)
```

In that case, a `TaskResponseVerbose` will have the following fields:

| Field             | Type | Description                                                                                                            |
| ----------------- | ---- | ---------------------------------------------------------------------------------------------------------------------- |
| agent_state       | dict | Large object with all agent states during the progress of your task.                                                   |
| environment_frame | dict | Large nested object with all environment data, for PQA environments it includes contexts, paper metadata, and answers. |
| metadata          | dict | Extra metadata about your query.                                                                                       |

## Task Continuation

Once a task is submitted and the answer is returned,
EdisonScientific platform allow you to ask follow-up questions to the previous task.
It is also possible through the platform API.
To accomplish that,
we can use the `runtime_config` we discussed in the [Simple task running](#simple-task-running) section.

```python
from edison_client import EdisonClient, JobNames

client = EdisonClient(
    api_key="your_api_key",
)

task_data = {
    "name": JobNames.LITERATURE,
    "query": "How many species of birds are there?",
}
task_id = client.create_task(task_data)

continued_task_data = {
    "name": JobNames.LITERATURE,
    "query": (
        "From the previous answer,"
        " specifically, how many species of crows are there?"
    ),
    "runtime_config": {"continued_task_id": task_id},
}
(task_response,) = client.run_tasks_until_done(continued_task_data)
```

## Asynchronous tasks

Sometimes you may want to submit many jobs, while querying results at a later time.
In this way you can do other things while waiting for a response.
The platform API supports this as well rather than waiting for a result.

```python
from edison_client import EdisonClient, JobNames

client = EdisonClient()

task_data = {
    "name": JobNames.LITERATURE,
    "query": "How many species of birds are there?",
}
task_id = client.create_task(task_data)

# move on to do other things

task_status = client.get_task(task_id)
```

`task_status` contains information about the task.
For instance, its `status`, `task`, `environment_name` and `agent_name`, and other fields specific to the job.
You can continually query the status until it's `success` before moving on.
