# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['json_schema_for_humans', 'json_schema_for_humans.schema']

package_data = \
{'': ['*'],
 'json_schema_for_humans': ['templates/flat/*',
                            'templates/js/*',
                            'templates/md/*',
                            'templates/md_nested/*']}

install_requires = \
['Jinja2>3',
 'MarkupSafe>=2.0,<3.0',
 'PyYAML>=5.4.1,<7',
 'Pygments>=2.10.0,<3.0.0',
 'click>=8.0.1,<9.0.0',
 'dataclasses-json>=0.5.6,<0.6.0',
 'htmlmin>=0.1.12,<0.2.0',
 'markdown2>=2.4.1,<3.0.0',
 'pytz>=2021.1,<2022.0',
 'requests>=2.26.0,<3.0.0']

entry_points = \
{'console_scripts': ['generate-schema-doc = json_schema_for_humans.cli:main']}

setup_kwargs = {
    'name': 'json-schema-for-humans',
    'version': '0.43',
    'description': 'Generate static HTML documentation from JSON schemas',
    'long_description': '[![PyPI version](https://badge.fury.io/py/json-schema-for-humans.svg)](https://badge.fury.io/py/json-schema-for-humans)\n\n# JSON Schema for Humans\n\nQuickly generate a beautiful static HTML or Markdown page documenting a JSON schema\n\n[Documentation (with visual examples)](https://coveooss.github.io/json-schema-for-humans)\n\n## Features\n\n- Support for JSON Schema Draft-07\n- Since the result is static, it is easier to host and faster to load\n- HTML and Markdown output support\n- Different templates to choose from. More details: [HTML version](https://coveooss.github.io/json-schema-for-humans/examples/examples_js_default/Configuration.html#template_name) - [Markdown version](https://github.com/coveooss/json-schema-for-humans/blob/main/docs/examples/examples_md_default/Configuration.md#template_name)\n- Anchor links, allow to send a user to a specific section of the documentation\n- Support for references (even circular!)\n\n## Installation\n```\npip install json-schema-for-humans\n```\n\n## Usage\n\nOptions for generation of the doc are documented using the library itself: [HTML version](https://coveooss.github.io/json-schema-for-humans/examples/examples_js_default/Configuration.html) - [Markdown version](https://github.com/coveooss/json-schema-for-humans/blob/main/docs/examples/examples_md_default/Configuration.md)\n\nThey can be supplied in various ways:\n- Using a JSON or YAML configuration file with the CLI option `--config-file`\n- Using the CLI option `--config`\n- Using the `ConfigurationOption` object from code\n\nMore details are available in the appropriate sections below.\n\n### From CLI\n\n```\ngenerate-schema-doc [OPTIONS] SCHEMA_FILES_OR_DIR [RESULT_FILE_OR_DIR]\n```\n\n`SCHEMA_FILES_OR_DIR` can be:\n- a path to a single schema file;\n- a path to a directory, in this case all files with extensions json, yaml, or yml will be used; \n- a [glob pattern](https://docs.python.org/3/library/pathlib.html#pathlib.Path.glob) (starts from the current working directory); or\n- a comma-separated list of the above\n\nAll schemas provided must be a valid JSON Schema (in JSON or YAML format)\n\nExamples:\n- `my_schema.json`\n- `my_folder`\n- `my_folder/my_schema.yaml,another_schema.json`\n- `**/*.yaml.*`\n\nThe default value for `RESULT_FILE_OR_DIR` depends on the context:\n- the current working directory if more than one schema as been provided as input\n- `schema_doc.html` if rendering a single schema as HTML\n- `schema_doc.md` if rendering a single schema as Markdown\n\nIn a case where more than one schema is provided as input, `RESULT_FILE_OR_DIR` must be a directory. The output documentation will have the same name as the input schema, but with a different extension (`html` or `md`).\n\n#### CLI options\n\n#### --config\nSupply generation config parameters. The parameters are documented in the JSON schema `config_schema.json` at the root of the repo or see the generated doc: [HTML version](https://coveooss.github.io/json-schema-for-humans/examples/examples_js_default/Configuration.html) - [Markdown version](https://github.com/coveooss/json-schema-for-humans/blob/main/docs/examples/examples_md_default/Configuration.md).\n\nEach parameter is in the format `--config parameter_name=parameter_value`. Example: `--config expand_buttons=true`. The parameter value must be valid JSON.\n\nFor flags, you can also omit the value for `true` or prefix the parameter name with `no_` for `false`. Example: `--config expand_buttons` or `--config no_expand_buttons`.\n\n#### --config-file\nPath to a JSON or YAML configuration file respecting the schema `config_schema.json`.\n\nExample: `--config-file jsfh-conf.yaml` where `jsfh-conf.yaml` is in the current directory and contains the following:\n\n```yaml\ndescription_is_markdown: false\nexpand_buttons: true\ncopy_js: false\n```\n\n### From code\n\nThe following methods are available to import from `json_schema_for_humans.generate`\n\nMethod Name | Schema input | Output | CSS and JS copied?\n--- | --- | --- | ---\ngenerate_from_schema | `schema_file` as str or `pathlib.Path` | Rendered doc as a str | No\ngenerate_from_filename | `schema_file_name` as str or `pathlib.Path` | Rendered doc written to the file at path `result_file_name` | Yes\ngenerate_from_file_object | `schema_file` as an open file object (read mode) | Rendered doc written to the file at `result_file`, which must be an open file object (in write mode) | Yes\n\nNotes:\n- When using file objects, it is assumed that files are opened with encoding "utf-8"\n- CSS and JS files are copied to the current working directory with names "schema_doc.css" and "schema_doc.min.js" respectively, if necessary\n- Other parameters of these methods are analogous to the CLI parameters documented above.\n\n#### The GenerationConfiguration object\nTo reduce the number of parameters to pass from function to function in the code, there is a `GenerationConfiguration` object that should be used for providing options.\n\nExample:\n\n```python\nfrom json_schema_for_humans.generate import generate_from_filename\nfrom json_schema_for_humans.generation_configuration import GenerationConfiguration\n\nconfig = GenerationConfiguration(copy_css=False, expand_buttons=True)\n\ngenerate_from_filename("my_schema.json", "schema_doc.html", config=config)\n\n# Your doc is now in a file named "schema_doc.html". Next to it, "schema_doc.min.js" was copied, but not "schema_doc.css"\n# Your doc will contain a "Expand all" and a "Collapse all" button at the top\n\n```\n\n#### Pre-load schemas\n`generate_from_schema` has a `loaded_schemas` parameter that can be used to pre-load schemas. This must be a dict with the key being the real path of the schema file and the value being the result of loading the schema (with `json.load` or `yaml.safe_load`, for example).\n\nThis should not be necessary in normal scenarios.\n\n## What\'s supported\n\nSee the excellent [Understanding JSON Schema](https://json-schema.org/understanding-json-schema/index.html) to understand what are those checks\n\nThe following are supported:\n- Types\n- Regular expressions\n- String length\n- String format\n- Numeric types multiples and range\n- Constant and enumerated values\n- Required properties\n- Pattern properties\n- Default values\n- Array `minItems`, `maxItems`, `uniqueItems`, `items`, `prefixItems`, and `contains`\n- Combining schema with `oneOf`, `allOf`, `anyOf`, and `not`\n- Examples\n- Conditional subschemas\n\nThese are **not** supported at the moment (PRs welcome!):\n- Property names and size\n- Property dependencies\n- Media\n\n## References\n\nReferences are supported:\n\n- To another part of the schema, e.g. `{ $ref: "#/definitions/something" }`\n- To a local file, `{"$ref": "references.json"}`, `{"$ref": "references.json#/definitions/something"}`\n- To a URL, `{"$ref": "http://example.com/schema.json"}`, `{"$ref": "http://example.com/schema.json#/definitions/something"}`\n\n\nYou _can_ have a `description` next to a `$ref`, it will be displayed in priority to the description from the referenced element.\n\nIf you have several attributes using the same definition, the definition will only be rendered once.\nAll other usages of the same definition will be replaced with an anchor link to the first render of the definition.\nThis can be turned off using `--config no_link_to_reused_ref`. See `With references` in the examples.\n\n## Templates\n\nTemplates control the style of the generated documentation.\n\n### js\n\nThis is the default template. It uses Bootstrap along with minimal Javascript to allow for the following:\n\n- Properties are in expandable dynamic sections. You can include a button to expand or collapse all. (See doc: [HTML version](https://coveooss.github.io/json-schema-for-humans/examples/examples_js_default/Configuration.html#expand_buttons) - [Markdown version](https://github.com/coveooss/json-schema-for-humans/blob/main/docs/examples/examples_md_default/Configuration.md#expand_buttons))\n- Conditional subschemas (`anyOf`, `oneOf`, `allOf`) are in tabbed sections\n- Anchor links will scroll to, expand, and animate the target section \n- Long descriptions are collapsed by default\n\nWhen using this template, you need to include the Javascript file (`schema_doc.min.js`) that is automatically copied next to the output HTML file (`schema_doc.html` by default).\n\n### flat\n\n*Note*: This template is a work in progress\n\nIt is sometimes not possible or desirable to include custom Javascript in documentation. This template addresses this issue by removing interactive elements in favor of simpler HTML.\n\nAt the moment, this means the whole documentation is generated without any collapsible sections, which may make it hard to understand the schema structure. Contributions are welcomed to improve it!\n\n### MD (Markdown)\n\n*Note*: This template is a work in progress\n\nThis template allows users to publish the generated documentation without hosting an HTTP server.\n\nOn GitHub, this format is rendered directly when browsing code.\n\nA table of content is provided at the beginning of the file for easy navigation.\n\nYou can display some important information as badge using an option.\nSee doc: [HTML version](https://coveooss.github.io/json-schema-for-humans/examples/examples_js_default/Configuration.html#template_md_options_badge_as_image) - [Markdown version](https://github.com/coveooss/json-schema-for-humans/blob/main/docs/examples/examples_md_default/Configuration.md#template_md_options_badge_as_image)\n\nContributions are welcomed to improve it!\n\n\n## Contributing\n[See CONTRIBUTING.md](CONTRIBUTING.md)',
    'author': 'Denis Blanchette',
    'author_email': 'tools@coveo.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/coveooss/json-schema-for-humans',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
