from __future__ import annotations

from abc import ABC
from typing import Any, Generator, Generic, Iterable, Mapping, TypeVar

from multidict import MultiDict as BaseMultiDict
from multidict import MultiDictProxy, MultiMapping

from starlite.datastructures.upload_file import UploadFile

T = TypeVar("T")


class MultiMixin(Generic[T], MultiMapping[T], ABC):
    """Mixin providing common methods for multi dicts, used by.

    :class:`ImmutableMultiDict <starlite.datastructures.multi_dicts.ImmutableMultiDict>` and
    :class:`MultiDict <starlite.datastructures.multi_dicts.MultiDict>`
    """

    def dict(self) -> dict[str, list[Any]]:
        """Return the multi-dict as a dict of lists.

        Returns:
            A dict of lists
        """
        return {k: self.getall(k) for k in set(self.keys())}

    def multi_items(self) -> Generator[tuple[str, T], None, None]:
        """Get all keys and values, including duplicates.

        Returns:
            A list of tuples containing key-value pairs
        """
        for key in set(self):
            for value in self.getall(key):
                yield key, value


class MultiDict(BaseMultiDict[T], MultiMixin[T], Generic[T]):
    """MultiDict, using :class:`MultiDict <multidict.MultiDictProxy>`."""

    def __init__(self, args: MultiMapping | Mapping[str, T] | Iterable[tuple[str, T]] | None = None) -> None:
        """Initialize ``MultiDict`` from a.

        :class:`MultiMapping <multidict.MultiMapping>`, ``Mapping`` or an iterable of
        tuples.

        Args:
            args: Mapping-like structure to create the ``MultiDict`` from
        """
        super().__init__(args or {})

    def immutable(self) -> ImmutableMultiDict[T]:
        """Create an.

        :class:`ImmutableMultiDict <starlite.datastructures.multi_dicts.ImmutableMultiDict>` view.

        Returns:
            An immutable multi dict
        """
        return ImmutableMultiDict[T](self)


class ImmutableMultiDict(MultiDictProxy[T], MultiMixin[T], Generic[T]):
    """Immutable MultiDict, using.

    :class:`MultiDictProxy <multidict.MultiDictProxy>`.
    """

    def __init__(self, args: MultiMapping | Mapping[str, Any] | Iterable[tuple[str, Any]] | None = None) -> None:
        """Initialize ``ImmutableMultiDict`` from a.

        :class:`MultiMapping <multidict.MultiMapping>`, ``Mapping`` or an iterable of
        tuples.

        Args:
            args: Mapping-like structure to create the ``ImmutableMultiDict`` from
        """
        super().__init__(BaseMultiDict(args or {}))

    def mutable_copy(self) -> MultiDict[T]:
        """Create a mutable copy as a.

        :class:`MultiDict <starlite.datastructures.multi_dicts.MultiDict>`

        Returns:
            A mutable multi dict
        """
        return MultiDict(list(self.multi_items()))


class FormMultiDict(ImmutableMultiDict[Any]):
    """MultiDict for form data."""

    async def close(self) -> None:
        """Close all files in the multi-dict.

        Returns:
            None
        """
        for _, value in self.multi_items():
            if isinstance(value, UploadFile):
                await value.close()
