"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Pom = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const deps_1 = require("../deps");
const semver_1 = require("../util/semver");
const xmlfile_1 = require("../xmlfile");
const POM_XML_ATTRS = {
    '@xsi:schemaLocation': 'http://maven.apache.org/POM/4.0.0 http://maven.apache.org/xsd/maven-4.0.0.xsd',
    '@xmlns': 'http://maven.apache.org/POM/4.0.0',
    '@xmlns:xsi': 'http://www.w3.org/2001/XMLSchema-instance',
};
/**
 * (experimental) A Project Object Model or POM is the fundamental unit of work in Maven.
 *
 * It is
 * an XML file that contains information about the project and configuration
 * details used by Maven to build the project.
 *
 * @experimental
 */
class Pom extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, options) {
        var _b;
        super(project);
        this.properties = {};
        this.fileName = 'pom.xml';
        this.groupId = options.groupId;
        this.artifactId = options.artifactId;
        this.version = options.version;
        this.packaging = (_b = options.packaging) !== null && _b !== void 0 ? _b : 'jar';
        this.name = project.name;
        this.description = options.description;
        this.url = options.url;
        new xmlfile_1.XmlFile(project, this.fileName, { obj: () => this.synthPom(), marker: true });
    }
    /**
     * (experimental) Adds a key/value property to the pom.
     *
     * @param key the key.
     * @param value the value.
     * @experimental
     */
    addProperty(key, value) {
        this.properties[key] = value;
    }
    /**
     * (experimental) Adds a runtime dependency.
     *
     * @param spec Format `<groupId>/<artifactId>@<semver>`.
     * @experimental
     */
    addDependency(spec) {
        this.project.deps.addDependency(spec, deps_1.DependencyType.RUNTIME);
    }
    /**
     * (experimental) Adds a test dependency.
     *
     * @param spec Format `<groupId>/<artifactId>@<semver>`.
     * @experimental
     */
    addTestDependency(spec) {
        this.project.deps.addDependency(spec, deps_1.DependencyType.TEST);
    }
    /**
     * (experimental) Adds a build plugin to the pom.
     *
     * The plug in is also added as a BUILD dep to the project.
     *
     * @param spec dependency spec (`group/artifact@version`).
     * @param options plugin options.
     * @experimental
     */
    addPlugin(spec, options = {}) {
        var _b;
        for (const dep of (_b = options.dependencies) !== null && _b !== void 0 ? _b : []) {
            this.project.deps.addDependency(dep, deps_1.DependencyType.BUILD);
        }
        return this.project.deps.addDependency(spec, deps_1.DependencyType.BUILD, options);
    }
    synthPom() {
        return _resolve_1.resolve({
            project: {
                ...POM_XML_ATTRS,
                modelVersion: '4.0.0',
                groupId: this.groupId,
                artifactId: this.artifactId,
                version: this.version,
                packaging: this.packaging,
                name: this.name,
                description: this.description,
                url: this.url,
                properties: this.properties,
                ...this.synthDependencies(),
            },
        }, { omitEmpty: true });
    }
    synthDependencies() {
        const deps = this.project.deps.all;
        if (deps.length === 0) {
            return;
        }
        const dependencies = [];
        const plugins = [];
        for (const dep of deps) {
            switch (dep.type) {
                case deps_1.DependencyType.PEER:
                case deps_1.DependencyType.RUNTIME:
                    dependencies.push(mavenCoords(dep));
                    break;
                case deps_1.DependencyType.TEST:
                    dependencies.push({
                        ...mavenCoords(dep),
                        scope: 'test',
                    });
                    break;
                // build maps to plugins
                case deps_1.DependencyType.BUILD:
                    plugins.push({
                        ...mavenCoords(dep),
                        ...pluginConfig(dep.metadata),
                    });
                    break;
                default:
                    throw new Error(`unsupported dependency type: ${dep.type}`);
            }
        }
        return {
            build: { plugins: { plugin: plugins } },
            dependencies: { dependency: dependencies },
        };
    }
}
exports.Pom = Pom;
_a = JSII_RTTI_SYMBOL_1;
Pom[_a] = { fqn: "projen.java.Pom", version: "0.17.25" };
/**
 * Parses maven groupId and artifactId from a dependency name.
 *
 *     name    <=> <groupId>/<artifactId>
 *     version <=> <version>
 */
function mavenCoords(dep) {
    const name = dep.name;
    const parts = name.split('/');
    if (parts.length !== 2) {
        throw new Error(`invalid maven coordinates in dependency named "${name}". format is "<groupId>/<artifactId>". For example "org.junit.jupiter/junit-jupiter-engine"`);
    }
    return {
        groupId: parts[0],
        artifactId: parts[1],
        version: dep.version ? semver_1.toMavenVersionRange(dep.version) : undefined,
    };
}
function pluginConfig(options = {}) {
    var _b, _c;
    return {
        configuration: options.configuration,
        dependencies: (options.dependencies && options.dependencies.length > 0)
            ? { dependency: (_b = options.dependencies) === null || _b === void 0 ? void 0 : _b.map(d => mavenCoords(deps_1.Dependencies.parseDependency(d))) }
            : undefined,
        executions: (_c = options.executions) === null || _c === void 0 ? void 0 : _c.map(e => ({
            execution: {
                id: e.id,
                goals: e.goals.map(goal => ({ goal })),
            },
        })),
    };
}
//# sourceMappingURL=data:application/json;base64,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