"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseFargateService = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const events = require("@aws-cdk/aws-events");
const event_targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const common_functions_1 = require("./common/common-functions");
/**
 * @stability stable
 */
class BaseFargateService extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f;
        super(scope, id);
        /**
         * @stability stable
         */
        this.zoneName = '';
        /**
         * @stability stable
         */
        this.hasExternalLoadBalancer = false;
        /**
         * @stability stable
         */
        this.hasInternalLoadBalancer = false;
        /**
         * @stability stable
         */
        this.vpcSubnets = { subnetType: ec2.SubnetType.PRIVATE };
        this.hasSpotCapacity = false;
        /**
         * determine if vpcSubnets are all public ones
         */
        this.isPublicSubnets = false;
        this.enableLoadBalancerAlias = ((_b = props.route53Ops) === null || _b === void 0 ? void 0 : _b.enableLoadBalancerAlias) != false;
        if (props.vpc && props.cluster) {
            throw new Error('Cannot specify vpc and cluster at the same time');
        }
        if (props.clusterProps && props.cluster) {
            throw new Error('Cannot specify clusterProps and cluster at the same time');
        }
        this.vpc = props.cluster ? props.cluster.vpc : (_c = props.vpc) !== null && _c !== void 0 ? _c : common_functions_1.getOrCreateVpc(this),
            this.service = [];
        if (props.vpcSubnets) {
            this.vpcSubnets = props.vpcSubnets;
            this.validateSubnets(this.vpc, this.vpcSubnets);
        }
        // determine whether we need the external LB
        props.tasks.forEach(t => {
            // determine the accessibility
            if (t.external) {
                this.hasExternalLoadBalancer = true;
            }
            if (t.internal) {
                this.hasInternalLoadBalancer = true;
            }
        });
        const cluster = (_d = props.cluster) !== null && _d !== void 0 ? _d : new ecs.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            enableFargateCapacityProviders: true,
            ...props.clusterProps,
        });
        const spotOnlyStrategy = [
            {
                capacityProvider: 'FARGATE_SPOT',
                base: 0,
                weight: 1,
            },
            {
                capacityProvider: 'FARGATE',
                base: 0,
                weight: 0,
            },
        ];
        if (props.spot == true)
            this.hasSpotCapacity = true;
        props.tasks.forEach(t => {
            var _b, _c, _d, _e;
            const defaultContainerName = (_b = t.task.defaultContainer) === null || _b === void 0 ? void 0 : _b.containerName;
            const svc = new ecs.FargateService(this, `${defaultContainerName}Service`, {
                taskDefinition: t.task,
                cluster,
                serviceName: t.serviceName,
                capacityProviderStrategies: (_c = t.capacityProviderStrategy) !== null && _c !== void 0 ? _c : (props.spot ? spotOnlyStrategy : undefined),
                desiredCount: t.desiredCount,
                enableExecuteCommand: (_d = props.enableExecuteCommand) !== null && _d !== void 0 ? _d : false,
                vpcSubnets: this.vpcSubnets,
                assignPublicIp: this.isPublicSubnets,
                circuitBreaker: props.circuitBreaker != false ? {
                    rollback: true,
                } : undefined,
                maxHealthyPercent: t.maxHealthyPercent,
                minHealthyPercent: t.minHealthyPercent,
                healthCheckGracePeriod: t.healthCheckGracePeriod,
            });
            this.service.push(svc);
            /**
             * determine if we have spot capacity in this cluster
             * scenario 1: FARGATE_SPOT with weight > 0
             * scenario 2: FARGATE_SPOT with base > 0
             * scenario 3: props.spot = true
             */
            (_e = t.capacityProviderStrategy) === null || _e === void 0 ? void 0 : _e.forEach(s => {
                if (s.capacityProvider == 'FARGATE_SPOT' && ((s.weight && s.weight > 0)
                    || (s.base && s.base > 0))) {
                    this.hasSpotCapacity = true;
                }
            });
        });
        // ensure the dependency
        if (!props.cluster) {
            const cp = this.node.tryFindChild('Cluster');
            this.service.forEach(s => {
                s.node.addDependency(cp);
            });
        }
        ;
        // Route53
        this.zoneName = (_f = (_e = props.route53Ops) === null || _e === void 0 ? void 0 : _e.zoneName) !== null && _f !== void 0 ? _f : 'svc.local';
        // spot termination handler by default
        if (this.hasSpotCapacity && props.spotTerminationHandler !== false) {
            this.createSpotTerminationHandler(cluster);
        }
    }
    createSpotTerminationHandler(cluster) {
        // create the handler
        const handler = new lambda.DockerImageFunction(this, 'SpotTermHandler', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/spot-term-handler')),
            timeout: cdk.Duration.seconds(20),
        });
        // create event rule
        const rule = new events.Rule(this, 'OnTaskStateChangeEvent', {
            eventPattern: {
                source: ['aws.ecs'],
                detailType: ['ECS Task State Change'],
                detail: {
                    clusterArn: [cluster.clusterArn],
                },
            },
        });
        rule.addTarget(new event_targets.LambdaFunction(handler));
        handler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:DescribeServices',
                'elasticloadbalancing:DeregisterTargets',
                'ec2:DescribeSubnets',
            ],
            resources: ['*'],
        }));
    }
    validateSubnets(vpc, vpcSubnets) {
        const subnets = vpc.selectSubnets(vpcSubnets);
        // get all subnets in the VPC
        const allsubnetIds = vpc.publicSubnets.concat(vpc.privateSubnets).concat(vpc.isolatedSubnets).map(x => x.subnetId);
        // validate the given subnets
        subnets.subnetIds.forEach(s => {
            if (!allsubnetIds.includes(s)) {
                throw new Error(`${s} does not exist in the VPC`);
            }
            if (vpc.isolatedSubnets.map(i => i.subnetId).includes(s)) {
                throw new Error(`Isolated subnet ${s} is not allowed`);
            }
        });
        const hasPublic = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
        const hasPrivate = subnets.subnetIds.some(s => new Set(vpc.privateSubnets.map(x => x.subnetId)).has(s));
        if (hasPublic && hasPrivate) {
            throw new Error('You should provide either all public or all private subnets, not both.');
        }
        this.isPublicSubnets = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
    }
}
exports.BaseFargateService = BaseFargateService;
_a = JSII_RTTI_SYMBOL_1;
BaseFargateService[_a] = { fqn: "cdk-fargate-patterns.BaseFargateService", version: "0.3.54" };
//# sourceMappingURL=data:application/json;base64,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