"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Expires = exports.StorageClass = exports.ServerSideEncryption = exports.CacheControl = exports.BucketDeployment = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const efs = require("@aws-cdk/aws-efs");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const lambda_layer_awscli_1 = require("@aws-cdk/lambda-layer-awscli");
const case_1 = require("case");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
// tag key has a limit of 128 characters
const CUSTOM_RESOURCE_OWNER_TAG = 'aws-cdk:cr-owned';
/**
 * `BucketDeployment` populates an S3 bucket with the contents of .zip files from
 * other S3 buckets or from local disk
 */
class BucketDeployment extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.requestDestinationArn = false;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_s3_deployment_BucketDeploymentProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, BucketDeployment);
            }
            throw error;
        }
        if (props.distributionPaths) {
            if (!props.distribution) {
                throw new Error('Distribution must be specified if distribution paths are specified');
            }
            if (!cdk.Token.isUnresolved(props.distributionPaths)) {
                if (!props.distributionPaths.every(distributionPath => cdk.Token.isUnresolved(distributionPath) || distributionPath.startsWith('/'))) {
                    throw new Error('Distribution paths must start with "/"');
                }
            }
        }
        if (props.useEfs && !props.vpc) {
            throw new Error('Vpc must be specified if useEfs is set');
        }
        const accessPointPath = '/lambda';
        let accessPoint;
        if (props.useEfs && props.vpc) {
            const accessMode = '0777';
            const fileSystem = this.getOrCreateEfsFileSystem(scope, {
                vpc: props.vpc,
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            accessPoint = fileSystem.addAccessPoint('AccessPoint', {
                path: accessPointPath,
                createAcl: {
                    ownerUid: '1001',
                    ownerGid: '1001',
                    permissions: accessMode,
                },
                posixUser: {
                    uid: '1001',
                    gid: '1001',
                },
            });
            accessPoint.node.addDependency(fileSystem.mountTargetsAvailable);
        }
        // Making VPC dependent on BucketDeployment so that CFN stack deletion is smooth.
        // Refer comments on https://github.com/aws/aws-cdk/pull/15220 for more details.
        if (props.vpc) {
            this.node.addDependency(props.vpc);
        }
        const mountPath = `/mnt${accessPointPath}`;
        const handler = new lambda.SingletonFunction(this, 'CustomResourceHandler', {
            uuid: this.renderSingletonUuid(props.memoryLimit, props.ephemeralStorageSize, props.vpc),
            code: lambda.Code.fromAsset(path.join(__dirname, 'lambda')),
            layers: [new lambda_layer_awscli_1.AwsCliLayer(this, 'AwsCliLayer')],
            runtime: lambda.Runtime.PYTHON_3_7,
            environment: props.useEfs ? {
                MOUNT_PATH: mountPath,
            } : undefined,
            handler: 'index.handler',
            lambdaPurpose: 'Custom::CDKBucketDeployment',
            timeout: cdk.Duration.minutes(15),
            role: props.role,
            memorySize: props.memoryLimit,
            ephemeralStorageSize: props.ephemeralStorageSize,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
            filesystem: accessPoint ? lambda.FileSystem.fromEfsAccessPoint(accessPoint, mountPath) : undefined,
            logRetention: props.logRetention,
        });
        const handlerRole = handler.role;
        if (!handlerRole) {
            throw new Error('lambda.SingletonFunction should have created a Role');
        }
        const sources = props.sources.map((source) => source.bind(this, { handlerRole }));
        props.destinationBucket.grantReadWrite(handler);
        if (props.accessControl) {
            props.destinationBucket.grantPutAcl(handler);
        }
        if (props.distribution) {
            handler.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['cloudfront:GetInvalidation', 'cloudfront:CreateInvalidation'],
                resources: ['*'],
            }));
        }
        // to avoid redundant stack updates, only include "SourceMarkers" if one of
        // the sources actually has markers.
        const hasMarkers = sources.some(source => source.markers);
        const crUniqueId = `CustomResource${this.renderUniqueId(props.memoryLimit, props.ephemeralStorageSize, props.vpc)}`;
        this.cr = new cdk.CustomResource(this, crUniqueId, {
            serviceToken: handler.functionArn,
            resourceType: 'Custom::CDKBucketDeployment',
            properties: {
                SourceBucketNames: sources.map(source => source.bucket.bucketName),
                SourceObjectKeys: sources.map(source => source.zipObjectKey),
                SourceMarkers: hasMarkers ? sources.map(source => source.markers ?? {}) : undefined,
                DestinationBucketName: props.destinationBucket.bucketName,
                DestinationBucketKeyPrefix: props.destinationKeyPrefix,
                RetainOnDelete: props.retainOnDelete,
                Prune: props.prune ?? true,
                Exclude: props.exclude,
                Include: props.include,
                UserMetadata: props.metadata ? mapUserMetadata(props.metadata) : undefined,
                SystemMetadata: mapSystemMetadata(props),
                DistributionId: props.distribution?.distributionId,
                DistributionPaths: props.distributionPaths,
                // Passing through the ARN sequences dependencees on the deployment
                DestinationBucketArn: cdk.Lazy.string({ produce: () => this.requestDestinationArn ? props.destinationBucket.bucketArn : undefined }),
            },
        });
        let prefix = props.destinationKeyPrefix ?
            `:${props.destinationKeyPrefix}` :
            '';
        prefix += `:${this.cr.node.addr.slice(-8)}`;
        const tagKey = CUSTOM_RESOURCE_OWNER_TAG + prefix;
        // destinationKeyPrefix can be 104 characters before we hit
        // the tag key limit of 128
        // '/this/is/a/random/key/prefix/that/is/a/lot/of/characters/do/we/think/that/it/will/ever/be/this/long?????'
        // better to throw an error here than wait for CloudFormation to fail
        if (tagKey.length > 128) {
            throw new Error('The BucketDeployment construct requires that the "destinationKeyPrefix" be <=104 characters');
        }
        /*
         * This will add a tag to the deployment bucket in the format of
         * `aws-cdk:cr-owned:{keyPrefix}:{uniqueHash}`
         *
         * For example:
         * {
         *   Key: 'aws-cdk:cr-owned:deploy/here/:240D17B3',
         *   Value: 'true',
         * }
         *
         * This will allow for scenarios where there is a single S3 Bucket that has multiple
         * BucketDeployment resources deploying to it. Each bucket + keyPrefix can be "owned" by
         * 1 or more BucketDeployment resources. Since there are some scenarios where multiple BucketDeployment
         * resources can deploy to the same bucket and key prefix (e.g. using include/exclude) we
         * also append part of the id to make the key unique.
         *
         * As long as a bucket + keyPrefix is "owned" by a BucketDeployment resource, another CR
         * cannot delete data. There are a couple of scenarios where this comes into play.
         *
         * 1. If the LogicalResourceId of the CustomResource changes (e.g. the crUniqueId changes)
         * CloudFormation will first issue a 'Create' to create the new CustomResource and will
         * update the Tag on the bucket. CloudFormation will then issue a 'Delete' on the old CustomResource
         * and since the new CR "owns" the Bucket+keyPrefix it will not delete the contents of the bucket
         *
         * 2. If the BucketDeployment resource is deleted _and_ it is the only CR for that bucket+keyPrefix
         * then CloudFormation will first remove the tag from the bucket and then issue a "Delete" to the
         * CR. Since there are no tags indicating that this bucket+keyPrefix is "owned" then it will delete
         * the contents.
         *
         * 3. If the BucketDeployment resource is deleted _and_ it is *not* the only CR for that bucket:keyPrefix
         * then CloudFormation will first remove the tag from the bucket and then issue a "Delete" to the CR.
         * Since there are other CRs that also "own" that bucket+keyPrefix there will still be a tag on the bucket
         * and the contents will not be removed.
         *
         * 4. If the BucketDeployment resource _and_ the S3 Bucket are both removed, then CloudFormation will first
         * issue a "Delete" to the CR and since there is a tag on the bucket the contents will not be removed. If you
         * want the contents of the bucket to be removed on bucket deletion, then `autoDeleteObjects` property should
         * be set to true on the Bucket.
         */
        cdk.Tags.of(props.destinationBucket).add(tagKey, 'true');
    }
    /**
     * The bucket after the deployment
     *
     * If you want to reference the destination bucket in another construct and make sure the
     * bucket deployment has happened before the next operation is started, pass the other construct
     * a reference to `deployment.deployedBucket`.
     *
     * Doing this replaces calling `otherResource.node.addDependency(deployment)`.
     */
    get deployedBucket() {
        this.requestDestinationArn = true;
        this._deployedBucket = this._deployedBucket ?? s3.Bucket.fromBucketArn(this, 'DestinationBucket', cdk.Token.asString(this.cr.getAtt('DestinationBucketArn')));
        return this._deployedBucket;
    }
    renderUniqueId(memoryLimit, ephemeralStorageSize, vpc) {
        let uuid = '';
        // if the user specifes a custom memory limit, we define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (memoryLimit) {
            if (cdk.Token.isUnresolved(memoryLimit)) {
                throw new Error("Can't use tokens when specifying 'memoryLimit' since we use it to identify the singleton custom resource handler.");
            }
            uuid += `-${memoryLimit.toString()}MiB`;
        }
        // if the user specifies a custom ephemeral storage size, we define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (ephemeralStorageSize) {
            if (ephemeralStorageSize.isUnresolved()) {
                throw new Error("Can't use tokens when specifying 'ephemeralStorageSize' since we use it to identify the singleton custom resource handler.");
            }
            uuid += `-${ephemeralStorageSize.toMebibytes().toString()}MiB`;
        }
        // if the user specifies a VPC, we define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        // A VPC is a must if EFS storage is used and that's why we are only using VPC in uuid.
        if (vpc) {
            uuid += `-${vpc.node.addr}`;
        }
        return uuid;
    }
    renderSingletonUuid(memoryLimit, ephemeralStorageSize, vpc) {
        let uuid = '8693BB64-9689-44B6-9AAF-B0CC9EB8756C';
        uuid += this.renderUniqueId(memoryLimit, ephemeralStorageSize, vpc);
        return uuid;
    }
    /**
     * Function to get/create a stack singleton instance of EFS FileSystem per vpc.
     *
     * @param scope Construct
     * @param fileSystemProps EFS FileSystemProps
     */
    getOrCreateEfsFileSystem(scope, fileSystemProps) {
        const stack = cdk.Stack.of(scope);
        const uuid = `BucketDeploymentEFS-VPC-${fileSystemProps.vpc.node.addr}`;
        return stack.node.tryFindChild(uuid) ?? new efs.FileSystem(scope, uuid, fileSystemProps);
    }
}
exports.BucketDeployment = BucketDeployment;
_a = JSII_RTTI_SYMBOL_1;
BucketDeployment[_a] = { fqn: "@aws-cdk/aws-s3-deployment.BucketDeployment", version: "1.160.0" };
/**
 * Metadata
 */
function mapUserMetadata(metadata) {
    const mapKey = (key) => key.toLowerCase();
    return Object.keys(metadata).reduce((o, key) => ({ ...o, [mapKey(key)]: metadata[key] }), {});
}
function mapSystemMetadata(metadata) {
    const res = {};
    if (metadata.cacheControl) {
        res['cache-control'] = metadata.cacheControl.map(c => c.value).join(', ');
    }
    if (metadata.expires) {
        res.expires = metadata.expires.date.toUTCString();
    }
    if (metadata.contentDisposition) {
        res['content-disposition'] = metadata.contentDisposition;
    }
    if (metadata.contentEncoding) {
        res['content-encoding'] = metadata.contentEncoding;
    }
    if (metadata.contentLanguage) {
        res['content-language'] = metadata.contentLanguage;
    }
    if (metadata.contentType) {
        res['content-type'] = metadata.contentType;
    }
    if (metadata.serverSideEncryption) {
        res.sse = metadata.serverSideEncryption;
    }
    if (metadata.storageClass) {
        res['storage-class'] = metadata.storageClass;
    }
    if (metadata.websiteRedirectLocation) {
        res['website-redirect'] = metadata.websiteRedirectLocation;
    }
    if (metadata.serverSideEncryptionAwsKmsKeyId) {
        res['sse-kms-key-id'] = metadata.serverSideEncryptionAwsKmsKeyId;
    }
    if (metadata.serverSideEncryptionCustomerAlgorithm) {
        res['sse-c-copy-source'] = metadata.serverSideEncryptionCustomerAlgorithm;
    }
    if (metadata.accessControl) {
        res.acl = case_1.kebab(metadata.accessControl.toString());
    }
    return Object.keys(res).length === 0 ? undefined : res;
}
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
class CacheControl {
    constructor(
    /**
     * The raw cache control setting.
     */
    value) {
        this.value = value;
    }
    /**
     * Sets 'must-revalidate'.
     */
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    /**
     * Sets 'no-cache'.
     */
    static noCache() { return new CacheControl('no-cache'); }
    /**
     * Sets 'no-transform'.
     */
    static noTransform() { return new CacheControl('no-transform'); }
    /**
     * Sets 'public'.
     */
    static setPublic() { return new CacheControl('public'); }
    /**
     * Sets 'private'.
     */
    static setPrivate() { return new CacheControl('private'); }
    /**
     * Sets 'proxy-revalidate'.
     */
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    /**
     * Sets 'max-age=<duration-in-seconds>'.
     */
    static maxAge(t) { return new CacheControl(`max-age=${t.toSeconds()}`); }
    /**
     * Sets 's-maxage=<duration-in-seconds>'.
     */
    static sMaxAge(t) { return new CacheControl(`s-maxage=${t.toSeconds()}`); }
    /**
     * Constructs a custom cache control key from the literal value.
     */
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
_b = JSII_RTTI_SYMBOL_1;
CacheControl[_b] = { fqn: "@aws-cdk/aws-s3-deployment.CacheControl", version: "1.160.0" };
/**
 * Indicates whether server-side encryption is enabled for the object, and whether that encryption is
 * from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var ServerSideEncryption;
(function (ServerSideEncryption) {
    /**
     * 'AES256'
     */
    ServerSideEncryption["AES_256"] = "AES256";
    /**
     * 'aws:kms'
     */
    ServerSideEncryption["AWS_KMS"] = "aws:kms";
})(ServerSideEncryption = exports.ServerSideEncryption || (exports.ServerSideEncryption = {}));
/**
 * Storage class used for storing the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var StorageClass;
(function (StorageClass) {
    /**
     * 'STANDARD'
     */
    StorageClass["STANDARD"] = "STANDARD";
    /**
     * 'REDUCED_REDUNDANCY'
     */
    StorageClass["REDUCED_REDUNDANCY"] = "REDUCED_REDUNDANCY";
    /**
     * 'STANDARD_IA'
     */
    StorageClass["STANDARD_IA"] = "STANDARD_IA";
    /**
     * 'ONEZONE_IA'
     */
    StorageClass["ONEZONE_IA"] = "ONEZONE_IA";
    /**
     * 'INTELLIGENT_TIERING'
     */
    StorageClass["INTELLIGENT_TIERING"] = "INTELLIGENT_TIERING";
    /**
     * 'GLACIER'
     */
    StorageClass["GLACIER"] = "GLACIER";
    /**
     * 'DEEP_ARCHIVE'
     */
    StorageClass["DEEP_ARCHIVE"] = "DEEP_ARCHIVE";
})(StorageClass = exports.StorageClass || (exports.StorageClass = {}));
/**
 * Used for HTTP expires header, which influences downstream caches. Does NOT influence deletion of the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 *
 * @deprecated use core.Expiration
 */
class Expires {
    constructor(
    /**
     * The raw expiration date expression.
     */
    value) {
        this.value = value;
    }
    /**
     * Expire at the specified date
     * @param d date to expire at
     */
    static atDate(d) { try {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-s3-deployment.Expires#atDate", "use core.Expiration");
    }
    catch (error) {
        if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
            Error.captureStackTrace(error, this.atDate);
        }
        throw error;
    } return new Expires(d.toUTCString()); }
    /**
     * Expire at the specified timestamp
     * @param t timestamp in unix milliseconds
     */
    static atTimestamp(t) { try {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-s3-deployment.Expires#atTimestamp", "use core.Expiration");
    }
    catch (error) {
        if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
            Error.captureStackTrace(error, this.atTimestamp);
        }
        throw error;
    } return Expires.atDate(new Date(t)); }
    /**
     * Expire once the specified duration has passed since deployment time
     * @param t the duration to wait before expiring
     */
    static after(t) { try {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-s3-deployment.Expires#after", "use core.Expiration");
    }
    catch (error) {
        if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
            Error.captureStackTrace(error, this.after);
        }
        throw error;
    } return Expires.atDate(new Date(Date.now() + t.toMilliseconds())); }
    /**
     * Create an expiration date from a raw date string.
     */
    static fromString(s) { try {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-s3-deployment.Expires#fromString", "use core.Expiration");
    }
    catch (error) {
        if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
            Error.captureStackTrace(error, this.fromString);
        }
        throw error;
    } return new Expires(s); }
}
exports.Expires = Expires;
_c = JSII_RTTI_SYMBOL_1;
Expires[_c] = { fqn: "@aws-cdk/aws-s3-deployment.Expires", version: "1.160.0" };
//# sourceMappingURL=data:application/json;base64,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