# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ext', 'ext.context']

package_data = \
{'': ['*']}

install_requires = \
['discord.py>=1.6.0,<2.0.0']

setup_kwargs = {
    'name': 'discord.ext.context',
    'version': '0.1.7',
    'description': 'Globally accessable context for discord.py events.',
    'long_description': '# discord.ext.context\n\nA globally accessible context object for discord.py events.\n\nAble to be used in both base clients and the commands extension.\n\n## Installation\n```shell\npip install discord.ext.context\n```\n\n## Requirements\n- Python 3.7+\n- [discord.py 1.6+](https://pypi.org/project/discord.py/)\n\n## Register a Discord bot/client\n\nSubclass the `context.ContextClient` base class, ensuring that it\'s first in inheritance order.\n\n```python\nimport discord\nfrom discord.ext import context\n\nclass Client(context.ContextClient, discord.Client):\n    ...\n```\n\n## Using EventContext\n\nImport `context.ctx` from anywhere, and it\'ll have its attributes set based on the event that your call stack originates from.\n\n```python\nimport discord\nfrom discord.ext.context import ctx\n\nasync def log_reaction():\n    await ctx.channel.send(f"{ctx.user} reacted with {ctx.emoji}")\n\nclient = discord.Client()\n    \n@client.event\nasync def on_raw_reaction_add():\n    await log_reaction()\n```\n\n## Exceptions\n### `ContextNotSet`\nAccessing a context value before it\'s set will have this exception raised. To have a fallback value used instead, use the contextmanager [`with ctx.default():`](https://github.com/scragly/discord.ext.context#ctxdefaultall_default--message_novalue-emoji_novalue-user_novalue-channel_novalue-guild_novalue-cmd_ext_novalue).\n\n## Attributes\n\n### `ctx.message`: [`discord.PartialMessage`](https://discordpy.readthedocs.io/en/latest/api.html#discord.PartialMessage)\nShould always be a `PartialMessage`. If a `Message` instance is needed, an up to date copy can be retrieved with `PartialMessage.fetch()`.\n\n### `ctx.emoji`: [`discord.Emoji`](https://discordpy.readthedocs.io/en/latest/api.html#discord.Emoji) or [`discord.PartialEmoji`](https://discordpy.readthedocs.io/en/latest/api.html#discord.PartialEmoji)\nOften representing a reaction interacted with by a user; useful for user interactions that use reaction-based sessions.\n\n### `ctx.channel`: [`discord.abc.Messageable`](https://discordpy.readthedocs.io/en/latest/api.html#discord.abc.Messageable)\nWill always be a text channel or user-type object that\'s possible to send messages to. Does not include voice channels.\n\n### `ctx.user`: [`discord.Member`](https://discordpy.readthedocs.io/en/latest/api.html#discord.Member) or [`discord.User`](https://discordpy.readthedocs.io/en/latest/api.html#discord.User)\nIf setting a `discord.User` instance and the user shares only a single guild with the client, it\'ll replace it with the `discord.Member` instance.\n\n### `ctx.guild`: [`discord.Guild`](https://discordpy.readthedocs.io/en/latest/api.html#discord.Guild)\nWill only be set on guild-specific events.\n\n### `ctx.cmd_ctx`: [`discord.ext.commands.Context`](https://discordpy.readthedocs.io/en/latest/ext/commands/api.html#discord.ext.commands.Context)\nWill only be set on the command event, with other EventContext values being set using it.\n\n### `ctx.event`: `str`\nThe name of the event type this context originates from.\n\n### `ctx.client`: [`discord.Client`](https://discordpy.readthedocs.io/en/latest/api.html#discord.Client)\nThe instance of the discord.py client being hooked into.\n\n### `ctx.bot`: [`discord.Client`](https://discordpy.readthedocs.io/en/latest/api.html#discord.Client)\nAlias for `ctx.client`.\n\n## Methods\n\n### `ctx.set(*, message=None, emoji=None, user=None, channel=None, guild=None)`\nSets the values for the current context to be used across future call stacks. Won\'t impact asynchronous calls from other events.\n\n### `@ctx.register_hook(event)`\nDecorator for registering an event to be handled by the decorated function. Will override existing hooks if a duplicate exists.\n\n### `ctx.default(all_default, *, message=_NoValue, emoji=_NoValue, user=_NoValue, channel=_NoValue, guild=_NoValue, cmd_ext=_NoValue)`\nContext manager for registering default values to be used if a value isn\'t set. On leaving the context manager\'s scope, `ctx` will revert to original state.\n\nUse `all_default` to set all the available ctx.values to the one value. This can be useful for allowing None to be returned for nonset contexts.\n\n#### Examples\n```python\nwith ctx.default(None):\n    if ctx.channel:\n        await ctx.channel.send("Yes")\n```\n\n```python\nwith ctx.default(channel=fallback_channel, user=None):\n    if ctx.user:\n        await ctx.channel.send(f"{ctx.user.display_name}")\n```\n\nIf `ctx.channel` or `ctx.user` is not yet set, it\'ll be assigned the fallback arguments. This includes being able to set a value to `None` instead of raising `ContextNotSet`.\n\nIt can also be used as a decorator for a function.\n\n```python\n@ctx.default(channel=fallback_channel, user=None)\nasync def show_name():\n    if ctx.user:\n        await ctx.channel.send(f"{ctx.user.display_name}")\n```\n\n### `ctx.ephemeral(*, message=_NoValue, emoji=_NoValue, user=_NoValue, channel=_NoValue, guild=_NoValue, cmd_ext=_NoValue)`\nContext manager for overriding context values. On leaving the context manager\'s scope, `ctx` will revert to original state.\n\n### Examples\n```python\nwith ctx.ephemeral(channel=log_channel, user=None):\n    if ctx.user:\n        await ctx.channel.send(f"{ctx.user.display_name} did a thing.")\n```\n\nIf `ctx.channel` or `ctx.user` is not yet set, it\'ll be assigned the fallback arguments. This includes being able to set a value to `None` instead of raising `ContextNotSet`.\n\nIt can also be used as a decorator for a function.\n\n```python\n@ctx.ephemeral(channel=log_channel, user=None)\nasync def show_name():\n    if ctx.user:\n        await ctx.channel.send(f"{ctx.user.display_name} did a thing.")\n```\n\n## Events\n\nBy default, the following events are hooked by EventContext:\n\n### Messages\n- `message`\n- `message_delete`\n- `message_edit`\n- `raw_message_delete`\n- `raw_message_edit`\n\n### Reactions\n- `reaction_add`\n- `reaction_remove`\n- `raw_reaction_add`\n- `raw_reaction_remove`\n- `reaction_clear`\n- `reaction_clear_emoji`\n- `raw_reaction_clear`\n- `raw_reaction_clear_emoji`\n\n### Channels\n- `typing`\n- `guild_channel_update`\n- `guild_channel_create`\n- `guild_channel_delete`\n- `guild_channel_pins_update`\n- `webhooks_update`\n\n### Guilds\n- `guild_update`\n- `guild_join`\n- `guild_remove`\n- `guild_integrations_update`\n- `guild_emojis_update`\n- `guild_available`\n- `guild_unavailable`\n\n### Members\n- `member_update`\n- `member_join`\n- `member_remove`\n- `member_ban_hook`\n- `member_unban_hook`\n\n### Roles\n- `guild_role_update_hook`\n- `guild_role_create_hook`\n- `guild_role_delete_hook`\n\n### Commands\n- `command`\n\n\nYou can add more event hooks or replace the default ones with the decorator:\n```python\n@EventContext.register_hook(event_name)\ndef event_hook(*args, **kwargs):\n    ...\n```\n',
    'author': 'scragly',
    'author_email': '29337040+scragly@users.noreply.github.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/scragly/discord.ext.context',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
