"use strict";
const assert_1 = require("@aws-cdk/assert");
const certificatemanager = require("@aws-cdk/aws-certificatemanager");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'distribution with custom origin adds custom origin'(test) {
        const stack = new cdk.Stack();
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    originHeaders: {
                        "X-Custom-Header": "somevalue",
                    },
                    customOriginSource: {
                        domainName: "myorigin.com",
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                        }
                    ],
                }
            ]
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "AnAmazingWebsiteProbablyCFDistribution47E3983B": {
                    "Type": "AWS::CloudFront::Distribution",
                    "Properties": {
                        "DistributionConfig": {
                            "DefaultCacheBehavior": {
                                "AllowedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "CachedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "ForwardedValues": {
                                    "Cookies": {
                                        "Forward": "none"
                                    },
                                    "QueryString": false
                                },
                                "TargetOriginId": "origin1",
                                "ViewerProtocolPolicy": "redirect-to-https",
                                "Compress": true
                            },
                            "DefaultRootObject": "index.html",
                            "Enabled": true,
                            "HttpVersion": "http2",
                            "IPV6Enabled": true,
                            "Origins": [
                                {
                                    "CustomOriginConfig": {
                                        "HTTPPort": 80,
                                        "HTTPSPort": 443,
                                        "OriginKeepaliveTimeout": 5,
                                        "OriginProtocolPolicy": "https-only",
                                        "OriginReadTimeout": 30,
                                        "OriginSSLProtocols": [
                                            "TLSv1.2"
                                        ]
                                    },
                                    "DomainName": "myorigin.com",
                                    "Id": "origin1",
                                    "OriginCustomHeaders": [
                                        {
                                            "HeaderName": "X-Custom-Header",
                                            "HeaderValue": "somevalue"
                                        }
                                    ]
                                }
                            ],
                            "PriceClass": "PriceClass_100",
                            "ViewerCertificate": {
                                "CloudFrontDefaultCertificate": true
                            }
                        }
                    }
                }
            }
        });
        test.done();
    },
    'most basic distribution'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                        }
                    ]
                }
            ]
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Bucket83908E77": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                },
                "AnAmazingWebsiteProbablyCFDistribution47E3983B": {
                    "Type": "AWS::CloudFront::Distribution",
                    "Properties": {
                        "DistributionConfig": {
                            "DefaultRootObject": "index.html",
                            "Origins": [
                                {
                                    "DomainName": {
                                        "Fn::GetAtt": [
                                            "Bucket83908E77",
                                            "RegionalDomainName"
                                        ]
                                    },
                                    "Id": "origin1",
                                    "S3OriginConfig": {}
                                }
                            ],
                            "ViewerCertificate": {
                                "CloudFrontDefaultCertificate": true
                            },
                            "PriceClass": "PriceClass_100",
                            "DefaultCacheBehavior": {
                                "AllowedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "CachedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "TargetOriginId": "origin1",
                                "ViewerProtocolPolicy": "redirect-to-https",
                                "ForwardedValues": {
                                    "QueryString": false,
                                    "Cookies": { "Forward": "none" }
                                },
                                "Compress": true
                            },
                            "Enabled": true,
                            "IPV6Enabled": true,
                            "HttpVersion": "http2"
                        }
                    }
                }
            }
        });
        test.done();
    },
    'distribution with trusted signers on default distribution'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                            trustedSigners: ["1234"],
                        },
                    ]
                }
            ]
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Bucket83908E77": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                },
                "AnAmazingWebsiteProbablyCFDistribution47E3983B": {
                    "Type": "AWS::CloudFront::Distribution",
                    "Properties": {
                        "DistributionConfig": {
                            "DefaultRootObject": "index.html",
                            "Origins": [
                                {
                                    "DomainName": {
                                        "Fn::GetAtt": [
                                            "Bucket83908E77",
                                            "RegionalDomainName"
                                        ]
                                    },
                                    "Id": "origin1",
                                    "S3OriginConfig": {}
                                }
                            ],
                            "ViewerCertificate": {
                                "CloudFrontDefaultCertificate": true
                            },
                            "PriceClass": "PriceClass_100",
                            "DefaultCacheBehavior": {
                                "AllowedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "CachedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "TargetOriginId": "origin1",
                                "ViewerProtocolPolicy": "redirect-to-https",
                                "ForwardedValues": {
                                    "QueryString": false,
                                    "Cookies": { "Forward": "none" }
                                },
                                "TrustedSigners": [
                                    "1234"
                                ],
                                "Compress": true
                            },
                            "Enabled": true,
                            "IPV6Enabled": true,
                            "HttpVersion": "http2"
                        }
                    }
                }
            }
        });
        test.done();
    },
    'distribution with ViewerProtocolPolicy set to a non-default value'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            viewerProtocolPolicy: lib_1.ViewerProtocolPolicy.ALLOW_ALL,
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                        }
                    ]
                }
            ]
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Bucket83908E77": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                },
                "AnAmazingWebsiteProbablyCFDistribution47E3983B": {
                    "Type": "AWS::CloudFront::Distribution",
                    "Properties": {
                        "DistributionConfig": {
                            "DefaultRootObject": "index.html",
                            "Origins": [
                                {
                                    "DomainName": {
                                        "Fn::GetAtt": [
                                            "Bucket83908E77",
                                            "RegionalDomainName"
                                        ]
                                    },
                                    "Id": "origin1",
                                    "S3OriginConfig": {}
                                }
                            ],
                            "ViewerCertificate": {
                                "CloudFrontDefaultCertificate": true
                            },
                            "PriceClass": "PriceClass_100",
                            "DefaultCacheBehavior": {
                                "AllowedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "CachedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "TargetOriginId": "origin1",
                                "ViewerProtocolPolicy": "allow-all",
                                "ForwardedValues": {
                                    "QueryString": false,
                                    "Cookies": { "Forward": "none" }
                                },
                                "Compress": true
                            },
                            "Enabled": true,
                            "IPV6Enabled": true,
                            "HttpVersion": "http2",
                        }
                    }
                }
            }
        });
        test.done();
    },
    'distribution with disabled compression'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                            compress: false
                        }
                    ]
                }
            ]
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Bucket83908E77": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                },
                "AnAmazingWebsiteProbablyCFDistribution47E3983B": {
                    "Type": "AWS::CloudFront::Distribution",
                    "Properties": {
                        "DistributionConfig": {
                            "DefaultRootObject": "index.html",
                            "Origins": [
                                {
                                    "DomainName": {
                                        "Fn::GetAtt": [
                                            "Bucket83908E77",
                                            "RegionalDomainName"
                                        ]
                                    },
                                    "Id": "origin1",
                                    "S3OriginConfig": {}
                                }
                            ],
                            "ViewerCertificate": {
                                "CloudFrontDefaultCertificate": true
                            },
                            "PriceClass": "PriceClass_100",
                            "DefaultCacheBehavior": {
                                "AllowedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "CachedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "TargetOriginId": "origin1",
                                "ViewerProtocolPolicy": "redirect-to-https",
                                "ForwardedValues": {
                                    "QueryString": false,
                                    "Cookies": { "Forward": "none" }
                                },
                                "Compress": false
                            },
                            "Enabled": true,
                            "IPV6Enabled": true,
                            "HttpVersion": "http2"
                        }
                    }
                }
            }
        });
        test.done();
    },
    'distribution with resolvable lambda-association'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        const lambdaFunction = new lambda.SingletonFunction(stack, 'Lambda', {
            uuid: 'xxxx-xxxx-xxxx-xxxx',
            code: lambda.Code.inline('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X
        });
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                            lambdaFunctionAssociations: [{
                                    eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                                    lambdaFunction: lambdaFunction.latestVersion
                                }]
                        }
                    ]
                }
            ]
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
            "DistributionConfig": {
                "DefaultCacheBehavior": {
                    "LambdaFunctionAssociations": [
                        {
                            "EventType": "origin-request",
                            "LambdaFunctionARN": {
                                "Fn::Join": [
                                    "",
                                    [
                                        { "Fn::GetAtt": ["SingletonLambdaxxxxxxxxxxxxxxxx69D4268A", "Arn"] },
                                        ":$LATEST"
                                    ]
                                ]
                            }
                        }
                    ],
                },
            }
        }));
        test.done();
    },
    'distribution has a defaultChild'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        const distribution = new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket
                    },
                    behaviors: [{ isDefaultBehavior: true }]
                }
            ]
        });
        test.ok(distribution.node.defaultChild instanceof lib_1.CfnDistribution);
        test.done();
    },
    'allows multiple aliasConfiguration CloudFrontWebDistribution per stack'(test) {
        const stack = new cdk.Stack();
        const s3BucketSource = new s3.Bucket(stack, 'Bucket');
        const originConfigs = [{
                s3OriginSource: { s3BucketSource },
                behaviors: [{ isDefaultBehavior: true }]
            }];
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs,
            aliasConfiguration: { acmCertRef: 'acm_ref', names: ['www.example.com'] },
        });
        new lib_1.CloudFrontWebDistribution(stack, 'AnotherAmazingWebsiteProbably', {
            originConfigs,
            aliasConfiguration: { acmCertRef: 'another_acm_ref', names: ['ftp.example.com'] },
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
            "DistributionConfig": {
                "Aliases": ["www.example.com"],
                "ViewerCertificate": {
                    "AcmCertificateArn": "acm_ref",
                    "SslSupportMethod": "sni-only"
                }
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
            "DistributionConfig": {
                "Aliases": ["ftp.example.com"],
                "ViewerCertificate": {
                    "AcmCertificateArn": "another_acm_ref",
                    "SslSupportMethod": "sni-only"
                }
            }
        }));
        test.done();
    },
    'viewerCertificate': {
        'acmCertificate': {
            'base usage'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                const certificate = new certificatemanager.Certificate(stack, 'cert', {
                    domainName: 'example.com',
                });
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }]
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromAcmCertificate(certificate),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    "DistributionConfig": {
                        "Aliases": [],
                        "ViewerCertificate": {
                            "AcmCertificateArn": {
                                "Ref": "cert56CA94EB"
                            },
                            "SslSupportMethod": "sni-only"
                        }
                    }
                }));
                test.done();
            },
            'imported certificate fromCertificateArn'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                const certificate = certificatemanager.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }]
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromAcmCertificate(certificate),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    "DistributionConfig": {
                        "Aliases": [],
                        "ViewerCertificate": {
                            "AcmCertificateArn": "arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d",
                            "SslSupportMethod": "sni-only"
                        }
                    }
                }));
                test.done();
            },
            'advanced usage'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                const certificate = new certificatemanager.Certificate(stack, 'cert', {
                    domainName: 'example.com',
                });
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }]
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromAcmCertificate(certificate, {
                        securityPolicy: lib_1.SecurityPolicyProtocol.SSL_V3,
                        sslMethod: lib_1.SSLMethod.VIP,
                        aliases: ['example.com', 'www.example.com']
                    }),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    "DistributionConfig": {
                        "Aliases": ["example.com", "www.example.com"],
                        "ViewerCertificate": {
                            "AcmCertificateArn": {
                                "Ref": "cert56CA94EB"
                            },
                            "MinimumProtocolVersion": "SSLv3",
                            "SslSupportMethod": "vip"
                        }
                    }
                }));
                test.done();
            },
        },
        'iamCertificate': {
            'base usage'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }]
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromIamCertificate('test'),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    "DistributionConfig": {
                        "Aliases": [],
                        "ViewerCertificate": {
                            "IamCertificateId": "test",
                            "SslSupportMethod": "sni-only"
                        }
                    }
                }));
                test.done();
            },
            'advanced usage'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }]
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromIamCertificate('test', {
                        securityPolicy: lib_1.SecurityPolicyProtocol.TLS_V1,
                        sslMethod: lib_1.SSLMethod.VIP,
                        aliases: ['example.com']
                    }),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    "DistributionConfig": {
                        "Aliases": ["example.com"],
                        "ViewerCertificate": {
                            "IamCertificateId": "test",
                            "MinimumProtocolVersion": "TLSv1",
                            "SslSupportMethod": "vip"
                        }
                    }
                }));
                test.done();
            },
        },
        'cloudFrontDefaultCertificate': {
            'base usage'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }]
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromCloudFrontDefaultCertificate(),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    "DistributionConfig": {
                        "Aliases": [],
                        "ViewerCertificate": {
                            "CloudFrontDefaultCertificate": true
                        }
                    }
                }));
                test.done();
            },
            'aliases are set'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }]
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromCloudFrontDefaultCertificate('example.com', 'www.example.com'),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    "DistributionConfig": {
                        "Aliases": ["example.com", "www.example.com"],
                        "ViewerCertificate": {
                            "CloudFrontDefaultCertificate": true
                        }
                    }
                }));
                test.done();
            },
        },
        'errors': {
            'throws if both deprecated aliasConfiguration and viewerCertificate'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                test.throws(() => {
                    new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                        originConfigs: [{
                                s3OriginSource: { s3BucketSource: sourceBucket },
                                behaviors: [{ isDefaultBehavior: true }]
                            }],
                        aliasConfiguration: { acmCertRef: 'test', names: ['ftp.example.com'] },
                        viewerCertificate: lib_1.ViewerCertificate.fromCloudFrontDefaultCertificate('example.com', 'www.example.com'),
                    });
                }, /You cannot set both aliasConfiguration and viewerCertificate properties/);
                test.done();
            },
            'throws if invalid security policy for SSL method'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                test.throws(() => {
                    new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                        originConfigs: [{
                                s3OriginSource: { s3BucketSource: sourceBucket },
                                behaviors: [{ isDefaultBehavior: true }]
                            }],
                        viewerCertificate: lib_1.ViewerCertificate.fromIamCertificate('test', {
                            securityPolicy: lib_1.SecurityPolicyProtocol.TLS_V1_1_2016,
                            sslMethod: lib_1.SSLMethod.VIP
                        }),
                    });
                }, /TLSv1.1_2016 is not compabtible with sslMethod vip./);
                test.done();
            },
            // FIXME https://github.com/aws/aws-cdk/issues/4724
            'does not throw if acmCertificate explicitly not in us-east-1'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                const certificate = certificatemanager.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:eu-west-3:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }]
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromAcmCertificate(certificate),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    "DistributionConfig": {
                        "Aliases": [],
                        "ViewerCertificate": {
                            "AcmCertificateArn": "arn:aws:acm:eu-west-3:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d",
                            "SslSupportMethod": "sni-only"
                        }
                    }
                }));
                test.done();
            },
        }
    },
    'edgelambda.amazonaws.com is added to the trust policy of lambda'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        const fn = new lambda.Function(stack, 'Lambda', {
            code: lambda.Code.fromInline('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X
        });
        const lambdaVersion = new lambda.Version(stack, 'LambdaVersion', { lambda: fn });
        // WHEN
        new lib_1.CloudFrontWebDistribution(stack, 'MyDistribution', {
            originConfigs: [
                {
                    s3OriginSource: { s3BucketSource: sourceBucket },
                    behaviors: [
                        {
                            isDefaultBehavior: true, lambdaFunctionAssociations: [
                                {
                                    eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                                    lambdaFunction: lambdaVersion
                                }
                            ]
                        }
                    ]
                }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                "Statement": [
                    {
                        "Action": "sts:AssumeRole",
                        "Effect": "Allow",
                        "Principal": {
                            "Service": "lambda.amazonaws.com"
                        }
                    },
                    {
                        "Action": "sts:AssumeRole",
                        "Effect": "Allow",
                        "Principal": {
                            "Service": "edgelambda.amazonaws.com"
                        }
                    }
                ],
                "Version": "2012-10-17"
            }
        }));
        test.done();
    },
    'edgelambda.amazonaws.com is not added to lambda role for imported functions'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        const lambdaVersion = lambda.Version.fromVersionArn(stack, 'Version', 'arn:my-version');
        // WHEN
        new lib_1.CloudFrontWebDistribution(stack, 'MyDistribution', {
            originConfigs: [
                {
                    s3OriginSource: { s3BucketSource: sourceBucket },
                    behaviors: [
                        {
                            isDefaultBehavior: true, lambdaFunctionAssociations: [
                                {
                                    eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                                    lambdaFunction: lambdaVersion
                                }
                            ]
                        }
                    ]
                }
            ]
        });
        assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Role'));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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