# Generated by Finn Stutzenstein on 2019-10-29 10:55

from decimal import Decimal

from django.db import migrations, transaction


def change_pollmethods(apps, schema_editor):
    """ yn->YN, yna->YNA """
    AssignmentPoll = apps.get_model("assignments", "AssignmentPoll")
    pollmethod_map = {
        "yn": "YN",
        "yna": "YNA",
        "votes": "votes",
    }
    for poll in AssignmentPoll.objects.all():
        poll.pollmethod = pollmethod_map.get(poll.pollmethod, "YNA")
        poll.save(skip_autoupdate=True)


def set_poll_titles(apps, schema_editor):
    """
    Sets titles to their indexes
    """
    Assignment = apps.get_model("assignments", "Assignment")
    for assignment in Assignment.objects.all():
        for i, poll in enumerate(assignment.polls.order_by("pk").all()):
            poll.title = str(i + 1)
            poll.save(skip_autoupdate=True)


def set_onehunderd_percent_bases(apps, schema_editor):
    AssignmentPoll = apps.get_model("assignments", "AssignmentPoll")
    ConfigStore = apps.get_model("core", "ConfigStore")
    base_map = {
        "YES_NO_ABSTAIN": "YNA",
        "YES_NO": "YN",
        "VALID": "valid",
        "CAST": "cast",
        "DISABLED": "disabled",
    }
    try:
        config = ConfigStore.objects.get(key="assignments_poll_100_percent_base")
        value = base_map[config.value]
    except (ConfigStore.DoesNotExist, KeyError):
        value = "YNA"

    for poll in AssignmentPoll.objects.all():
        if poll.pollmethod == "votes" and value in ("YN", "YNA"):
            poll.onehundred_percent_base = "votes"
        elif poll.pollmethod == "YN" and value == "YNA":
            poll.onehundred_percent_base = "YN"
        else:
            poll.onehundred_percent_base = value
        poll.save(skip_autoupdate=True)


def set_majority_methods(apps, schema_editor):
    AssignmentPoll = apps.get_model("assignments", "AssignmentPoll")
    ConfigStore = apps.get_model("core", "ConfigStore")
    majority_map = {
        "simple_majority": "simple",
        "two-thirds_majority": "two_thirds",
        "three-quarters_majority": "three_quarters",
        "disabled": "disabled",
    }
    try:
        config = ConfigStore.objects.get(key="assignments_poll_default_majority_method")
        value = majority_map[config.value]
    except (ConfigStore.DoesNotExist, KeyError):
        value = "simple"

    for poll in AssignmentPoll.objects.all():
        poll.majority_method = value
        poll.save(skip_autoupdate=True)


def convert_votes(apps, schema_editor):
    AssignmentVote = apps.get_model("assignments", "AssignmentVote")
    value_map = {
        "Yes": "Y",
        "No": "N",
        "Abstain": "A",
        "Votes": "Y",
    }
    for vote in AssignmentVote.objects.all():
        vote.value = value_map[vote.value]
        vote.save(skip_autoupdate=True)


def convert_votesabstain(apps, schema_editor):
    AssignmentPoll = apps.get_model("assignments", "AssignmentPoll")
    AssignmentVote = apps.get_model("assignments", "AssignmentVote")
    for poll in AssignmentPoll.objects.all():
        if poll.votesabstain is not None and poll.votesabstain > Decimal(0):
            with transaction.atomic():
                option = poll.options.first()
                vote = AssignmentVote(
                    option=option, value="A", weight=poll.votesabstain
                )
                vote.save(skip_autoupdate=True)


def convert_votesno(apps, schema_editor):
    AssignmentPoll = apps.get_model("assignments", "AssignmentPoll")
    AssignmentVote = apps.get_model("assignments", "AssignmentVote")
    for poll in AssignmentPoll.objects.all():
        if poll.votesno is not None and poll.votesno > Decimal(0):
            with transaction.atomic():
                option = poll.options.first()
                vote = AssignmentVote(option=option, value="N", weight=poll.votesno)
                vote.save(skip_autoupdate=True)


def set_correct_state(apps, schema_editor):
    """ if poll.published, set state to published """
    AssignmentPoll = apps.get_model("assignments", "AssignmentPoll")
    AssignmentVote = apps.get_model("assignments", "AssignmentVote")
    for poll in AssignmentPoll.objects.all():
        # Voting, that are published (old field) but have no votes, will be
        # left at the created state...
        if AssignmentVote.objects.filter(option__poll__pk=poll.pk).exists():
            if poll.published:
                poll.state = 4  # published
            else:
                poll.state = 3  # finished
            poll.save(skip_autoupdate=True)


class Migration(migrations.Migration):

    dependencies = [
        ("assignments", "0008_voting_1"),
    ]

    operations = [
        migrations.RunPython(change_pollmethods),
        migrations.RunPython(set_poll_titles),
        migrations.RunPython(set_onehunderd_percent_bases),
        migrations.RunPython(set_majority_methods),
        migrations.RunPython(convert_votes),
        migrations.RunPython(convert_votesabstain),
        migrations.RunPython(convert_votesno),
        migrations.RunPython(set_correct_state),
    ]
