# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['science_optimization',
 'science_optimization.algorithms',
 'science_optimization.algorithms.cutting_plane',
 'science_optimization.algorithms.decomposition',
 'science_optimization.algorithms.derivative_free',
 'science_optimization.algorithms.lagrange',
 'science_optimization.algorithms.linear_programming',
 'science_optimization.algorithms.search_direction',
 'science_optimization.algorithms.unidimensional',
 'science_optimization.algorithms.utils',
 'science_optimization.builder',
 'science_optimization.examples',
 'science_optimization.function',
 'science_optimization.problems',
 'science_optimization.profiling',
 'science_optimization.solvers',
 'science_optimization.solvers.pareto_samplers',
 'science_optimization.test',
 'src']

package_data = \
{'': ['*']}

install_requires = \
['cycler>=0.11,<1.0',
 'decorator>=5.1,<6.0',
 'joblib>=1.1,<2.0',
 'kiwisolver>=1.3,<2.0',
 'matplotlib>=3.4,<4.0',
 'networkx>=2.6,<3.0',
 'numpy>=1.21,<2.0',
 'ortools>=9.1,<10.0',
 'pandas>=1.3,<2.0',
 'pillow>=8.4,<9.0',
 'protobuf>=3.19,<4.0',
 'pybind11>=2.10.3,<3.0.0',
 'pyparsing>=3.0,<4.0',
 'python-dateutil>=2.8,<3.0',
 'pytz>=2021.3,<2022.0',
 'pywavelets>=1.1,<2.0',
 'scipy>=1.7,<2.0',
 'six>=1.16,<2.0',
 'xlrd>=2.0,<3.0']

setup_kwargs = {
    'name': 'science-optimization',
    'version': '8.0.1',
    'description': 'A framework for optimization.',
    'long_description': '# Optimization Framework \n\nThis is a framework for linear and non-linear optimization developed by ENACOM Group.\n\nIt solves a problem in the form:\n\n```\nmin  f(x)\n\ns.t. g(x) <= 0\n     h(x)  = 0\n     x_min <= x <= x_max\n```\n\nwhere `x` is an n-dimensional vector, `f(x)` maps the n-dimensional space to the o-dimensional space,\nwith `o` being the number of objectives. \n## Prerequisites\n\nWhat you need to be able to run:\n\n* Python 3.6 +\n\nAll requirements are in the file `requirements.txt`.\nTo install all packages at once: \n```shell\n$ pip install -r requirements.txt\n```\n\n## Installing science-optimization\n\nBuilding science-optimization requires the following software installed:\n\n1. Python 3\n    - Make sure that distutils is installed before continuing. For example in Debian GNU/Linux, disutils is included in the python3-dev package.\n2. PyBind11\n    - Note  that this is PyBind11, not PyBind.\n3. GNU C Compiler version >= 5.0\n\nTo install the package in any environment:\n\n- clone this repo\n- `$ pip install ./science-optimization`\n\n## Conventions\n\n### Development\n\n- In Python programming, it is a common practice to initialize "private" methods/attributes with an underscore \n(e.g. _private_method(), _private_attribute). This framework follows this convention\n\n- Attributes sets and gets are implemented using the native \n[`@property`](https://www.programiz.com/python-programming/property) decorator\n\n- An abstract base class inherits from `abc.ABCMeta`\n\n- Google is the default Docstring format\n\n- A package must contain an `__init__.py` file with the package initialization\n\n- PyCharm 2017+ is highly recommended\n\nA detailed explanation of code development can be found [here](https://github.com/LucasSubli/EnacomPythonExample) and \n[here](https://sourcemaking.com/design_patterns).\n\nTutorials on Python programming can be found \n[here](https://www.digitalocean.com/community/tutorial_series/object-oriented-programming-in-python-3).\n\n### Optimization\n\nGiven that `n` is the problem dimension and `m` is the number of points:\n\n- A single point is a `nx1` matrix\n\n- Multiple points are a `nxm` matrix\n\n- Gradient is a `nxm` matrix\n\n- Hessian is a `nxnxm` matrix\n\n## Packages\n\n### Builder\n\nThis package is responsible for building an optimization problem. It was built based on the \n[Builder Design Pattern](https://sourcemaking.com/design_patterns/builder) and it has the following classes.\n\n1. `OptimizationProblem()`: the base class of every optimization problem.\n\n    - Attributes:\n\n        * `objective`: `Objective()` instance\n    \n        * `constraints`: `Contraints()` instance\n    \n        * `variables`: `Variables()` instance\n    \n    - Methods:\n\n        * `_build_problem(builder)`: receives a `BuilderOptimizationProblem()` instance, checks problem\n    consistency and sets attributes\n    \n        * `_check_consistency(variables, objectives, constraints)`: receives attributes and check problem\n    consistency\n    \n        * `info()`: prints to user the assembled problem\n        \n        * `__call__()`: overloaded method that turns an `OptimizationProblem` instance a callable function, that\n        evaluates `f, g, h, df, dg, dh, hf, hg, hh` for a given point `x`\n\n2. `BuilderOptimizationProblem()`: abstract class responsible for defining the builder\'s methods of the \noptimization problem.\n\n    - Abstract methods:\n    \n        * `build_objectives()`: abstract method responsible for building the problem\'s objectives\n        \n        * `build_constraints()`: abstract method responsible for building the problem\'s constraints\n        \n        * `build_variables()`: abstract method responsible for building the problem\'s variables\n\n3. `Objective()`: container class of objectives functions.\n\n    - Attributes:\n    \n        * `objective_functions`: a list with all functions of the problem. Each function\n        is a `FunctionComposite()` instance.\n        \n    - Methods:\n    \n        * `C()`: return the matrix C of linear objectives coefficients\n        \n        * `d()`: return the vector d of linear objectives constants\n\n4. `Contraints()`: container class of constraints functions.\n\n    - Attributes:\n    \n        * `equality_contraints`: a list with equality constraints functions. Each function\n        is a `FunctionComposite()` instance.\n        \n        * `inequality_contraints`: a list with inequality constraints functions. Each function\n        is a `FunctionComposite()` instance.\n        \n    - Methods:\n    \n        * `equality_contraints_feasibility(x)`: evaluates the feasibility of `x` on these constraints\n        \n        * `inequality_contraints_feasibility(x)`: evaluates the feasibility of `x` on these constraints\n        \n        * `A()`: returns the matrix `A` of linear inequality constraints coefficients\n        \n        * `b()`: returns the vector `b` of linear inequality constraints bounds\n        \n        * `Aeq()`: returns the matrix `Aeq` of linear equality constraints coefficients\n        \n        * `beq()`: returns the vector `beq` of linear equality constraints bounds\n\n5. `Variables()`: container class of the problem variables.\n\n    - Attributes:\n        \n        * `x_min`: variables\' lower bounds\n        \n        * `x_max`: variables\' upper bounds\n        \n        * `x_type`: list with variables\' type (\'c\': continuous or \'d\': discrete)\n        \n\n### Function\n\nThis package has the definitions of a function. Is was base on the \n[Composite](https://sourcemaking.com/design_patterns/composite) design pattern and \ncontains the following classes:\n\n1. `BaseFunction()`: base class of every other class inside this package.\n\n    - Attributes:\n    \n        * `parameters`: abstract attribute for functions parameters (e.g. coefficients)\n        \n        * `eps`: precision for numerical calculations\n        \n    - Methods:\n    \n        * `eval(x)`: abstract method for the evaluation of `x`\n        \n        * `gradient(x)`: numerical calculation of the Gradient of `x`\n        \n        * `hessian(x)`: numerical calculation of the Hessian of `x`\n        \n        * `dimension()`: function dimension `n`\n\n2. `FunctionsComposite()`: container class of `BaseFunction()`\'s children.\n\n    - Attributes:\n    \n        * `functions`: a list of functions\n        \n    - Methods:\n    \n        * `eval(x)`: abstract method for the evaluation of `x` in `functions`\n        \n        * `gradient(x)`: numerical calculation of the Gradient of `x` in `functions`\n        \n        * `hessian(x)`: numerical calculation of the Hessian of `x` in `functions`\n        \n        * `add(f)`: adds function `f` to `functions`\n        \n        * `remove(idx)`: removes element `functions[idx]`\n        \n        * `clear()`: removes all functions from list\n        \n         \n3. `LinearFunction()`: implements a function in the form `f(x) = c\'x + d`\n\n    - Attributes:\n    \n        * `parameters`: dictionary with `c` and `d` keys\n        \n    - Methods:\n        \n        * `eval(x)`: method for the evaluation of `x`\n        \n        * `gradient(x)`: analytical calculation of the Gradient of `x`\n        \n        * `hessian(x)`: analytical calculation of the Hessian of `x`    \n\n4. `QuadraticFunction()`: implements a function in the form `f(x) = x\'Qx + c\'x + d`\n\n    - Attributes:\n    \n        * `parameters`: dictionary with `Q`, `c` and `d` keys\n        \n    - Methods:\n        \n        * `eval(x)`: method for the evaluation of `x`\n        \n        * `gradient(x)`: analytical calculation of the Gradient of `x`\n        \n        * `hessian(x)`: analytical calculation of the Hessian of `x`\n\n5. `PolynomialFunction()`: implements a polynomial function\n\n    - Attributes:\n    \n        * `parameters`: dictionary with exponents `e` and `c` coefficients\n        \n    - Methods:\n        \n        * `eval(x)`: method for the evaluation of `x`\n        \n        * `gradient(x)`: analytical calculation of the Gradient of `x`\n        \n        * `hessian(x)`: analytical calculation of the Hessian of `x`\n\n### Problems\n\nThis package is responsible for creating the interface of `BuilderOptimizationProblem()` for each optimization problem\ninstance. The classes must follow the format:\n\n1. `Problem(BuilderOptimizationProblem)`: inherits from BuilderOptimizationProblem\n\n    - Attributes: necessary problem attributes\n    \n    - Methods:\n    \n        * `build_objectives()`: concrete method responsible for building the problem\'s objectives\n        \n        * `build_constraints()`: concrete method responsible for building the problem\'s constraints\n        \n        * `build_variables()`: concrete method responsible for building the problem\'s variables\n\nThe class `Generic(BuilderOptimizationProblem)` builds a generic optimization problem in the canonical\nformat defined at the beginning of this document. However, it also possible to implement customized \noptimization problems inheriting from `BuilderOptimizationProblem` class, such as `Diet` and `KleeMinty`.\n\nFor linear programming, the problem `MIP` is a straightforward abstraction of a problem in the format:\n\n```\n    min  c\'  @ x\n    s.t. A   @ x <= b\n         Aeq @ x == beq\n         x_min  <= x <=  x_max\n```\n\n### Algorithms\n\nThis package contains the implementations of the optimization algorithms. It is organized in sub-packages according\nto algorithm\'s families. Each sub-package contains one abstract implementation of the algorithm\'s family, it is named\n`BaseAlgorithmFamily()`\n\n1. `BaseAlgorithms()`: base class for every `BaseAlgorithmFamily()`\n\n    - Attributes:\n    \n        * `eps`: algorithms\' numerical precision\n        \n    - Methods:\n    \n        * `optimize()`: abstract method for the implementation core\n        \nThis framework currently supports the following optimization algorithms:\n\n* Cutting-plane family:\n    * Ellipsoidal (Multiobjective)\n\n* Decomposition :\n    * Nonlinear dual decomposition\n    \n* Derivative-free :\n    * Nelder-Mead simplex (constrained)\n\n* Lagrange (constrained):\n    * Augmented lagrangian method (using Quasi Newton)\n\n* Linear programming:\n    * Scipy simplex and interior-point\n    * GLOP for LP\n    * CBC for MIP\n\n* Unidimensional search:\n    * Enhanced golden section\n    * Multimodal golden section\n\n* Search direction family (unconstrained):\n    * Gradient method\n    * Modified Newton method\n    * Quasi Newton method\n    \n\n### Solvers\n\nThis package contains the interface to optimization solvers implemented in the framework.\n\n1. `Optimizer()`: optimization for built-in methods, i.e. `algorithms`.\n\n    - Attributes\n    \n        * `optimization_problem`: `OptimizationProblem()` instance\n        \n        * `algorithm`: a concrete algorithm implementation instance (e.g. `GradientAlgorithm()`)\n        \n    - Methods\n    \n        * `solve`: implements `algorithm.optimize()`\n\n2. `OptimizationResults()`: container for optimization results\n\n    - Attributes\n    \n        * `x`: the solution of the optimization\n        \n        * `fx`: function value at solution point\n        \n        * `sucess`: whether or not the solvers exited successfully\n        \n        * `message`: description of the cause of termination\n        \n        * `n_iterations`: number of iterations\n        \n        * `n_function_evaluations`: number of function evaluations\n    \n    - Methods\n    \n        * `info()`: displays optimization results info.\n\n3. `pareto_samplers package`: used to find the pareto border of a multiobjective problem, the implemented\n                              sampling strategies are:\n\n    - Lambda sampler\n    - Epsilon sampler\n    - Nondominated sampler\n    - Mu sampler\n\n### Examples\n\nThis package contains modules implementing examples of building and solving an optimization problem. It can also\nbe used for profiling via `@profiling` decorator.\n\n### Profiling\n\nImplementation of the `@profiling` decorator.\n',
    'author': 'Matheus Mendonça',
    'author_email': 'matheus.mendonca@enacom.com.br',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4',
}
from build import *
build(setup_kwargs)

setup(**setup_kwargs)
