# AUTOGENERATED! DO NOT EDIT! File to edit: 00_Genodata.ipynb (unless otherwise specified).

__all__ = ['read_bgen', 'bgen2dask', 'extract_bed', 'Genodata']

# Cell
import numpy as np
import pandas as pd
import dask.array as da
from bgen_reader import open_bgen
from pandas_plink import read_plink

# Cell
def read_bgen(file, sample_file=None):
    '''the function to read genotype data'''
    bg = open_bgen(file,verbose=False)
    snp,aa0,aa1 = [],[],[]
    for c,p,alleles in zip(bg.chromosomes,bg.positions,bg.allele_ids):
        a0,a1 = alleles.split(',')
        aa0.append(a0)
        aa1.append(a1)
        snp.append(':'.join(['chr'+str(int(c)),str(p),a0,a1]))  # '05' first change to int, then change to str
    bim = pd.DataFrame({'chrom':bg.chromosomes.astype(int),'snp':snp,'pos':bg.positions,'a0':aa0,'a1':aa1})
    if sample_file is not None:
        fam = pd.read_csv(sample_file, header=0, delim_whitespace=True, quotechar='"',skiprows=1)
        fam.columns = ['fid','iid','missing','sex']
        fam = fam
    return bim,fam,bg

# Cell
def bgen2dask(bgen,index,step=500):
    '''The function to covert bgen to dask array'''
    genos = []
    n = len(index)
    for i in range(0,n,step):
        onecode_geno = bgen.read(index[i:min(n,i+step)])  #samples x variants
        geno = onecode_geno.argmax(axis=2).astype(np.int8)
        genos.append(da.from_array(geno))
    return(da.concatenate(genos,axis=1).T)

# Cell
def extract_bed(geno,idx,row=True,step=500):  #row = True by variants, row = False by samples
    if isinstance(geno,da.core.Array):
        if row:
            geno = geno[idx,:]
        else:
            geno = geno[:,idx]
    else:
        if row:
            int_idx = list(idx[idx].index)
            geno = bgen2dask(geno,int_idx,step)
        else:
            geno = geno.read() # read all variants
            geno = geno[:,idx]
    return geno

# Cell
class Genodata:
    def __init__(self,geno_path,sample_path=None):
        self.bim,self.fam,self.bed = self.read_geno(geno_path,sample_path)

    def __repr__(self):
        return "bim:% s \n fam:% s \n bed:%s" % (self.bim, self.fam, self.bed)

    def read_geno(self,geno_file,sample_file):
        if geno_file.endswith('.bed'):
            return read_plink(geno_file[:-4], verbose=False)
        elif geno_file.endswith('.bgen'):
            sample_file = geno_file.replace('.bgen', '.sample')
            return read_bgen(geno_file,sample_file)
        else:
            raise ValueError('Plesae provide the genotype files with PLINK binary format or BGEN format')


    def geno_in_stat(self,stat,notin=False):
        '''The function to find an overlap region between geno data with sumstat'''
        variants = stat.SNP
        self.extractbyvariants(variants,notin)


    def geno_in_unr(self,unr):
        '''The function to find an overlap samples between geno data with unr'''
        samples = unr.IID
        self.extractbysamples(samples)

    def extractbyregion(self,region):
        bim = self.bim
        idx = (bim.chrom == region[0]) & (bim.pos >= region[1]) & (bim.pos <= region[2])
        #update bim,bed
        self.bim = bim[idx]
        self.bed = extract_bed(self.bed,idx)

    def extractbyvariants(self,variants,notin=False):  #variants is list or pd.Series
        idx = self.bim.snp.isin(variants)
        if notin:
            idx = idx == False
        #update bim,bed
        self.bim = self.bim[idx]
        self.bed = extract_bed(self.bed,idx)

    def extractbysamples(self,samples,notin=False): #samples is list or pd.Series
        samples = pd.Series(samples,dtype=str)
        idx = self.fam.iid.astype(str).isin(samples)
        if notin:
            idx = idx == False
        #update fam,bed
        self.fam = self.fam[idx]
        self.bed = extract_bed(self.bed,idx,row=False)