# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ses_log_handler']

package_data = \
{'': ['*']}

install_requires = \
['boto3>=1.0.0,<2.0.0']

setup_kwargs = {
    'name': 'ses-log-handler',
    'version': '1.0.2',
    'description': 'Log messages to email via Amazon SES',
    'long_description': '# SES log handler\n\nLog messages to email via Amazon SES\n\nRate limiting and bouncing are currently not supported.\n\nIf you are using Django then this is also possible using [Django\'s AdminEmailHandler](https://docs.djangoproject.com/en/3.1/topics/logging/#id4) and [django-ses](https://github.com/django-ses/django-ses) which supports rate limiting via AWS_SES_AUTO_THROTTLE.\n\n\n## Quick start\n\n### Installation\n\n```bash\npip install ses-log-handler\n```\n\n\nIf you using IAM roles to get credentials for AWS or have the correct environment variables defined (see [Boto3 configuration guide](https://boto3.amazonaws.com/v1/documentation/api/latest/guide/configuration.html)) then you can simply set a `sender` and `recipients` addresses for the handler:\n\n```python\nimport logging\n\nlogger = logging.getLogger(__name__)\n\n\nses_handler = SESHandler(\n    sender=\'from@example.com\',\n    recipients=[\'to@example.com\'],\n)\nses_handler.setLevel(logging.ERROR)\nlogger.addHandler(ses_handler)\n```\n\nThere is an example of how to configure the log hander using [`dictConfig()`](#dict-config) and [`fileConfig()`](#file-config)\n\n*Note: It is advised you set the log level to either `CRITICAL` or `ERROR`. This will stop the reciver_mails from being spammed by logs and you incuring a large SES bill.*\n\n\n\n## Configuration options\n\nIf you want to explicitly set the access, secret and region this can also be when instantiating the `SESHandler`.\n\nIt also possible to force the subject line to be the same for each email\n\n```python\nmail_handler = SESHandler(\n    sender=\'from@example.com\',\n    recipients=(\'to@example.com\'),\n    access_key=\'<access-key>\',\n    secret_key=\'<secret-key>\',\n    region=\'<region>\',\n    subject=\'Error: Exception raised\',\n)\n```\n\n\n#### sender\n\nRequired: Who the log email should be sent from. The domain should be added and configured as a [verified sender domain](https://console.aws.amazon.com/ses/home?region=us-east-1#verified-senders-domain:)\n\n#### recipients\n\nRequired: A list of recipients who should receive the log emails.\n\nIf your SES account only has "sandbox" access (see [SES dashboard](https://console.aws.amazon.com/ses/home?region=us-east-1#dashboard:)) these email addresses must be added as [verified sender email address](https://console.aws.amazon.com/ses/home?region=us-east-1#verified-senders-email:)\n\n\n#### access_key\n\nOptional: The AWS access key for a user who has access to send emails via SES.\n\nIt is [best practices for managing AWS access keys](https://docs.aws.amazon.com/general/latest/gr/aws-access-keys-best-practices.html) use instance roles instead of passing access keys to your application.\n\nIf the access_key argument is not provided then the SESHandler (which uses boto3) will fall back to getting credentials from either the instance role or environment variables. See the [boto3 credentials guide](https://boto3.amazonaws.com/v1/documentation/api/latest/guide/credentials.html) for more information.\n\n\n#### secret_key\n\nOptional: The AWS secret key for a user who has access to send emails via SES.\n\nIt is [best practices for managing AWS access keys](https://docs.aws.amazon.com/general/latest/gr/aws-access-keys-best-practices.html) use instance roles instead of passing access keys to your application.\n\nIf the access_key argument is not provided then the SESHandler (which uses boto3) will fall back to getting credentials from either the instance role or environment variables. See the [boto3 credentials guide](https://boto3.amazonaws.com/v1/documentation/api/latest/guide/credentials.html) for more information.\n\n#### region\n\nOptional: The AWS region which should be used to send emails via SES\n\nBy default the region that your application is running in will be used\n\n\n#### subject\n\nOptional: Force overrides the subject line to be the same for each email.\n\nThe default is `<log_level>: <message>`\n\n\n\n## Configuration examples\n\nThe below examples both create a logging config which logs everything `ERROR` and above to SES and anything `WARNING` and below to the console.\n\n### Dict config\n\n```python\nimport logging\n\nLOGGING = {\n    \'version\': 1,\n    \'disable_existing_loggers\': False,\n    \'formatters\': {\n        \'general\': {\n            \'format\': \'%(asctime)s %(name)s.%(funcName)s %(levelname)s [%(lineno)d] %(message)s\',  # NOQA: E501\n        }\n    },\n    \'handlers\': {\n        \'console\': {\n            \'class\': \'logging.StreamHandler\',\n            \'formatter\': \'general\',\n            \'level\': \'INFO\',\n        },\n        \'ses\': {\n            \'class\': \'ses_log_handler.SESHandler\',\n            \'formatter\': \'general\',\n            \'level\': \'ERROR\',\n            \'sender\': \'from@example.com\',\n            \'recipients\': [\'to@example.com\'],\n        },\n    },\n    \'root\': {\n        \'level\': \'INFO\',\n        \'handlers\': [\'ses\', \'console\'],\n   }\n}\nlogging.config.dictConfig(LOGGING)\n```\n\n\n### File config\n\n```ini\n[loggers]\nkeys=root\n\n[handlers]\nkeys=sesHandler,consoleHandler\n\n[formatters]\nkeys=generalFormatter\n\n[logger_root]\nlevel=INFO\nhandlers=sesHandler,consoleHandler\n\n[handler_consoleHandler]\nclass=StreamHandler\nlevel=INFO\nformatter=generalFormatter\nargs=(sys.stdout,)\n\n[handler_sesHandler]\nclass=ses_log_handler.SESHandler\nlevel=ERROR\nformatter=generalFormatter\nargs=[\'from@example.com\', [\'to@example.com\']]\n\n[formatter_generalFormatter]\nformat=%(asctime)s %(name)s.%(funcName)s %(levelname)s [%(lineno)d] %(message)s\n```\n\nThen to load this config you can use\n```python\nimport logging\n\nlogging.config.fileConfig(\n    \'logging.conf\',\n    disable_existing_loggers=False\n)\n```\n',
    'author': 'Matt Pye',
    'author_email': 'pyematt@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/pyepye/ses-log-handler',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
