"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToSqs = void 0;
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
const cdk = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayToSqs class.
 */
class ApiGatewayToSqs extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the ApiGatewayToSqs class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the dead letter queue, if applicable
        this.deadLetterQueue = defaults.buildDeadLetterQueue(this, {
            existingQueueObj: props.existingQueueObj,
            deployDeadLetterQueue: props.deployDeadLetterQueue,
            deadLetterQueueProps: props.deadLetterQueueProps,
            maxReceiveCount: props.maxReceiveCount
        });
        // Setup the queue
        [this.sqsQueue] = defaults.buildQueue(this, 'queue', {
            queueProps: props.queueProps,
            deadLetterQueue: this.deadLetterQueue
        });
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, props.apiGatewayProps, props.logGroupProps);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        // Setup the API Gateway resource
        const apiGatewayResource = this.apiGateway.root.addResource('message');
        // Create
        let createRequestTemplate = "Action=SendMessage&MessageBody=$util.urlEncode(\"$input.body\")";
        if (props.createRequestTemplate) {
            createRequestTemplate = props.createRequestTemplate;
        }
        if (props.allowCreateOperation && props.allowCreateOperation === true) {
            this.addActionToPolicy("sqs:SendMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "POST",
                apiResource: this.apiGateway.root,
                requestTemplate: createRequestTemplate,
                contentType: "'application/x-www-form-urlencoded'"
            });
        }
        // Read
        let readRequestTemplate = "Action=ReceiveMessage";
        if (props.readRequestTemplate) {
            readRequestTemplate = props.readRequestTemplate;
        }
        if (props.allowReadOperation === undefined || props.allowReadOperation === true) {
            this.addActionToPolicy("sqs:ReceiveMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "GET",
                apiResource: this.apiGateway.root,
                requestTemplate: readRequestTemplate,
                contentType: "'application/x-www-form-urlencoded'"
            });
        }
        // Delete
        let deleteRequestTemplate = "Action=DeleteMessage&ReceiptHandle=$util.urlEncode($input.params('receiptHandle'))";
        if (props.deleteRequestTemplate) {
            deleteRequestTemplate = props.deleteRequestTemplate;
        }
        if (props.allowDeleteOperation && props.allowDeleteOperation === true) {
            this.addActionToPolicy("sqs:DeleteMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "DELETE",
                apiResource: apiGatewayResource,
                requestTemplate: deleteRequestTemplate,
                contentType: "'application/x-www-form-urlencoded'"
            });
        }
    }
    addActionToPolicy(action) {
        this.apiGatewayRole.addToPolicy(new iam.PolicyStatement({
            resources: [
                this.sqsQueue.queueArn
            ],
            actions: [`${action}`]
        }));
    }
}
exports.ApiGatewayToSqs = ApiGatewayToSqs;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7OztBQUtILHdDQUF3QztBQUN4QywyREFBMkQ7QUFDM0Qsd0NBQTBDO0FBQzFDLHFDQUFxQzs7OztBQTBGckMsTUFBYSxlQUFnQixTQUFRLGdCQUFTOzs7Ozs7Ozs7SUFnQjVDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBMkI7UUFDbkUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQiw2Q0FBNkM7UUFDN0MsSUFBSSxDQUFDLGVBQWUsR0FBRyxRQUFRLENBQUMsb0JBQW9CLENBQUMsSUFBSSxFQUFFO1lBQ3pELGdCQUFnQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7WUFDeEMscUJBQXFCLEVBQUUsS0FBSyxDQUFDLHFCQUFxQjtZQUNsRCxvQkFBb0IsRUFBRSxLQUFLLENBQUMsb0JBQW9CO1lBQ2hELGVBQWUsRUFBRSxLQUFLLENBQUMsZUFBZTtTQUN2QyxDQUFDLENBQUM7UUFFSCxrQkFBa0I7UUFDbEIsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEdBQUcsUUFBUSxDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsT0FBTyxFQUFFO1lBQ25ELFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtZQUM1QixlQUFlLEVBQUUsSUFBSSxDQUFDLGVBQWU7U0FDdEMsQ0FBQyxDQUFDO1FBRUgsd0JBQXdCO1FBQ3hCLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxJQUFJLENBQUMsd0JBQXdCLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLEdBQUcsUUFBUSxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQ3JHLEtBQUssQ0FBQyxlQUFlLEVBQUUsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBRTlDLDZCQUE2QjtRQUM3QixJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7WUFDM0QsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLDBCQUEwQixDQUFDO1NBQ2hFLENBQUMsQ0FBQztRQUVILGlDQUFpQztRQUNqQyxNQUFNLGtCQUFrQixHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUV2RSxTQUFTO1FBQ1QsSUFBSSxxQkFBcUIsR0FBRyxpRUFBaUUsQ0FBQztRQUU5RixJQUFJLEtBQUssQ0FBQyxxQkFBcUIsRUFBRTtZQUMvQixxQkFBcUIsR0FBRyxLQUFLLENBQUMscUJBQXFCLENBQUM7U0FDckQ7UUFFRCxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsSUFBSSxLQUFLLENBQUMsb0JBQW9CLEtBQUssSUFBSSxFQUFFO1lBQ3JFLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1lBQzFDLFFBQVEsQ0FBQywyQkFBMkIsQ0FBQztnQkFDbkMsT0FBTyxFQUFFLEtBQUs7Z0JBQ2QsSUFBSSxFQUFFLEdBQUcsR0FBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUU7Z0JBQ3hELGNBQWMsRUFBRSxJQUFJLENBQUMsY0FBYztnQkFDbkMsU0FBUyxFQUFFLE1BQU07Z0JBQ2pCLFdBQVcsRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUk7Z0JBQ2pDLGVBQWUsRUFBRSxxQkFBcUI7Z0JBQ3RDLFdBQVcsRUFBRSxxQ0FBcUM7YUFDbkQsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxPQUFPO1FBQ1AsSUFBSSxtQkFBbUIsR0FBRyx1QkFBdUIsQ0FBQztRQUVsRCxJQUFJLEtBQUssQ0FBQyxtQkFBbUIsRUFBRTtZQUM3QixtQkFBbUIsR0FBRyxLQUFLLENBQUMsbUJBQW1CLENBQUM7U0FDakQ7UUFFRCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsS0FBSyxTQUFTLElBQUksS0FBSyxDQUFDLGtCQUFrQixLQUFLLElBQUksRUFBRTtZQUMvRSxJQUFJLENBQUMsaUJBQWlCLENBQUMsb0JBQW9CLENBQUMsQ0FBQztZQUM3QyxRQUFRLENBQUMsMkJBQTJCLENBQUM7Z0JBQ25DLE9BQU8sRUFBRSxLQUFLO2dCQUNkLElBQUksRUFBRSxHQUFHLEdBQUcsQ0FBQyxHQUFHLENBQUMsVUFBVSxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFO2dCQUN4RCxjQUFjLEVBQUUsSUFBSSxDQUFDLGNBQWM7Z0JBQ25DLFNBQVMsRUFBRSxLQUFLO2dCQUNoQixXQUFXLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJO2dCQUNqQyxlQUFlLEVBQUUsbUJBQW1CO2dCQUNwQyxXQUFXLEVBQUUscUNBQXFDO2FBQ25ELENBQUMsQ0FBQztTQUNKO1FBRUQsU0FBUztRQUNULElBQUkscUJBQXFCLEdBQUcsb0ZBQW9GLENBQUM7UUFFakgsSUFBSSxLQUFLLENBQUMscUJBQXFCLEVBQUU7WUFDL0IscUJBQXFCLEdBQUcsS0FBSyxDQUFDLHFCQUFxQixDQUFDO1NBQ3JEO1FBRUQsSUFBSSxLQUFLLENBQUMsb0JBQW9CLElBQUksS0FBSyxDQUFDLG9CQUFvQixLQUFLLElBQUksRUFBRTtZQUNyRSxJQUFJLENBQUMsaUJBQWlCLENBQUMsbUJBQW1CLENBQUMsQ0FBQztZQUM1QyxRQUFRLENBQUMsMkJBQTJCLENBQUM7Z0JBQ25DLE9BQU8sRUFBRSxLQUFLO2dCQUNkLElBQUksRUFBRSxHQUFHLEdBQUcsQ0FBQyxHQUFHLENBQUMsVUFBVSxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFO2dCQUN4RCxjQUFjLEVBQUUsSUFBSSxDQUFDLGNBQWM7Z0JBQ25DLFNBQVMsRUFBRSxRQUFRO2dCQUNuQixXQUFXLEVBQUUsa0JBQWtCO2dCQUMvQixlQUFlLEVBQUUscUJBQXFCO2dCQUN0QyxXQUFXLEVBQUUscUNBQXFDO2FBQ25ELENBQUMsQ0FBQztTQUNKO0lBQ0gsQ0FBQztJQUVPLGlCQUFpQixDQUFDLE1BQWM7UUFDdEMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3RELFNBQVMsRUFBRTtnQkFDVCxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVE7YUFDdkI7WUFDRCxPQUFPLEVBQUUsQ0FBRSxHQUFHLE1BQU0sRUFBRSxDQUFFO1NBQ3pCLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQztDQUNGO0FBbEhELDBDQWtIQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCAyMDIxIEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuLy8gSW1wb3J0c1xuaW1wb3J0ICogYXMgYXBpIGZyb20gJ0Bhd3MtY2RrL2F3cy1hcGlnYXRld2F5JztcbmltcG9ydCAqIGFzIHNxcyBmcm9tICdAYXdzLWNkay9hd3Mtc3FzJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGRlZmF1bHRzIGZyb20gJ0Bhd3Mtc29sdXRpb25zLWNvbnN0cnVjdHMvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGxvZ3MgZnJvbSAnQGF3cy1jZGsvYXdzLWxvZ3MnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBBcGlHYXRld2F5VG9TcXNQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhcGlHYXRld2F5UHJvcHM/OiBhcGkuUmVzdEFwaVByb3BzIHwgYW55XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBleGlzdGluZ1F1ZXVlT2JqPzogc3FzLlF1ZXVlLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHF1ZXVlUHJvcHM/OiBzcXMuUXVldWVQcm9wcyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGVwbG95RGVhZExldHRlclF1ZXVlPzogYm9vbGVhbixcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkZWFkTGV0dGVyUXVldWVQcm9wcz86IHNxcy5RdWV1ZVByb3BzLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtYXhSZWNlaXZlQ291bnQ/OiBudW1iZXIsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFsbG93Q3JlYXRlT3BlcmF0aW9uPzogYm9vbGVhbixcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjcmVhdGVSZXF1ZXN0VGVtcGxhdGU/OiBzdHJpbmcsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhbGxvd1JlYWRPcGVyYXRpb24/OiBib29sZWFuLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmVhZFJlcXVlc3RUZW1wbGF0ZT86IHN0cmluZyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhbGxvd0RlbGV0ZU9wZXJhdGlvbj86IGJvb2xlYW5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRlbGV0ZVJlcXVlc3RUZW1wbGF0ZT86IHN0cmluZyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGxvZ0dyb3VwUHJvcHM/OiBsb2dzLkxvZ0dyb3VwUHJvcHNcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIEFwaUdhdGV3YXlUb1NxcyBleHRlbmRzIENvbnN0cnVjdCB7XG4gIHB1YmxpYyByZWFkb25seSBhcGlHYXRld2F5OiBhcGkuUmVzdEFwaTtcbiAgcHVibGljIHJlYWRvbmx5IGFwaUdhdGV3YXlSb2xlOiBpYW0uUm9sZTtcbiAgcHVibGljIHJlYWRvbmx5IGFwaUdhdGV3YXlDbG91ZFdhdGNoUm9sZTogaWFtLlJvbGU7XG4gIHB1YmxpYyByZWFkb25seSBhcGlHYXRld2F5TG9nR3JvdXA6IGxvZ3MuTG9nR3JvdXA7XG4gIHB1YmxpYyByZWFkb25seSBzcXNRdWV1ZTogc3FzLlF1ZXVlO1xuICBwdWJsaWMgcmVhZG9ubHkgZGVhZExldHRlclF1ZXVlPzogc3FzLkRlYWRMZXR0ZXJRdWV1ZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBBcGlHYXRld2F5VG9TcXNQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICAvLyBTZXR1cCB0aGUgZGVhZCBsZXR0ZXIgcXVldWUsIGlmIGFwcGxpY2FibGVcbiAgICB0aGlzLmRlYWRMZXR0ZXJRdWV1ZSA9IGRlZmF1bHRzLmJ1aWxkRGVhZExldHRlclF1ZXVlKHRoaXMsIHtcbiAgICAgIGV4aXN0aW5nUXVldWVPYmo6IHByb3BzLmV4aXN0aW5nUXVldWVPYmosXG4gICAgICBkZXBsb3lEZWFkTGV0dGVyUXVldWU6IHByb3BzLmRlcGxveURlYWRMZXR0ZXJRdWV1ZSxcbiAgICAgIGRlYWRMZXR0ZXJRdWV1ZVByb3BzOiBwcm9wcy5kZWFkTGV0dGVyUXVldWVQcm9wcyxcbiAgICAgIG1heFJlY2VpdmVDb3VudDogcHJvcHMubWF4UmVjZWl2ZUNvdW50XG4gICAgfSk7XG5cbiAgICAvLyBTZXR1cCB0aGUgcXVldWVcbiAgICBbdGhpcy5zcXNRdWV1ZV0gPSBkZWZhdWx0cy5idWlsZFF1ZXVlKHRoaXMsICdxdWV1ZScsIHtcbiAgICAgIHF1ZXVlUHJvcHM6IHByb3BzLnF1ZXVlUHJvcHMsXG4gICAgICBkZWFkTGV0dGVyUXVldWU6IHRoaXMuZGVhZExldHRlclF1ZXVlXG4gICAgfSk7XG5cbiAgICAvLyBTZXR1cCB0aGUgQVBJIEdhdGV3YXlcbiAgICBbdGhpcy5hcGlHYXRld2F5LCB0aGlzLmFwaUdhdGV3YXlDbG91ZFdhdGNoUm9sZSwgdGhpcy5hcGlHYXRld2F5TG9nR3JvdXBdID0gZGVmYXVsdHMuR2xvYmFsUmVzdEFwaSh0aGlzLFxuICAgICAgcHJvcHMuYXBpR2F0ZXdheVByb3BzLCBwcm9wcy5sb2dHcm91cFByb3BzKTtcblxuICAgIC8vIFNldHVwIHRoZSBBUEkgR2F0ZXdheSByb2xlXG4gICAgdGhpcy5hcGlHYXRld2F5Um9sZSA9IG5ldyBpYW0uUm9sZSh0aGlzLCAnYXBpLWdhdGV3YXktcm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdhcGlnYXRld2F5LmFtYXpvbmF3cy5jb20nKVxuICAgIH0pO1xuXG4gICAgLy8gU2V0dXAgdGhlIEFQSSBHYXRld2F5IHJlc291cmNlXG4gICAgY29uc3QgYXBpR2F0ZXdheVJlc291cmNlID0gdGhpcy5hcGlHYXRld2F5LnJvb3QuYWRkUmVzb3VyY2UoJ21lc3NhZ2UnKTtcblxuICAgIC8vIENyZWF0ZVxuICAgIGxldCBjcmVhdGVSZXF1ZXN0VGVtcGxhdGUgPSBcIkFjdGlvbj1TZW5kTWVzc2FnZSZNZXNzYWdlQm9keT0kdXRpbC51cmxFbmNvZGUoXFxcIiRpbnB1dC5ib2R5XFxcIilcIjtcblxuICAgIGlmIChwcm9wcy5jcmVhdGVSZXF1ZXN0VGVtcGxhdGUpIHtcbiAgICAgIGNyZWF0ZVJlcXVlc3RUZW1wbGF0ZSA9IHByb3BzLmNyZWF0ZVJlcXVlc3RUZW1wbGF0ZTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMuYWxsb3dDcmVhdGVPcGVyYXRpb24gJiYgcHJvcHMuYWxsb3dDcmVhdGVPcGVyYXRpb24gPT09IHRydWUpIHtcbiAgICAgIHRoaXMuYWRkQWN0aW9uVG9Qb2xpY3koXCJzcXM6U2VuZE1lc3NhZ2VcIik7XG4gICAgICBkZWZhdWx0cy5hZGRQcm94eU1ldGhvZFRvQXBpUmVzb3VyY2Uoe1xuICAgICAgICBzZXJ2aWNlOiBcInNxc1wiLFxuICAgICAgICBwYXRoOiBgJHtjZGsuQXdzLkFDQ09VTlRfSUR9LyR7dGhpcy5zcXNRdWV1ZS5xdWV1ZU5hbWV9YCxcbiAgICAgICAgYXBpR2F0ZXdheVJvbGU6IHRoaXMuYXBpR2F0ZXdheVJvbGUsXG4gICAgICAgIGFwaU1ldGhvZDogXCJQT1NUXCIsXG4gICAgICAgIGFwaVJlc291cmNlOiB0aGlzLmFwaUdhdGV3YXkucm9vdCxcbiAgICAgICAgcmVxdWVzdFRlbXBsYXRlOiBjcmVhdGVSZXF1ZXN0VGVtcGxhdGUsXG4gICAgICAgIGNvbnRlbnRUeXBlOiBcIidhcHBsaWNhdGlvbi94LXd3dy1mb3JtLXVybGVuY29kZWQnXCJcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIC8vIFJlYWRcbiAgICBsZXQgcmVhZFJlcXVlc3RUZW1wbGF0ZSA9IFwiQWN0aW9uPVJlY2VpdmVNZXNzYWdlXCI7XG5cbiAgICBpZiAocHJvcHMucmVhZFJlcXVlc3RUZW1wbGF0ZSkge1xuICAgICAgcmVhZFJlcXVlc3RUZW1wbGF0ZSA9IHByb3BzLnJlYWRSZXF1ZXN0VGVtcGxhdGU7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmFsbG93UmVhZE9wZXJhdGlvbiA9PT0gdW5kZWZpbmVkIHx8IHByb3BzLmFsbG93UmVhZE9wZXJhdGlvbiA9PT0gdHJ1ZSkge1xuICAgICAgdGhpcy5hZGRBY3Rpb25Ub1BvbGljeShcInNxczpSZWNlaXZlTWVzc2FnZVwiKTtcbiAgICAgIGRlZmF1bHRzLmFkZFByb3h5TWV0aG9kVG9BcGlSZXNvdXJjZSh7XG4gICAgICAgIHNlcnZpY2U6IFwic3FzXCIsXG4gICAgICAgIHBhdGg6IGAke2Nkay5Bd3MuQUNDT1VOVF9JRH0vJHt0aGlzLnNxc1F1ZXVlLnF1ZXVlTmFtZX1gLFxuICAgICAgICBhcGlHYXRld2F5Um9sZTogdGhpcy5hcGlHYXRld2F5Um9sZSxcbiAgICAgICAgYXBpTWV0aG9kOiBcIkdFVFwiLFxuICAgICAgICBhcGlSZXNvdXJjZTogdGhpcy5hcGlHYXRld2F5LnJvb3QsXG4gICAgICAgIHJlcXVlc3RUZW1wbGF0ZTogcmVhZFJlcXVlc3RUZW1wbGF0ZSxcbiAgICAgICAgY29udGVudFR5cGU6IFwiJ2FwcGxpY2F0aW9uL3gtd3d3LWZvcm0tdXJsZW5jb2RlZCdcIlxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgLy8gRGVsZXRlXG4gICAgbGV0IGRlbGV0ZVJlcXVlc3RUZW1wbGF0ZSA9IFwiQWN0aW9uPURlbGV0ZU1lc3NhZ2UmUmVjZWlwdEhhbmRsZT0kdXRpbC51cmxFbmNvZGUoJGlucHV0LnBhcmFtcygncmVjZWlwdEhhbmRsZScpKVwiO1xuXG4gICAgaWYgKHByb3BzLmRlbGV0ZVJlcXVlc3RUZW1wbGF0ZSkge1xuICAgICAgZGVsZXRlUmVxdWVzdFRlbXBsYXRlID0gcHJvcHMuZGVsZXRlUmVxdWVzdFRlbXBsYXRlO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5hbGxvd0RlbGV0ZU9wZXJhdGlvbiAmJiBwcm9wcy5hbGxvd0RlbGV0ZU9wZXJhdGlvbiA9PT0gdHJ1ZSkge1xuICAgICAgdGhpcy5hZGRBY3Rpb25Ub1BvbGljeShcInNxczpEZWxldGVNZXNzYWdlXCIpO1xuICAgICAgZGVmYXVsdHMuYWRkUHJveHlNZXRob2RUb0FwaVJlc291cmNlKHtcbiAgICAgICAgc2VydmljZTogXCJzcXNcIixcbiAgICAgICAgcGF0aDogYCR7Y2RrLkF3cy5BQ0NPVU5UX0lEfS8ke3RoaXMuc3FzUXVldWUucXVldWVOYW1lfWAsXG4gICAgICAgIGFwaUdhdGV3YXlSb2xlOiB0aGlzLmFwaUdhdGV3YXlSb2xlLFxuICAgICAgICBhcGlNZXRob2Q6IFwiREVMRVRFXCIsXG4gICAgICAgIGFwaVJlc291cmNlOiBhcGlHYXRld2F5UmVzb3VyY2UsXG4gICAgICAgIHJlcXVlc3RUZW1wbGF0ZTogZGVsZXRlUmVxdWVzdFRlbXBsYXRlLFxuICAgICAgICBjb250ZW50VHlwZTogXCInYXBwbGljYXRpb24veC13d3ctZm9ybS11cmxlbmNvZGVkJ1wiXG4gICAgICB9KTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGFkZEFjdGlvblRvUG9saWN5KGFjdGlvbjogc3RyaW5nKSB7XG4gICAgdGhpcy5hcGlHYXRld2F5Um9sZS5hZGRUb1BvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgdGhpcy5zcXNRdWV1ZS5xdWV1ZUFyblxuICAgICAgXSxcbiAgICAgIGFjdGlvbnM6IFsgYCR7YWN0aW9ufWAgXVxuICAgIH0pKTtcbiAgfVxufSJdfQ==