'''
stemdiff.detectors
------------------
Known detectors for package stemdiff.

This module is basically a container of classes.
Each class describes a pixelated STEM detector, from which we get datafiles.
The description is not difficult - all we need to define is
detector name, detector size, data type, upscaling coefficient,
and how to read/save datafiles in given detector format.
All these parameters are described below in TimePix detector class.
Therefore, the new classes = new detectors can be added quite easily:

* Copy class describing detector TimePix
* Rename the class as needed, for example: My_new_STEM_detector
* Re-define all properties and methods of the new class as necessary
* When you are done, the new detector can be used within STEMDIFF package
'''

import sys
import inspect
import numpy as np
    
def list_of_known_detectors():
    '''
    Get a list of known detectors = classes defined in stemdiff.detectors.

    Returns
    -------
    detectors : list
        List of known detectors = classes defined in stemdiff.detectors module.
    '''
    # Prepare list of known detectors
    detectors = []
    # Get names of all classes in current module
    # Based on stackoveflow: https://stackoverflow.com/q/1796180
    for name, obj in inspect.getmembers(sys.modules[__name__]):
        if inspect.isclass(obj):
            detectors.append(obj)
    # Return list of known detectors
    return(detectors)

def print_known_detectors():
    '''
    Print a list of known detectors = classes defined in stemdiff.detectors.

    Returns
    -------
    Nothing
        The list of detectors is just printed on the screen.
    '''
    detectors = list_of_known_detectors()
    print('List of knonw detectors = classes defined in stemdiff.detectors:')
    for detector in detectors:
        print(detector)

class TimePix:
    '''
    Definition of TimePix detector.
    
    Parameters
    ----------
    detector_name : str, default is 'TimePix'
        Name of the detector.
        Keep the default unless you have specific reasons.
    detector_size : integer, default is 256
        Size of the detector in pixels.
        Keep the default unless you have specific reasons.
    data_type : numpy data type, optional, default is np.uint16
        Type of data, which are saved in the binary file.
        TimePix detector saves the data as 16-bit integers.
        This corresponds to np.uint16 (more info in NumPy documentation).
    upscale : integer, default is 4
        Upscaling coefficient.
        Final image size = detector_size * upscale.
        The upscaling coefficient increases the detector resolution.
        Surprisingly enough, the upscaling helps to improve final resolution.
    
    Returns
    -------
    TimePix detector object.
    '''
    
    def __init__(self,
                 detector_name='TimePix', 
                 detector_size=256, data_type=np.uint16, upscale=4):
        '''
        Initialize parameters of TimePix detector.
        The parameters are described above in class definition.
        '''
        self.detector_name = detector_name
        self.detector_size = detector_size
        self.data_type = data_type
        self.upscale = upscale
    
    def read_datafile(self, filename, arr_size=None):
        '''
        Read datafile in TimePix detector format.

        Parameters
        ----------
        filename : str or path
            Name of the datafile to read.
        arr_size : int, optional, default is None
            Size of the square array to reade.
            Typically, we read original datafiles with size = detector.size.
            Nonetheless, we can read saved also datafiles with size = arr_size.

        Returns
        -------
        arr : 2D-numpy array
            2D-array containing image from TimePix detector.
            Each element of the array = the intensity detected at given pixel.
        '''
        # Read binary datafile (to 1D-array)
        arr = np.fromfile(filename, dtype=self.data_type)
        # Reshape the 1D-array to 2D-square array
        if arr_size:
            # If arr_size was given,
            # the datafile must have been saved with cut edges
            # and, as a result, we read a smaller array with array_size.
            arr = arr.reshape(self.detector_size, self.detector_size)
        else:
            # If arr_size was not given,
            # the datafile must be in its original format
            # and, as a result, we read it with default detector.size.
            arr = arr.reshape(self.detector_size, self.detector_size)
        # Return the final square array
        return(arr)
    
    def save_datafile(self, filename):
        '''
        Save datafile in TimePix detector format.
        '''
        # TODO
        # Save the datafile to a binary file
        # Flatten array to 1D, save as binary file with given format.
        pass
