"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateMachine = exports.LogLevel = exports.StateMachineType = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const state_graph_1 = require("./state-graph");
const stepfunctions_generated_1 = require("./stepfunctions.generated");
/**
 * Two types of state machines are available in AWS Step Functions: EXPRESS AND STANDARD.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/concepts-standard-vs-express.html
 *
 * @default STANDARD
 */
var StateMachineType;
(function (StateMachineType) {
    /**
     * Express Workflows are ideal for high-volume, event processing workloads.
     */
    StateMachineType["EXPRESS"] = "EXPRESS";
    /**
     * Standard Workflows are ideal for long-running, durable, and auditable workflows.
     */
    StateMachineType["STANDARD"] = "STANDARD";
})(StateMachineType = exports.StateMachineType || (exports.StateMachineType = {}));
/**
 * Defines which category of execution history events are logged.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/cloudwatch-log-level.html
 *
 * @default ERROR
 */
var LogLevel;
(function (LogLevel) {
    /**
     * No Logging
     */
    LogLevel["OFF"] = "OFF";
    /**
     * Log everything
     */
    LogLevel["ALL"] = "ALL";
    /**
     * Log all errors
     */
    LogLevel["ERROR"] = "ERROR";
    /**
     * Log fatal errors
     */
    LogLevel["FATAL"] = "FATAL";
})(LogLevel = exports.LogLevel || (exports.LogLevel = {}));
/**
 * A new or imported state machine.
 */
class StateMachineBase extends core_1.Resource {
    /**
     * Import a state machine
     */
    static fromStateMachineArn(scope, id, stateMachineArn) {
        class Import extends StateMachineBase {
            constructor() {
                super(...arguments);
                this.stateMachineArn = stateMachineArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Grant the given identity permissions to start an execution of this state
     * machine.
     */
    grantStartExecution(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: ['states:StartExecution'],
            resourceArns: [this.stateMachineArn],
        });
    }
}
/**
 * Define a StepFunctions State Machine
 */
class StateMachine extends StateMachineBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.stateMachineName,
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('states.amazonaws.com'),
        });
        const graph = new state_graph_1.StateGraph(props.definition.startState, `State Machine ${id} definition`);
        graph.timeout = props.timeout;
        this.stateMachineType = props.stateMachineType ? props.stateMachineType : StateMachineType.STANDARD;
        let loggingConfiguration;
        if (props.logs) {
            const conf = props.logs;
            loggingConfiguration = {
                destinations: [{ cloudWatchLogsLogGroup: { logGroupArn: conf.destination.logGroupArn } }],
                includeExecutionData: conf.includeExecutionData,
                level: conf.level || 'ERROR',
            };
            // https://docs.aws.amazon.com/step-functions/latest/dg/cw-logs.html#cloudwatch-iam-policy
            this.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: [
                    'logs:CreateLogDelivery',
                    'logs:GetLogDelivery',
                    'logs:UpdateLogDelivery',
                    'logs:DeleteLogDelivery',
                    'logs:ListLogDeliveries',
                    'logs:PutResourcePolicy',
                    'logs:DescribeResourcePolicies',
                    'logs:DescribeLogGroups',
                ],
                resources: ['*'],
            }));
        }
        const resource = new stepfunctions_generated_1.CfnStateMachine(this, 'Resource', {
            stateMachineName: this.physicalName,
            stateMachineType: props.stateMachineType ? props.stateMachineType : undefined,
            roleArn: this.role.roleArn,
            definitionString: core_1.Stack.of(this).toJsonString(graph.toGraphJson()),
            loggingConfiguration,
        });
        resource.node.addDependency(this.role);
        for (const statement of graph.policyStatements) {
            this.addToRolePolicy(statement);
        }
        this.stateMachineName = this.getResourceNameAttribute(resource.attrName);
        this.stateMachineArn = this.getResourceArnAttribute(resource.ref, {
            service: 'states',
            resource: 'stateMachine',
            resourceName: this.physicalName,
            sep: ':',
        });
    }
    /**
     * Add the given statement to the role's policy
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Return the given named metric for this State Machine's executions
     *
     * @default sum over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensions: { StateMachineArn: this.stateMachineArn },
            statistic: 'sum',
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the number of executions that failed
     *
     * @default sum over 5 minutes
     */
    metricFailed(props) {
        return this.metric('ExecutionsFailed', props);
    }
    /**
     * Metric for the number of executions that were throttled
     *
     * @default sum over 5 minutes
     */
    metricThrottled(props) {
        return this.metric('ExecutionThrottled', props);
    }
    /**
     * Metric for the number of executions that were aborted
     *
     * @default sum over 5 minutes
     */
    metricAborted(props) {
        return this.metric('ExecutionsAborted', props);
    }
    /**
     * Metric for the number of executions that succeeded
     *
     * @default sum over 5 minutes
     */
    metricSucceeded(props) {
        return this.metric('ExecutionsSucceeded', props);
    }
    /**
     * Metric for the number of executions that succeeded
     *
     * @default sum over 5 minutes
     */
    metricTimedOut(props) {
        return this.metric('ExecutionsTimedOut', props);
    }
    /**
     * Metric for the number of executions that were started
     *
     * @default sum over 5 minutes
     */
    metricStarted(props) {
        return this.metric('ExecutionsStarted', props);
    }
    /**
     * Metric for the interval, in milliseconds, between the time the execution starts and the time it closes
     *
     * @default sum over 5 minutes
     */
    metricTime(props) {
        return this.metric('ExecutionTime', props);
    }
}
exports.StateMachine = StateMachine;
//# sourceMappingURL=data:application/json;base64,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