"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InputType = exports.TaskInput = void 0;
const fields_1 = require("./fields");
/**
 * Type union for task classes that accept multiple types of payload
 */
class TaskInput {
    /**
     *
     * @param type type of task input
     * @param value payload for the corresponding input type.
     * It can be a JSON-encoded object, context, data, etc.
     */
    constructor(type, value) {
        this.type = type;
        this.value = value;
    }
    /**
     * Use a literal string as task input
     *
     * This might be a JSON-encoded object, or just a text.
     */
    static fromText(text) {
        return new TaskInput(InputType.TEXT, text);
    }
    /**
     * Use an object as task input
     *
     * This object may contain Data and Context fields
     * as object values, if desired.
     */
    static fromObject(obj) {
        return new TaskInput(InputType.OBJECT, obj);
    }
    /**
     * Use a part of the execution data as task input
     *
     * Use this when you want to use a subobject or string from
     * the current state machine execution as complete payload
     * to a task.
     */
    static fromDataAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.Data.stringAt(path));
    }
    /**
     * Use a part of the task context as task input
     *
     * Use this when you want to use a subobject or string from
     * the current task context as complete payload
     * to a task.
     */
    static fromContextAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.Context.stringAt(path));
    }
}
exports.TaskInput = TaskInput;
/**
 * The type of task input
 */
var InputType;
(function (InputType) {
    /**
     * Use a literal string
     * This might be a JSON-encoded object, or just text.
     * valid JSON text: standalone, quote-delimited strings; objects; arrays; numbers; Boolean values; and null.
     *
     * example: `literal string`
     * example: {"json": "encoded"}
     */
    InputType[InputType["TEXT"] = 0] = "TEXT";
    /**
     * Use an object which may contain Data and Context fields
     * as object values, if desired.
     *
     * example:
     * {
     *  literal: 'literal',
     *  SomeInput: sfn.Data.stringAt('$.someField')
     * }
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-data.html
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/input-output-contextobject.html
     */
    InputType[InputType["OBJECT"] = 1] = "OBJECT";
})(InputType = exports.InputType || (exports.InputType = {}));
//# sourceMappingURL=data:application/json;base64,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