import itertools
import logging
import re

from typing import TYPE_CHECKING, Optional, List, Dict, Any, Tuple, Set
from collections import namedtuple
from os import environ
from time import perf_counter as time

from jinja2 import Template, Environment
from jinja2.meta import find_undeclared_variables
from locust.exception import StopUser

from ..tasks import GrizzlyTask
from ..types import RequestType, TestdataType, GrizzlyVariableType
from ..utils import merge_dicts
from .ast import get_template_variables
from . import GrizzlyVariables


if TYPE_CHECKING:
    from ..context import GrizzlyContext


logger = logging.getLogger(__name__)


def initialize_testdata(grizzly: 'GrizzlyContext', tasks: List[GrizzlyTask]) -> Tuple[TestdataType, Set[str]]:
    testdata: TestdataType = {}
    template_variables = get_template_variables(tasks)

    found_variables = set()
    for variable in itertools.chain(*template_variables.values()):
        module_name, variable_type, variable_name = GrizzlyVariables.get_variable_spec(variable)

        if module_name is None and variable_type is None:
            found_variables.add(variable_name)
        else:
            prefix = f'{module_name}.' if module_name != 'grizzly.testdata.variables' else ''
            found_variables.add(f'{prefix}{variable_type}.{variable_name}')

    declared_variables = set(grizzly.state.variables.keys())

    # check except between declared variables and variables found in templates
    missing_in_templates = []
    for variable in declared_variables:
        if variable not in found_variables:
            missing_in_templates.append(variable)
    assert len(missing_in_templates) == 0, f'variables has been declared, but cannot be found in templates: {",".join(missing_in_templates)}'

    missing_declarations = []
    for variable in found_variables:
        if variable not in declared_variables:
            missing_declarations.append(variable)
    assert len(missing_declarations) == 0, f'variables has been found in templates, but have not been declared: {",".join(missing_declarations)}'

    initialized_datatypes: Dict[str, Any] = {}
    external_dependencies: Set[str] = set()

    for scenario, variables in template_variables.items():
        testdata[scenario] = {}

        for variable in variables:
            module_name, variable_type, variable_name = GrizzlyVariables.get_variable_spec(variable)
            if module_name is not None and variable_type is not None:
                variable_datatype = f'{variable_type}.{variable_name}'
                if module_name != 'grizzly.testdata.variables':
                    variable_datatype = f'{module_name}.{variable_datatype}'
            else:
                variable_datatype = variable_name

            if variable_datatype not in initialized_datatypes:
                initialized_datatypes[variable_datatype], dependencies = GrizzlyVariables.get_variable_value(grizzly, variable_datatype)
                external_dependencies.update(dependencies)

            testdata[scenario][variable] = initialized_datatypes[variable_datatype]

    return testdata, external_dependencies


def transform(grizzly: 'GrizzlyContext', data: Dict[str, Any], objectify: Optional[bool] = True) -> Dict[str, Any]:
    testdata: Dict[str, Any] = {}

    for key, value in data.items():
        module_name, variable_type, variable_name = GrizzlyVariables.get_variable_spec(key)

        if '.' in key:
            if module_name is not None and variable_type is not None:
                if value == '__on_consumer__':
                    variable_type_instance = GrizzlyVariables.load_variable(module_name, variable_type)
                    initial_value = grizzly.state.variables.get(key, None)
                    variable_instance = variable_type_instance.obtain(variable_name, initial_value)

                    start_time = time()
                    exception: Optional[Exception] = None
                    try:
                        value = variable_instance[variable_name]
                    except Exception as e:
                        exception = e
                        logger.error(str(e), exc_info=grizzly.state.verbose)
                    finally:
                        response_time = int((time() - start_time) * 1000)
                        grizzly.state.locust.environment.events.request.fire(
                            request_type=RequestType.VARIABLE(),
                            name=key,
                            response_time=response_time,
                            response_length=0,
                            context=None,
                            exception=exception,
                        )

                    if exception is not None:
                        raise StopUser()

            paths: List[str] = key.split('.')
            variable = paths.pop(0)
            path = paths.pop()
            struct = {path: value}
            paths.reverse()

            for path in paths:
                struct = {path: {**struct}}

            if variable in testdata:
                testdata[variable] = merge_dicts(testdata[variable], struct)
            else:
                testdata[variable] = {**struct}
        else:
            testdata[key] = value

    if objectify:
        return _objectify(testdata)
    else:
        return testdata


def _objectify(testdata: Dict[str, Any]) -> Dict[str, Any]:
    for variable, attributes in testdata.items():
        if not isinstance(attributes, dict):
            continue

        attributes = _objectify(attributes)
        testdata[variable] = namedtuple('Testdata', attributes.keys())(**attributes)

    return testdata


def create_context_variable(grizzly: 'GrizzlyContext', variable: str, value: str) -> Dict[str, Any]:
    if '{{' in value and '}}' in value:
        grizzly.scenario.orphan_templates.append(value)

    casted_value = resolve_variable(grizzly, value)

    variable = variable.lower().replace(' ', '_').replace('/', '.')

    return transform(grizzly, {variable: casted_value}, objectify=False)


def resolve_variable(grizzly: 'GrizzlyContext', value: str, guess_datatype: Optional[bool] = True, only_grizzly: bool = False) -> GrizzlyVariableType:
    if len(value) < 1:
        return value

    quote_char: Optional[str] = None
    if value[0] in ['"', "'"] and value[0] == value[-1]:
        quote_char = value[0]
        value = value[1:-1]

    resolved_variable: GrizzlyVariableType
    if '{{' in value and '}}' in value and not only_grizzly:
        template = Template(value)
        j2env = Environment(autoescape=False)
        template_parsed = j2env.parse(value)
        template_variables = find_undeclared_variables(template_parsed)

        for template_variable in template_variables:
            assert template_variable in grizzly.state.variables, f'value contained variable "{template_variable}" which has not been set'

        resolved_variable = template.render(**grizzly.state.variables)
    elif '$conf' in value or '$env' in value:
        regex = r"\$(conf|env)::([^\$]+)\$"

        matches = re.finditer(regex, value, re.MULTILINE)

        if matches:
            resolved_variable = value

        for match in matches:
            match_type = match.group(1)
            variable_name = match.group(2)

            if match_type == 'conf':
                assert variable_name in grizzly.state.configuration, f'configuration variable "{variable_name}" is not set'
                variable_value = grizzly.state.configuration[variable_name]
            elif match_type == 'env':
                assert variable_name in environ, f'environment variable "{variable_name}" is not set'
                variable_value = environ.get(variable_name, None)

            if not isinstance(variable_value, str):
                variable_value = str(variable_value)

            value = value.replace(f'${match_type}::{variable_name}$', variable_value)

        resolved_variable = value

        if len(value) > 4 and value[0] == '$' and value[1] != '.':  # $. is jsonpath expression...
            raise ValueError(f'"{value}" is not a correctly specified templating variable, variables must match "$(conf|env)::<variable name>$"')
    else:
        resolved_variable = value

    if guess_datatype:
        resolved_variable = GrizzlyVariables.guess_datatype(resolved_variable)
    elif quote_char is not None and isinstance(resolved_variable, str) and resolved_variable.count(' ') > 0:
        resolved_variable = f'{quote_char}{resolved_variable}{quote_char}'

    return resolved_variable
