import requests
from flyers import logger, strs, assets


class SoundType(object):
    ALARM = 'alarm'
    ANTICIPATE = 'anticipate'
    BELL = 'bell'
    BIRDSONG = 'birdsong'
    BLOOM = 'bloom'
    CALYPSO = 'calypso'
    CHIME = 'chime'
    CHOO = 'choo'
    DESCENT = 'descent'
    ELECTRONIC = 'electronic'
    FANFARE = 'fanfare'
    GLASS = 'glass'
    GOTOSLEEP = 'gotosleep'
    HEALTHNOTIFICATION = 'healthnotification'
    HORN = 'horn'
    LADDER = 'ladder'
    MAILSEND = 'mailsend'
    MINUET = 'minuet'
    MULTIWAYINVITATION = 'multiwayinvitation'
    NEWMAIL = 'newmail'
    NEWSFLASH = 'newsflash'
    NOIR = 'noir'
    PAYMENTSUCCESS = 'paymentsuccess'
    SHAKE = 'shake'
    SHERWOODFOREST = 'sherwoodforest'
    SPELL = 'spell'
    SUSPENSE = 'suspense'
    TELEGRAPH = 'telegraph'
    TIPTOES = 'tiptoes'
    TYPEWRITERS = 'typewriters'
    UPDATE = 'update'


class LevelType(object):
    ACTIVE = "active"
    TIME_SENSITIVE = "timeSensitive"
    PASSIVE = "PASSIVE"


class PushOptions(object):
    def __init__(self, sound: str = None,
                 level_type: str = None, badge: int = 1,
                 is_archive: bool = False, automatically_copy: bool = False):
        self.sound = sound
        self.level_type = level_type
        self.badge = badge
        self.is_archive = is_archive
        self.automatically_copy = automatically_copy


class PushMessage(object):
    def __init__(self, content: str, title: str = None, icon: str = None, group: str = None,
                 url: str = None, copy_content: str = None, options: PushOptions = None):
        self.title = title
        self.content = content
        self.icon = icon
        self.group = group
        self.url = url
        self.copy_content = copy_content
        self.options = options

    def __str__(self):
        return str(self.__dict__)


class BarkClient(object):

    def __init__(self, domain: str, base_receivers: list[str] = None):
        self.domain: str = domain
        self.base_receivers: list[str] = base_receivers or []

    def batch_push(self, message: PushMessage, receivers: list[str] = None) -> []:
        all_receivers = self.base_receivers + (receivers or [])
        if len(all_receivers) == 0:
            logger.warning('The receivers is empty, message = {}'.format(message))
            return []

        result = []
        for r in all_receivers:
            succeed, cause = self.push(message, r)
            if not succeed:
                result.append({
                    'key': r,
                    'cause': cause
                })
        return result

    def push(self, message: PushMessage, receiver: str) -> (bool, str):
        assets.check_null(message, 'message')
        assets.check_blank(receiver, 'receiver')

        request_url = 'https://{domain}/{key}'.format(domain=self.domain, key=receiver)
        if not strs.is_blank(message.title):
            request_url += '/{title}'.format(title=message.title)
        request_url += '/{content}'.format(content=message.content)

        params = {
            'icon': message.icon,
            'group': message.group,
            'url': message.url,
            'copy': message.copy_content
        }
        if message.options:
            params['sound'] = message.options.sound
            params['level'] = message.options.level_type
            params['badge'] = message.options.badge
            params['isArchive'] = int(message.options.is_archive or False)
            params['automaticallyCopy'] = int(message.options.automatically_copy or False)

        resp = requests.get(request_url, params=params)
        logger.info('Request to {}, response: \n'.format(request_url))
        print(resp.text)

        data = resp.json()
        if data['code'] != 200:
            return False, data['message']
        return True, None
