"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubActionRole = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
/**
 * Creates or references a GitHub OIDC provider and accompanying role that trusts the provider.
 *
 * This role can be used to authenticate against AWS instead of using long-lived AWS user credentials
 * stored in GitHub secrets.
 *
 * You can do this manually in the console, or create a separate stack that uses this construct.
 * You must `cdk deploy` once (with your normal AWS credentials) to have this role created for you.
 *
 * You can then make note of the role arn in the stack output and send it into the Github Workflow app via
 * the `gitHubActionRoleArn` property. The role arn will be `arn:aws:iam::<accountId>:role/GithubActionRole`.
 *
 * @see https://docs.github.com/en/actions/deployment/security-hardening-your-deployments/configuring-openid-connect-in-amazon-web-services
 * @stability stable
 */
class GitHubActionRole extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        const rawEndpoint = 'token.actions.githubusercontent.com';
        const providerUrl = `https://${rawEndpoint}`;
        // uses the given provider or creates a new one.
        const provider = (_b = props.provider) !== null && _b !== void 0 ? _b : new iam.OpenIdConnectProvider(this, 'github-provider', {
            url: providerUrl,
            clientIds: ['sts.amazonaws.com'],
        });
        // create a role that references the provider as a trusted entity
        const principal = new iam.FederatedPrincipal(provider.openIdConnectProviderArn, {
            StringLike: {
                [`${rawEndpoint}:sub`]: formatRepos(props.repos),
            },
        }, 'sts:AssumeRoleWithWebIdentity');
        // permit this role from assuming all of the CDK bootstrap roles
        const oidcPolicyStatement = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['sts:AssumeRole'],
            resources: ['*'],
            conditions: {
                'ForAnyValue:StringEquals': {
                    'iam:ResourceTag/aws-cdk:bootstrap-role': [
                        'deploy',
                        'lookup',
                        'file-publishing',
                        'image-publishing',
                    ],
                },
            },
        });
        // permit this role from accessing ecr repositories for docker assets
        const ecrPolicyStatement = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['ecr:GetAuthorizationToken'],
            resources: ['*'],
        });
        this.role = new iam.Role(this, 'github-action-role', {
            roleName: (_c = props.roleName) !== null && _c !== void 0 ? _c : 'GitHubActionRole',
            assumedBy: principal,
            inlinePolicies: {
                AssumeBootstrapRoles: new iam.PolicyDocument({
                    statements: [oidcPolicyStatement, ecrPolicyStatement],
                }),
            },
        });
        // show the role arn in the stack output
        new aws_cdk_lib_1.CfnOutput(this, 'roleArn', {
            value: this.role.roleArn,
        });
    }
    /**
     * Reference an existing GitHub Actions provider.
     *
     * You do not need to pass in an arn because the arn for such
     * a provider is always the same.
     *
     * @stability stable
     */
    static existingGitHubActionsProvider(scope) {
        return iam.OpenIdConnectProvider.fromOpenIdConnectProviderArn(scope, 'GitHubActionProvider', `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:oidc-provider/token.actions.githubusercontent.com`);
    }
}
exports.GitHubActionRole = GitHubActionRole;
_a = JSII_RTTI_SYMBOL_1;
GitHubActionRole[_a] = { fqn: "cdk-pipelines-github.GitHubActionRole", version: "0.1.3" };
function formatRepos(repos) {
    const formattedRepos = [];
    for (const repo of repos) {
        formattedRepos.push(`repo:${repo}:*`);
    }
    return formattedRepos;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib2lkYy1wcm92aWRlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9vaWRjLXByb3ZpZGVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsNkNBQTZDO0FBQzdDLDJDQUEyQztBQUMzQywyQ0FBdUM7Ozs7Ozs7Ozs7Ozs7Ozs7QUFldkMsTUFBYSxnQkFBaUIsU0FBUSxzQkFBUzs7OztJQWE3QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTRCOztRQUNwRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sV0FBVyxHQUFHLHFDQUFxQyxDQUFDO1FBQzFELE1BQU0sV0FBVyxHQUFHLFdBQVcsV0FBVyxFQUFFLENBQUM7UUFFN0MsZ0RBQWdEO1FBQ2hELE1BQU0sUUFBUSxTQUFHLEtBQUssQ0FBQyxRQUFRLG1DQUFJLElBQUksR0FBRyxDQUFDLHFCQUFxQixDQUFDLElBQUksRUFBRSxpQkFBaUIsRUFBRTtZQUN4RixHQUFHLEVBQUUsV0FBVztZQUNoQixTQUFTLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQztTQUNqQyxDQUFDLENBQUM7UUFFSCxpRUFBaUU7UUFDakUsTUFBTSxTQUFTLEdBQUcsSUFBSSxHQUFHLENBQUMsa0JBQWtCLENBQzFDLFFBQVEsQ0FBQyx3QkFBd0IsRUFDakM7WUFDRSxVQUFVLEVBQUU7Z0JBQ1YsQ0FBQyxHQUFHLFdBQVcsTUFBTSxDQUFDLEVBQUUsV0FBVyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUM7YUFDakQ7U0FDRixFQUNELCtCQUErQixDQUNoQyxDQUFDO1FBRUYsZ0VBQWdFO1FBQ2hFLE1BQU0sbUJBQW1CLEdBQUcsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ2xELE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFLENBQUMsZ0JBQWdCLENBQUM7WUFDM0IsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO1lBQ2hCLFVBQVUsRUFBRTtnQkFDViwwQkFBMEIsRUFBRTtvQkFDMUIsd0NBQXdDLEVBQUU7d0JBQ3hDLFFBQVE7d0JBQ1IsUUFBUTt3QkFDUixpQkFBaUI7d0JBQ2pCLGtCQUFrQjtxQkFDbkI7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILHFFQUFxRTtRQUNyRSxNQUFNLGtCQUFrQixHQUFHLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNqRCxNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRSxDQUFDLDJCQUEyQixDQUFDO1lBQ3RDLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztTQUNqQixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsb0JBQW9CLEVBQUU7WUFDbkQsUUFBUSxRQUFFLEtBQUssQ0FBQyxRQUFRLG1DQUFJLGtCQUFrQjtZQUM5QyxTQUFTLEVBQUUsU0FBUztZQUNwQixjQUFjLEVBQUU7Z0JBQ2Qsb0JBQW9CLEVBQUUsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDO29CQUMzQyxVQUFVLEVBQUUsQ0FBQyxtQkFBbUIsRUFBRSxrQkFBa0IsQ0FBQztpQkFDdEQsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsd0NBQXdDO1FBQ3hDLElBQUksdUJBQVMsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQzdCLEtBQUssRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU87U0FDekIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7Ozs7Ozs7O0lBeEVNLE1BQU0sQ0FBQyw2QkFBNkIsQ0FBQyxLQUFnQjtRQUMxRCxPQUFPLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyw0QkFBNEIsQ0FDM0QsS0FBSyxFQUNMLHNCQUFzQixFQUN0QixnQkFBZ0IsaUJBQUcsQ0FBQyxVQUFVLG9EQUFvRCxDQUNuRixDQUFDO0lBQ0osQ0FBQzs7QUFSSCw0Q0EyRUM7OztBQUVELFNBQVMsV0FBVyxDQUFDLEtBQWU7SUFDbEMsTUFBTSxjQUFjLEdBQUcsRUFBRSxDQUFDO0lBQzFCLEtBQUssTUFBTSxJQUFJLElBQUksS0FBSyxFQUFFO1FBQ3hCLGNBQWMsQ0FBQyxJQUFJLENBQUMsUUFBUSxJQUFJLElBQUksQ0FBQyxDQUFDO0tBQ3ZDO0lBQ0QsT0FBTyxjQUFjLENBQUM7QUFDeEIsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEF3cywgQ2ZuT3V0cHV0IH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ2F3cy1jZGstbGliL2F3cy1pYW0nO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBHaXRIdWJBY3Rpb25Sb2xlUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJlcG9zOiBzdHJpbmdbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJvbGVOYW1lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcHJvdmlkZXI/OiBpYW0uSU9wZW5JZENvbm5lY3RQcm92aWRlcjtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBHaXRIdWJBY3Rpb25Sb2xlIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBleGlzdGluZ0dpdEh1YkFjdGlvbnNQcm92aWRlcihzY29wZTogQ29uc3RydWN0KTogaWFtLklPcGVuSWRDb25uZWN0UHJvdmlkZXIge1xuICAgIHJldHVybiBpYW0uT3BlbklkQ29ubmVjdFByb3ZpZGVyLmZyb21PcGVuSWRDb25uZWN0UHJvdmlkZXJBcm4oXG4gICAgICBzY29wZSxcbiAgICAgICdHaXRIdWJBY3Rpb25Qcm92aWRlcicsXG4gICAgICBgYXJuOmF3czppYW06OiR7QXdzLkFDQ09VTlRfSUR9Om9pZGMtcHJvdmlkZXIvdG9rZW4uYWN0aW9ucy5naXRodWJ1c2VyY29udGVudC5jb21gLFxuICAgICk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSByb2xlOiBpYW0uSVJvbGU7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEdpdEh1YkFjdGlvblJvbGVQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCByYXdFbmRwb2ludCA9ICd0b2tlbi5hY3Rpb25zLmdpdGh1YnVzZXJjb250ZW50LmNvbSc7XG4gICAgY29uc3QgcHJvdmlkZXJVcmwgPSBgaHR0cHM6Ly8ke3Jhd0VuZHBvaW50fWA7XG5cbiAgICAvLyB1c2VzIHRoZSBnaXZlbiBwcm92aWRlciBvciBjcmVhdGVzIGEgbmV3IG9uZS5cbiAgICBjb25zdCBwcm92aWRlciA9IHByb3BzLnByb3ZpZGVyID8/IG5ldyBpYW0uT3BlbklkQ29ubmVjdFByb3ZpZGVyKHRoaXMsICdnaXRodWItcHJvdmlkZXInLCB7XG4gICAgICB1cmw6IHByb3ZpZGVyVXJsLFxuICAgICAgY2xpZW50SWRzOiBbJ3N0cy5hbWF6b25hd3MuY29tJ10sXG4gICAgfSk7XG5cbiAgICAvLyBjcmVhdGUgYSByb2xlIHRoYXQgcmVmZXJlbmNlcyB0aGUgcHJvdmlkZXIgYXMgYSB0cnVzdGVkIGVudGl0eVxuICAgIGNvbnN0IHByaW5jaXBhbCA9IG5ldyBpYW0uRmVkZXJhdGVkUHJpbmNpcGFsKFxuICAgICAgcHJvdmlkZXIub3BlbklkQ29ubmVjdFByb3ZpZGVyQXJuLFxuICAgICAge1xuICAgICAgICBTdHJpbmdMaWtlOiB7XG4gICAgICAgICAgW2Ake3Jhd0VuZHBvaW50fTpzdWJgXTogZm9ybWF0UmVwb3MocHJvcHMucmVwb3MpLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgICdzdHM6QXNzdW1lUm9sZVdpdGhXZWJJZGVudGl0eScsXG4gICAgKTtcblxuICAgIC8vIHBlcm1pdCB0aGlzIHJvbGUgZnJvbSBhc3N1bWluZyBhbGwgb2YgdGhlIENESyBib290c3RyYXAgcm9sZXNcbiAgICBjb25zdCBvaWRjUG9saWN5U3RhdGVtZW50ID0gbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgYWN0aW9uczogWydzdHM6QXNzdW1lUm9sZSddLFxuICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgIGNvbmRpdGlvbnM6IHtcbiAgICAgICAgJ0ZvckFueVZhbHVlOlN0cmluZ0VxdWFscyc6IHtcbiAgICAgICAgICAnaWFtOlJlc291cmNlVGFnL2F3cy1jZGs6Ym9vdHN0cmFwLXJvbGUnOiBbXG4gICAgICAgICAgICAnZGVwbG95JyxcbiAgICAgICAgICAgICdsb29rdXAnLFxuICAgICAgICAgICAgJ2ZpbGUtcHVibGlzaGluZycsXG4gICAgICAgICAgICAnaW1hZ2UtcHVibGlzaGluZycsXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBwZXJtaXQgdGhpcyByb2xlIGZyb20gYWNjZXNzaW5nIGVjciByZXBvc2l0b3JpZXMgZm9yIGRvY2tlciBhc3NldHNcbiAgICBjb25zdCBlY3JQb2xpY3lTdGF0ZW1lbnQgPSBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICBhY3Rpb25zOiBbJ2VjcjpHZXRBdXRob3JpemF0aW9uVG9rZW4nXSxcbiAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgfSk7XG5cbiAgICB0aGlzLnJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ2dpdGh1Yi1hY3Rpb24tcm9sZScsIHtcbiAgICAgIHJvbGVOYW1lOiBwcm9wcy5yb2xlTmFtZSA/PyAnR2l0SHViQWN0aW9uUm9sZScsXG4gICAgICBhc3N1bWVkQnk6IHByaW5jaXBhbCxcbiAgICAgIGlubGluZVBvbGljaWVzOiB7XG4gICAgICAgIEFzc3VtZUJvb3RzdHJhcFJvbGVzOiBuZXcgaWFtLlBvbGljeURvY3VtZW50KHtcbiAgICAgICAgICBzdGF0ZW1lbnRzOiBbb2lkY1BvbGljeVN0YXRlbWVudCwgZWNyUG9saWN5U3RhdGVtZW50XSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gc2hvdyB0aGUgcm9sZSBhcm4gaW4gdGhlIHN0YWNrIG91dHB1dFxuICAgIG5ldyBDZm5PdXRwdXQodGhpcywgJ3JvbGVBcm4nLCB7XG4gICAgICB2YWx1ZTogdGhpcy5yb2xlLnJvbGVBcm4sXG4gICAgfSk7XG4gIH1cbn1cblxuZnVuY3Rpb24gZm9ybWF0UmVwb3MocmVwb3M6IHN0cmluZ1tdKSB7XG4gIGNvbnN0IGZvcm1hdHRlZFJlcG9zID0gW107XG4gIGZvciAoY29uc3QgcmVwbyBvZiByZXBvcykge1xuICAgIGZvcm1hdHRlZFJlcG9zLnB1c2goYHJlcG86JHtyZXBvfToqYCk7XG4gIH1cbiAgcmV0dXJuIGZvcm1hdHRlZFJlcG9zO1xufSJdfQ==