# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['autocsr', 'autocsr.protos']

package_data = \
{'': ['*']}

install_requires = \
['Jinja2>=2.11.3,<3.0.0',
 'PyYAML>=5.4.1,<6.0.0',
 'cryptography>=3.4.7,<4.0.0',
 'protobuf>=3.15.8,<4.0.0',
 'pyOpenSSL>=20.0.1,<21.0.0',
 'pyasn1-modules>=0.2.8,<0.3.0',
 'pycryptodome>=3.10.1,<4.0.0',
 'python-pkcs11>=0.7.0,<0.8.0',
 'typer>=0.3.2,<0.4.0']

entry_points = \
{'console_scripts': ['autocsr = autocsr.cli:main']}

setup_kwargs = {
    'name': 'autocsr',
    'version': '1.0.11',
    'description': 'AutoCSR is a command-line tool and library for automatically generating Certificate Signing Requests from easy to define configuration files.',
    'long_description': '[![Build Status](https://travis-ci.com/maxwolfe/autocsr.svg?token=qz3Kxzoztoakrxm4CFDZ&branch=master)](https://travis-ci.com/maxwolfe/autocsr)\n\n# AutoCSR: Automatic Certificate Signing Request Generation\n\n`AutoCSR` is a command-line tool and library for automatically generating Certificate Signing Requests from basic user input and easy to define templates.\n\n`AutoCSR` was developed to empower non-security professionals to quickly and easily generate their own simple Certificate Signing Requests with minimal security knowledge required.\n\n`AutoCSR` also provides security professionals with the ability to define complex Certificate Signing Requests with templates that can be easily shared with non-security professionals to generate complex Certificate Signing Requests without the need for detailed instructions or handholding.\n\n## Install\n```\npip install autocsr\n```\n\n## Usage\n```\nUsage: autocsr [OPTIONS] COMMAND [ARGS]...\n\nCommands:\n  build   Create certificate signing requests from a config file.\n  create  Create a new Certificate Signing Request with little customization.\n  prompt  Prompt the user for Certificate Signing Request fields.\n```\n\n## Quickstart\n\n### Create a simple CSR with a Common Name\nThe easiest way to generate a Certificate Signing Request is by specifying the common name and letting `autocsr create` handle the rest! This is perfect for simple Certificate Signing Requests that don\'t require any custom attributes, extensions, or specific key formats.\n\n```\nmax@wolfetop:/app# autocsr create "My Common Name"\nWhere to store the new key? [./My Common Name.pem]:\nWhere to store the new csr? [./My Common Name.csr]:\nCreated new CSR at ./My Common Name.csr\n```\n\nYou can also specify the `key-path` and `output-path` to avoid the mandatory prompts.\n\n```\nmax@wolfetop:/app# autocsr create "My Common Name" --key-path my_common_name.pem --output-path my_common_name.csr\nCreated new CSR at my_common_name.csr\n```\n\n### Create a simple CSR with a Prompt\nIf you need to generate a Certificate Signing Request without custom attributes or extensions, but still want to customize the subject and signing information, `autocsr prompt` can guide you through the process while offering you safe defaults and a list of options where applicable.\n\n```\nmax@wolfetop:/app# autocsr prompt "My Prompted CSR"\nWhere to store the new key? [./My Prompted CSR.pem]:\nWhere to store the new csr? [./My Prompted CSR.csr]:\nWhat is your country identifier? (2 characters) []:\nWhat is your state or province name? []:\nWhat is your locality name? []:\nWhat is your organization name? []:\nWhat is your organizational unit name? []:\nWhat is your email address? []:\nWhat is the desired hash algorithm to use? (SHA256, SHA224, SHA384, SHA512, SHA512_224, SHA512_256, BLAKE2b, BLAKE2s, SHA3_224, SHA3_256, SHA3_384, SHA3_512, SHAKE128, SHAKE256) [SHA256]:\nWhat is the desired type of key? (RSA, DSA, EC) [RSA]:\nWhat is the desired key size? (1024, 2048, 4096) [2048]:\nCreated new CSR at ./My Prompted CSR.csr\n```\n\n### Create a CSR with a Template\nFor more customized Certificate Signing Requests and those that need to be repeatedly generated in a consistent way, it will often be desirable to add the details of your Certificate Signing Request inside a template. `autocsr build` can be used to construct one or many Certificate Signing Requests from the details present in a template.\n\n#### Create a Template\n```\n# quick_csr.yaml\nMy First CSR:\n  subject:\n    common_name: My first AutoCSR\n  key_info:\n    key_path: /tmp/my_first_key.key\n    create: True\n  output_path: /tmp/my_first_autocsr.csr\n```\n\n#### Run AutoCSR\n```\nmax@wolfetop:/app# autocsr build quick_csr.yaml\nCreated new CSR at /tmp/my_first_autocsr.csr\n```\n\n#### Validate New CSR\n```\nmax@wolfetop:/app# openssl req -text -noout -verify -in /tmp/my_first_autocsr.csr\nverify OK\nCertificate Request:\n    Data:\n        Version: 1 (0x0)\n        Subject: CN = My first AutoCSR\n        Subject Public Key Info:\n            Public Key Algorithm: rsaEncryption\n                RSA Public-Key: (2048 bit)\n                Modulus:\n                    00:93:f6:52:6e:51:64:0a:a6:95:d5:89:71:11:bf:\n                    50:c6:cc:54:e7:9a:06:ec:16:0a:3f:dc:8f:ee:57:\n                    50:6f:bd:6b:92:89:50:d5:97:5c:74:ca:86:08:41:\n                    52:af:13:5a:a9:8c:3d:79:64:14:77:fe:ef:52:d6:\n                    57:6c:59:01:f4:02:03:a7:b0:c6:24:9c:1d:26:72:\n                    15:f8:8b:58:25:85:83:b4:b4:26:7b:4f:db:59:93:\n                    09:07:02:d3:8f:92:1d:d1:c6:94:9c:6a:06:77:de:\n                    f5:5e:b8:4a:30:86:c5:6e:81:35:f4:cb:88:e7:79:\n                    a3:91:22:c9:03:92:9c:8a:3a:3c:49:58:fe:18:e2:\n                    e2:18:c3:6d:e0:a2:7d:21:62:80:dd:54:fb:4b:85:\n                    ed:08:5f:10:0b:af:2e:66:bc:57:53:a3:d9:06:23:\n                    ce:97:63:54:4c:8e:13:0f:01:1a:3e:9d:80:53:91:\n                    71:f8:3f:93:03:41:d7:64:2b:5e:b6:d1:b8:17:bd:\n                    10:6f:56:b5:d2:ec:3d:1a:91:0e:7a:2e:f2:ff:d4:\n                    03:33:8b:91:48:6b:e3:e6:ea:f2:49:48:49:81:5a:\n                    c7:b9:5a:ef:85:ce:71:61:28:7e:28:8c:07:23:48:\n                    e3:c3:7e:74:46:bc:88:fa:84:9b:d3:16:98:9b:58:\n                    29:9d\n                Exponent: 65537 (0x10001)\n        Attributes:\n        Requested Extensions:\n    Signature Algorithm: sha256WithRSAEncryption\n         07:6c:ac:32:92:04:f2:57:42:8f:93:92:09:92:77:01:c6:5e:\n         e4:7f:17:f6:78:fd:8a:83:8a:d7:55:3f:f1:c4:ba:09:1c:9e:\n         2a:04:db:e3:2f:b9:c1:d0:49:53:59:47:6f:d0:3e:ae:c3:4d:\n         96:f2:f0:f8:b2:9f:67:62:fc:4b:32:35:c7:f3:cc:78:83:d0:\n         82:0a:b6:f0:90:83:12:10:73:49:36:ac:f2:27:85:91:b1:9d:\n         0d:22:d7:2f:34:84:0a:2f:c1:d3:ee:62:82:72:78:64:93:17:\n         83:7c:68:65:89:e5:ad:cc:e3:f0:c8:03:1e:18:c0:11:89:af:\n         9f:5d:7a:23:a0:c9:c7:97:44:fd:18:40:6e:aa:02:cf:bb:8a:\n         17:6c:24:64:3b:a5:9b:0c:c8:52:e1:8f:8f:83:ec:8b:14:5c:\n         a7:38:83:f8:67:6b:2d:3e:1a:02:39:2a:57:27:3a:c0:62:b7:\n         bc:90:6c:b6:f5:2c:32:f5:87:dc:b0:0c:b2:93:d3:2d:8d:cb:\n         0b:a1:e6:70:aa:b9:67:bf:9b:89:ae:25:12:08:08:83:ee:7e:\n         58:33:e6:53:37:fb:28:7c:79:98:39:bf:b4:8b:b9:e3:b5:75:\n         8d:bd:b6:ce:e1:11:69:81:ab:37:d9:f0:3c:6e:35:b1:23:d8:\n         6a:10:be:2e\n```\n\n## Creating Templates\nA template is an overall definition of the data you want to be include in your Certificate Signing Request. Most fields can be optionally excluded or have safe default values for those who don\'t want to worry too much about the details.\n\n### Subject\nThe subject of the Certificate Signing Request contains various metadata about the certificate, the only required portion being the `common_name`.\n\n```\n# subject_example.yaml\nA Subject CSR:\n  subject:\n    common_name: All about the subject\n    country_name: US\n    state_or_province_name: California\n    locality_name: Bay Area\n    organization_name: SecurityWolfe\n    organizational_unit_name: The Cool Team\n    email_address: max@securitywolfe.com\n  key_info:\n    key_path: /tmp/subject_example.key\n    create: True\n  output_path: /tmp/subject_example.csr\n```\n\n#### Explicit Subject Example\n\n```\nCertificate Request:\n    Data:\n        Version: 1 (0x0)\n        Subject: CN = All about the subject, C = US, ST = California, L = Bay Area, O = SecurityWolfe, OU = The Cool Team, emailAddress = max@securitywolfe.com\n```\n\n### Key and Signing Information\n`AutoCSR` allows keys to be generated on the fly or loaded directly from files. You can also explicitly define the hash algorithm to use for signing.\n\n```\n# key_example.yaml\nRSA With Explicit Key Parameters:\n  subject:\n    common_name: RSA CSR with explicit key parameters\n  key_info:\n    key_path: /tmp/my_rsa.key\n    create: True\n    key_type: RSA\n    key_size: 2048\n    public_exponent: 65537\n  hash_type: SHA256\n  output_path: /tmp/my_rsa_autocsr.csr\n\nDSA From Key File:\n  subject:\n    common_name: DSA CSR from key file\n  key_info:\n    key_path: /tmp/my_dsa.key\n  output_path: /tmp/my_dsa_autocsr.csr\n\nEC With Explicit Key Parameters:\n  subject:\n    common_name: EC CSR with explicit key parameters\n  key_info:\n    key_path: /tmp/my_ec.key\n    create: True\n    key_type: EC\n    curve: SECP256R1\n  hash_type: SHA512\n  output_path: /tmp/my_ec_autocsr.csr\n```\n\nRunning `autocsr build` on this template will generate three Certificate Signing Requests:\n\n#### RSA Example CSR\n```\nCertificate Request:\n    Data:\n        Version: 1 (0x0)\n        Subject: CN = RSA CSR with explicit key parameters\n        Subject Public Key Info:\n            Public Key Algorithm: rsaEncryption\n                RSA Public-Key: (2048 bit)\n                Modulus: ...\n                Exponent: 65537 (0x10001)\n        Attributes:\n        Requested Extensions:\n    Signature Algorithm: sha256WithRSAEncryption\n```\n\n#### DSA Example CSR\n```\nCertificate Request:\n    Data:\n        Version: 1 (0x0)\n        Subject: CN = DSA CSR from key file\n        Subject Public Key Info:\n            Public Key Algorithm: dsaEncryption\n                pub: ...\n                P: ...\n                Q: ...\n                G: ...\n        Attributes:\n        Requested Extensions:\n    Signature Algorithm: dsa_with_SHA256\n```\n\n#### EC Example CSR\n```\nCertificate Request:\n    Data:\n        Version: 1 (0x0)\n        Subject: CN = EC CSR with explicit key parameters\n        Subject Public Key Info:\n            Public Key Algorithm: id-ecPublicKey\n                Public-Key: (256 bit)\n                pub: ...\n                ASN1 OID: prime256v1\n                NIST CURVE: P-256\n        Attributes:\n        Requested Extensions:\n    Signature Algorithm: ecdsa-with-SHA512\n```\n\n### Attributes\nCertificates can optionally contain a plethora of pre-defined and custom attributes which map an `oid` to a binary value. Because we primarily use YAML for our template, we require that attribute values are base64 encoded in the template when defining attributes. Optionally for predefined attributes, a string name can be used instead of the dotted-string `oid`.\n\n```\n# attribute_example.yaml\nCSR with Attributes:\n  subject:\n    common_name: My CSR with custom and well-known attributes\n  key_info:\n    key_path: /tmp/my_attributes.key\n    create: True\n  output_path: /tmp/my_attributes_autocsr.csr\n  attributes:\n    - oid: issuerAltName\n      b64_value: TXkgSXNzdWVyIEFsdCBOYW1l\n    - oid: 1.2.345.678  # A custom OID\n      b64_value: TXkgQ3VzdG9tIEF0dHJpYnV0ZQ==\n```\n\n#### Custom and Predefined Attributes Example\n\n```\nCertificate Request:\n    Data:\n        Version: 1 (0x0)\n        Subject: CN = My CSR with custom and well-known attributes\n        Subject Public Key Info:\n            Public Key Algorithm: rsaEncryption\n                RSA Public-Key: (2048 bit)\n                Modulus: ...\n                Exponent: 65537 (0x10001)\n        Attributes:\n            X509v3 Issuer Alternative Name:My Issuer Alt Name\n            1.2.345.678              :My Custom Attribute\n```\n\n### Extensions\nCertificate Signing Requests offer a plethora of predefined extensions. An exhaustive list of the available extensions are available [here](https://cryptography.io/en/latest/x509/reference/#x-509-extensions), but I will provide a few examples of modeling extensions in templates below. Keep in mind that like attributes, extensions that require bytes as input will need to have their data represented in base64.\n\n```\n# extension_example.yaml\nMy CSR with Extensions:\n  subject:\n    common_name: Some Common Extensions for CSRs\n  key_info:\n    key_path: /tmp/my_extension_example.key\n    create: True\n  output_path: /tmp/my_extension_autocsr.csr\n  extensions:\n    - critical: True\n      extension_type: OCSPNoCheck\n    - critical: True\n      subject_key_identifier:\n        b64_digest: TXkgRXhhbXBsZSBTdWJqZWN0IEtleSBJZGVudGlmaWVy\n    - critical: False\n      extended_key_usage:\n        usages:\n          - "serverAuth"\n          - "1.2.3.4.5"\n    - critical: True\n      key_usage:\n        digital_signature: True\n        content_commitment: False\n        key_encipherment: True\n        data_encipherment: False\n        key_agreement: True\n        key_cert_sign: False\n        crl_sign: True\n        encipher_only: False\n        decipher_only: True\n```\n\n#### Various Extensions Example\n```\nCertificate Request:\n    Data:\n        Version: 1 (0x0)\n        Subject: CN = Some Common Extensions for CSRs\n        Subject Public Key Info:\n            Public Key Algorithm: rsaEncryption\n                RSA Public-Key: (2048 bit)\n                Modulus: ...\n                Exponent: 65537 (0x10001)\n        Attributes:\n        Requested Extensions:\n            OCSP No Check: critical\n\n            X509v3 Subject Key Identifier: critical\n                4D:79:20:45:78:61:6D:70:6C:65:20:53:75:62:6A:65:63:74:20:4B:65:79:20:49:64:65:6E:74:69:66:69:65:72\n            X509v3 Extended Key Usage:\n                TLS Web Server Authentication, 1.2.3.4.5\n            X509v3 Key Usage: critical\n                Digital Signature, Key Encipherment, Key Agreement, CRL Sign, Decipher Only\n```\n\n### Jinja Templates\nAlternatively to using `.yaml` files, you can use `.jinja2` files modeled after YAML, but including environment variables to be replaced. If you wanted to recreate the first example using environment variables, you can do so like this:\n\n```\n# quick_csr.jinja2\n{{ NAME }}\'s First CSR:\n  subject:\n    common_name: {{ NAME }}\'s first AutoCSR\n  key_info:\n    key_path: /tmp/{{ NAME }}s_first_key.key\n    create: True\n  output_path: /tmp/{{ NAME }}s_first_autocsr.csr\n```\n\n#### Run AutoCSR on a Jinja Template\n```\nmax@wolfetop:/app# NAME=Max autocsr quick_csr.jinja2\nCreated new CSR at /tmp/Maxs_first_autocsr.csr\n```\n\n#### Validate Jinja Templated CSR\n```\nmax@wolfetop:/app# openssl req -text -noout -verify -in /tmp/Maxs_first_autocsr.csr\nverify OK\nCertificate Request:\n    Data:\n        Version: 1 (0x0)\n        Subject: CN = Max\'s first AutoCSR\n        Subject Public Key Info:\n            Public Key Algorithm: rsaEncryption\n                RSA Public-Key: (2048 bit)\n```\n',
    'author': 'Max Wolfe',
    'author_email': 'max@securitywolfe.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/maxwolfe/autocsr',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
