"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TextractGenericSyncSfnTask = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
/**
 * Calls Textract synchronous API.
 * Supports the Textract APIs:  'GENERIC' | 'ANALYZEID' | 'EXPENSE'
 * When GENERIC is called with features in the manifest definition, will call the AnalzyeDocument API.
 * Takes the configuration from "Payload"."manifest"
 * Will retry on recoverable errors based on textractAsyncCallMaxRetries
 * errors for retry: ['ThrottlingException', 'LimitExceededException', 'InternalServerError', 'ProvisionedThroughputExceededException'],
 *
 * Input: "Payload"."manifest"
 * Output: Textract JSON Schema at  s3_output_bucket/s3_output_prefix
 *
 * Example (Python)
 * ```python
 *         textract_sync_task = tcdk.TextractGenericSyncSfnTask(
            self,
            "TextractSync",
            s3_output_bucket=document_bucket.bucket_name,
            s3_output_prefix=s3_output_prefix,
            integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            lambda_log_level="DEBUG",
            timeout=Duration.hours(24),
            input=sfn.TaskInput.from_object({
                "Token":
                sfn.JsonPath.task_token,
                "ExecutionId":
                sfn.JsonPath.string_at('$$.Execution.Id'),
                "Payload":
                sfn.JsonPath.entire_payload,
            }),
            result_path="$.textract_result")
  ```
 */
class TextractGenericSyncSfnTask extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.version = '0.0.1';
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, TextractGenericSyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        if (!props.s3OutputBucket || !props.s3OutputPrefix) {
            throw new Error('no s3OutputBucket and s3OutputPrefix passed in');
        }
        var textractAPI = props.textractAPI === undefined ? 'GENERIC' : props.textractAPI;
        var textractStateMachineTimeoutMinutes = props.textractStateMachineTimeoutMinutes === undefined ? 2880 : props.textractStateMachineTimeoutMinutes;
        var lambdaLogLevel = props.lambdaLogLevel === undefined ? 'DEBUG' : props.lambdaLogLevel;
        var lambdaTimeout = props.lambdaTimeout === undefined ? 300 : props.lambdaTimeout;
        var lambdaMemory = props.lambdaMemory === undefined ? 300 : props.lambdaMemory;
        var textractAsyncCallMaxRetries = props.textractAsyncCallMaxRetries === undefined ? 100 : props.textractAsyncCallMaxRetries;
        var textractAsyncCallBackoffRate = props.textractAsyncCallBackoffRate === undefined ? 1.1 : props.textractAsyncCallBackoffRate;
        var textractAsyncCallInterval = props.textractAsyncCallInterval === undefined ? 1 : props.textractAsyncCallInterval;
        var enableCloudWatchMetricsAndDashboard = props.enableCloudWatchMetricsAndDashboard === undefined ? false :
            props.enableCloudWatchMetricsAndDashboard;
        var s3InputPrefix = props.s3InputPrefix === undefined ? '' : props.s3InputPrefix;
        var s3OutputPrefix = props.s3OutputPrefix === undefined ? '' : props.s3OutputPrefix;
        this.textractSyncCallFunction = new lambda.DockerImageFunction(this, 'TextractSyncCall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/textract_sync/')),
            memorySize: lambdaMemory,
            architecture: lambda.Architecture.X86_64,
            timeout: aws_cdk_lib_1.Duration.seconds(lambdaTimeout),
            environment: {
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_OUTPUT_PREFIX: props.s3OutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
                TEXTRACT_API: textractAPI,
            },
        });
        const textractSyncCallTask = new tasks.LambdaInvoke(this, 'TextractSyncCallTask', { lambdaFunction: this.textractSyncCallFunction });
        textractSyncCallTask.addRetry({
            maxAttempts: textractAsyncCallMaxRetries,
            backoffRate: textractAsyncCallBackoffRate,
            interval: aws_cdk_lib_1.Duration.seconds(textractAsyncCallInterval),
            errors: ['ThrottlingException', 'LimitExceededException', 'InternalServerError', 'ProvisionedThroughputExceededException'],
        });
        this.textractSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'textract:Analyze*',
                'textract:Detect*',
            ],
            resources: ['*'],
        }));
        /** ################ INPUT BUCKET POLICIES */
        if (props.inputPolicyStatements === undefined) {
            if (props.s3InputBucket === undefined) {
                this.textractSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:GetObject', 's3:ListBucket'],
                    resources: ['*'],
                }));
            }
            else {
                this.textractSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:GetObject', 's3:ListBucket'],
                    resources: [
                        path.join(`arn:aws:s3:::${props.s3InputBucket}`, s3InputPrefix, '/'),
                        path.join(`arn:aws:s3:::${props.s3InputBucket}`, s3InputPrefix, '/*'),
                    ],
                }));
            }
        }
        else {
            for (var policyStatement of props.inputPolicyStatements) {
                this.textractSyncCallFunction.addToRolePolicy(policyStatement);
            }
        }
        /** ##################### OUTPUT BUCKET POLICIES */
        if (props.outputPolicyStatements === undefined) {
            if (props.s3OutputBucket === undefined) {
                this.textractSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:PutObject'],
                    resources: ['*'],
                }));
            }
            else {
                this.textractSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:PutObject'],
                    resources: [
                        path.join(`arn:aws:s3:::${props.s3OutputBucket}`, s3OutputPrefix, '/'),
                        path.join(`arn:aws:s3:::${props.s3OutputBucket}`, s3OutputPrefix, '/*'),
                    ],
                }));
            }
        }
        else {
            for (var policyStatement of props.outputPolicyStatements) {
                this.textractSyncCallFunction.addToRolePolicy(policyStatement);
            }
        }
        const workflow_chain = sfn.Chain.start(textractSyncCallTask);
        this.stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: workflow_chain,
            timeout: aws_cdk_lib_1.Duration.hours(textractStateMachineTimeoutMinutes),
        });
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN) {
            this.textractSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    'states:SendTaskFailure', 'states:SendTaskSuccess',
                ],
                resources: ['*'],
            }));
        }
        // =========
        // DASHBOARD
        // =========
        if (enableCloudWatchMetricsAndDashboard) {
            const appName = this.node.tryGetContext('appName');
            const customMetricNamespace = 'TextractConstructGenericSync';
            // OPERATIONAL
            const syncDurationMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-DurationFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'Duration',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'durationMs')
                    .whereString('message', '=', `textract_sync_${textractAPI}_call_duration_in_ms:`),
                metricValue: '$durationMs',
            });
            this.syncDurationMetric = syncDurationMetricFilter.metric({ statistic: 'avg' });
            const syncNumberPagesMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'NumberPages',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                    .whereString('message', '=', `textract_sync_${textractAPI}_number_of_pages_processed:`),
                metricValue: '$pages',
            });
            this.syncNumberPagesMetric = syncNumberPagesMetricFilter.metric({ statistic: 'sum' });
            const syncNumberPagesSendMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesSendFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'NumberPages',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                    .whereString('message', '=', `textract_sync_${textractAPI}_number_of_pages_send_to_process:`),
                metricValue: '$pages',
            });
            this.syncNumberPagesSendMetric = syncNumberPagesSendMetricFilter.metric({ statistic: 'sum' });
            const syncTimedOutMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-TimedOutFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'DurationSeconds',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message1', 'durationS', 'message2')
                    .whereString('message1', '=', 'Task timed out after').whereString('message2', '=', 'seconds'),
                metricValue: '$durationS',
            });
            this.syncTimedOutMetric = syncTimedOutMetricFilter.metric({ statistic: 'sum' });
            // OPERATIONAL STOP
            // CALCUATED OPERATIONAL METRICS
            const pagesPerSecond = new cloudwatch.MathExpression({
                expression: 'pages / (duration / 1000)',
                usingMetrics: {
                    pages: this.syncNumberPagesMetric,
                    duration: this.syncDurationMetric,
                },
            });
            // CALCUATED OPERATIONAL METRICS STOP
            const errorFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-ErrorFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'Errors',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('ERROR', 'Error', 'error'),
                metricValue: '1',
            });
            const limitExceededExceptionFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-limitExceededExceptionFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'LimitExceededException',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.LimitExceededException'),
                metricValue: '1',
            });
            const throttlingException = new aws_logs_1.MetricFilter(this, `${appName}-throttlingExceptionFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'ThrottlingException',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ThrottlingException'),
                metricValue: '1',
            });
            const provisionedThroughputExceededException = new aws_logs_1.MetricFilter(this, `${appName}-provisionedThroughputFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'ProvisionedThroughputExceededException',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ProvisionedThroughputExceededException'),
                metricValue: '1',
            });
            const internalServerError = new aws_logs_1.MetricFilter(this, `${appName}-internalServerErrorFilter`, {
                logGroup: this.textractSyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'InternalServerError',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.InternalServerError'),
                metricValue: '1',
            });
            const errorMetric = errorFilterMetric.metric({ statistic: 'sum' });
            const limitExceededMetric = limitExceededExceptionFilterMetric.metric({ statistic: 'sum' });
            const throttlingMetric = throttlingException.metric({ statistic: 'sum' });
            const provisionedThroughputMetric = provisionedThroughputExceededException.metric({ statistic: 'sum' });
            const internalServerErrorMetric = internalServerError.metric({ statistic: 'sum' });
            const textractStartDocumentTextThrottle = new cloudwatch.Metric({
                namespace: 'AWS/Textract',
                metricName: 'ThrottledCount',
                dimensionsMap: {
                    Operation: 'StartDocumentTextDetection',
                },
                statistic: 'sum',
            });
            // DASHBOARD LAMBDA
            const dashboardWidth = 24;
            // const widgetStandardHeight=9;
            // DASHBOARD
            new cloudwatch.Dashboard(this, '${appName}-TestDashboard', {
                end: 'end',
                periodOverride: cloudwatch.PeriodOverride.AUTO,
                start: 'start',
                widgets: [
                    [
                        new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Operational Data Row widgets', width: dashboardWidth })),
                    ],
                    [
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'PagesPerSecond', left: [pagesPerSecond], width: Math.floor(dashboardWidth / 2) })),
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'Duration', left: [this.syncDurationMetric], width: Math.floor(dashboardWidth / 4) })),
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPagesProcessed', left: [this.syncNumberPagesMetric], width: Math.floor(dashboardWidth / 4) })),
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPagesSendToProcess', left: [this.syncNumberPagesSendMetric], width: Math.floor(dashboardWidth / 4) })),
                    ],
                    [
                        new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Async Textract Exceptions Row', width: dashboardWidth })),
                    ],
                    [
                        new cloudwatch.GraphWidget({ title: 'Errors', left: [errorMetric], width: Math.floor(dashboardWidth / 5) }),
                        new cloudwatch.GraphWidget({ title: 'LimitExceeded', left: [limitExceededMetric], width: Math.floor(dashboardWidth / 5) }),
                        new cloudwatch.GraphWidget({ title: 'Throttling', left: [throttlingMetric], width: Math.floor(dashboardWidth / 5) }),
                        new cloudwatch.GraphWidget({ title: 'ProvisionedThrougput', left: [provisionedThroughputMetric], width: Math.floor(dashboardWidth / 5) }),
                        new cloudwatch.GraphWidget({ title: 'InternalServerError', left: [internalServerErrorMetric], width: Math.floor(dashboardWidth / 5) }),
                    ],
                    [
                        new cloudwatch.TextWidget({ markdown: '# Textract', width: dashboardWidth }),
                    ],
                    [
                        new cloudwatch.GraphWidget({ title: 'Textract-StartDetectText-ThrottledCount', left: [textractStartDocumentTextThrottle] }),
                    ],
                    [
                        new cloudwatch.TextWidget({ markdown: '# textractSyncCallFunction', width: dashboardWidth }),
                    ],
                    [
                        new cloudwatch.GraphWidget({ title: 'Sync-Function-Errors', left: [this.textractSyncCallFunction.metricErrors()], width: Math.floor(dashboardWidth / 4) }),
                        new cloudwatch.GraphWidget({ title: 'Sync-Function-Invocations', left: [this.textractSyncCallFunction.metricInvocations()], width: Math.floor(dashboardWidth / 4) }),
                        new cloudwatch.GraphWidget({ title: 'Sync-Function-Throttles', left: [this.textractSyncCallFunction.metricThrottles()], width: Math.floor(dashboardWidth / 4) }),
                        new cloudwatch.GraphWidget({ title: 'Sync-Function-TimedOut', left: [this.syncTimedOutMetric], width: Math.floor(dashboardWidth / 4) }),
                    ],
                ],
            });
        }
        // END DASHBOARD
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
         * @internal
         */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
         * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
         *
         * the scoped access policy should be generated accordingly.
         *
         * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
         */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.TextractGenericSyncSfnTask = TextractGenericSyncSfnTask;
_a = JSII_RTTI_SYMBOL_1;
TextractGenericSyncSfnTask[_a] = { fqn: "amazon-textract-idp-cdk-constructs.TextractGenericSyncSfnTask", version: "0.0.17" };
TextractGenericSyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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