"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TextractGenericAsyncSfnTask = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const sns = require("aws-cdk-lib/aws-sns");
const aws_sns_subscriptions_1 = require("aws-cdk-lib/aws-sns-subscriptions");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
var textractSync_1 = require("./textractSync");
Object.defineProperty(exports, "TextractGenericSyncSfnTask", { enumerable: true, get: function () { return textractSync_1.TextractGenericSyncSfnTask; } });
var textractDecider_1 = require("./textractDecider");
Object.defineProperty(exports, "TextractPOCDecider", { enumerable: true, get: function () { return textractDecider_1.TextractPOCDecider; } });
var textractClassificationConfigurator_1 = require("./textractClassificationConfigurator");
Object.defineProperty(exports, "TextractClassificationConfigurator", { enumerable: true, get: function () { return textractClassificationConfigurator_1.TextractClassificationConfigurator; } });
var textractOutputConfigToJSON_1 = require("./textractOutputConfigToJSON");
Object.defineProperty(exports, "TextractAsyncToJSON", { enumerable: true, get: function () { return textractOutputConfigToJSON_1.TextractAsyncToJSON; } });
var textractGenerateCSV_1 = require("./textractGenerateCSV");
Object.defineProperty(exports, "TextractGenerateCSV", { enumerable: true, get: function () { return textractGenerateCSV_1.TextractGenerateCSV; } });
var textractA2I_1 = require("./textractA2I");
Object.defineProperty(exports, "TextractA2ISfnTask", { enumerable: true, get: function () { return textractA2I_1.TextractA2ISfnTask; } });
var rdsCSVToAurora_1 = require("./rdsCSVToAurora");
Object.defineProperty(exports, "CSVToAuroraTask", { enumerable: true, get: function () { return rdsCSVToAurora_1.CSVToAuroraTask; } });
var comprehendClassification_1 = require("./comprehendClassification");
Object.defineProperty(exports, "ComprehendGenericSyncSfnTask", { enumerable: true, get: function () { return comprehendClassification_1.ComprehendGenericSyncSfnTask; } });
var spacyClassification_1 = require("./spacyClassification");
Object.defineProperty(exports, "SpacySfnTask", { enumerable: true, get: function () { return spacyClassification_1.SpacySfnTask; } });
var documentSplitter_1 = require("./documentSplitter");
Object.defineProperty(exports, "DocumentSplitter", { enumerable: true, get: function () { return documentSplitter_1.DocumentSplitter; } });
var rdsAuroraServerless_1 = require("./rdsAuroraServerless");
Object.defineProperty(exports, "RDSAuroraServerless", { enumerable: true, get: function () { return rdsAuroraServerless_1.RDSAuroraServerless; } });
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
/**
 * This Task calls the Textract through the asynchronous API.
 *
 * Which API to call is defined in
 *
 * When GENERIC is called with features in the manifest definition, will call the AnalzyeDocument API.
 *
 * Takes the configuration from "Payload"."manifest"
 *
 * Will retry on recoverable errors based on textractAsyncCallMaxRetries
 * errors for retry: ['ThrottlingException', 'LimitExceededException', 'InternalServerError', 'ProvisionedThroughputExceededException'],
 *
 * Internally calls Start* calls with OutputConfig and SNSNotification.
 * Another Lambda functions waits for SNS Notification event and notifies the Step Function flow with the task token.
 *
 * Step Function JSON input requirements
 *
 * **Input**: "Payload"."manifest"
 *
 * **Output**: "TextractTempOutputJsonPath" points to potentially paginated Textract JSON Schema output at "TextractTempOutputJsonPath" (using the example code it will be at: "textract_result"."TextractTempOutputJsonPath")
 *
 * Works together with TextractAsyncToJSON, which takes the s3_output_bucket/s3_temp_output_prefix location as input
 *
 * Example (Python)
 * ```python
    textract_async_task = tcdk.TextractGenericAsyncSfnTask(
        self,
        "TextractAsync",
        s3_output_bucket=s3_output_bucket,
        s3_temp_output_prefix=s3_temp_output_prefix,
        integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        lambda_log_level="DEBUG",
        timeout=Duration.hours(24),
        input=sfn.TaskInput.from_object({
            "Token":
            sfn.JsonPath.task_token,
            "ExecutionId":
            sfn.JsonPath.string_at('$$.Execution.Id'),
            "Payload":
            sfn.JsonPath.entire_payload,
        }),
        result_path="$.textract_result")
    ```

 */
class TextractGenericAsyncSfnTask extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        /**resources
         * DynamoDB table
         * textractAsyncSNSRole
         */
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, TextractGenericAsyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        var textractStateMachineTimeoutMinutes = props.textractStateMachineTimeoutMinutes === undefined ? 2880 : props.textractStateMachineTimeoutMinutes;
        var lambdaLogLevel = props.lambdaLogLevel === undefined ? 'DEBUG' : props.lambdaLogLevel;
        var textractAPI = props.textractAPI === undefined ? 'GENERIC' : props.textractAPI;
        var textractAsyncCallMaxRetries = props.textractAsyncCallMaxRetries === undefined ? 100 : props.textractAsyncCallMaxRetries;
        var textractAsyncCallBackoffRate = props.textractAsyncCallBackoffRate === undefined ? 1.1 : props.textractAsyncCallBackoffRate;
        var textractAsyncCallInterval = props.textractAsyncCallInterval === undefined ? 1 : props.textractAsyncCallInterval;
        var s3TempOutputPrefix = props.s3TempOutputPrefix === undefined ? '' : props.s3TempOutputPrefix;
        var s3InputPrefix = props.s3InputPrefix === undefined ? '' : props.s3InputPrefix;
        var enableCloudWatchMetricsAndDashboard = props.enableCloudWatchMetricsAndDashboard === undefined ? false :
            props.enableCloudWatchMetricsAndDashboard;
        /** RESOURCE DYNAMODB TABLE for TASK TOKEN */
        if (props.taskTokenTable === undefined) {
            this.taskTokenTable = new dynamodb.Table(this, 'TextractTaskTokenTable', {
                partitionKey: {
                    name: 'ID',
                    type: dynamodb.AttributeType.STRING,
                },
                billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                timeToLiveAttribute: 'ttltimestamp',
            });
        }
        else {
            this.taskTokenTable = props.taskTokenTable;
        }
        this.taskTokenTableName = this.taskTokenTable.tableName;
        /** RESOURCE: SNS Role for Textract to use*/
        if (props.snsRoleTextract === undefined) {
            this.textractAsyncSNSRole = new iam.Role(this, 'TextractAsyncSNSRole', {
                assumedBy: new iam.ServicePrincipal('textract.amazonaws.com'),
                managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSQSFullAccess'),
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSNSFullAccess'),
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonS3ReadOnlyAccess'),
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonTextractFullAccess')],
            });
        }
        else {
            this.textractAsyncSNSRole = props.snsRoleTextract;
        }
        this.textractAsyncSNS = new sns.Topic(this, 'TextractAsyncSNS');
        this.textractAsyncCallFunction = new lambda.DockerImageFunction(this, 'TextractAsyncCall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/textract_async/')),
            memorySize: 128,
            architecture: lambda.Architecture.X86_64,
            environment: {
                NOTIFICATION_SNS: this.textractAsyncSNS.topicArn,
                NOTIFICATION_ROLE_ARN: this.textractAsyncSNSRole.roleArn,
                TOKEN_STORE_DDB: this.taskTokenTableName,
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_TEMP_OUTPUT_PREFIX: props.s3TempOutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
                TEXTRACT_API: textractAPI,
            },
        });
        const textractAsyncCallTask = new tasks.LambdaInvoke(this, 'TextractAsyncCallTask', { lambdaFunction: this.textractAsyncCallFunction });
        textractAsyncCallTask.addRetry({
            maxAttempts: textractAsyncCallMaxRetries,
            backoffRate: textractAsyncCallBackoffRate,
            interval: aws_cdk_lib_1.Duration.seconds(textractAsyncCallInterval),
            errors: ['ThrottlingException', 'LimitExceededException', 'InternalServerError', 'ProvisionedThroughputExceededException'],
        });
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'textract:Start*',
                'textract:Get*',
            ],
            resources: ['*'],
        }));
        /** ################ INPUT BUCKET POLICIES */
        if (props.inputPolicyStatements === undefined) {
            if (props.s3InputBucket === undefined) {
                this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:GetObject', 's3:ListBucket'],
                    resources: ['*'],
                }));
            }
            else {
                this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:GetObject'],
                    resources: [
                        path.join(`arn:aws:s3:::${props.s3InputBucket}`, '/*'),
                        path.join(`arn:aws:s3:::${props.s3InputBucket}`, s3InputPrefix, '/*'),
                    ],
                }));
                this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:ListBucket'],
                    resources: [
                        path.join(`arn:aws:s3:::${props.s3InputBucket}`),
                    ],
                }));
            }
        }
        else {
            for (var policyStatement of props.inputPolicyStatements) {
                this.textractAsyncCallFunction.addToRolePolicy(policyStatement);
            }
        }
        /** ##################### OUTPUT BUCKET POLICIES */
        if (props.outputPolicyStatements === undefined) {
            if (props.s3OutputBucket === undefined) {
                this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:PutObject'],
                    resources: ['*'],
                }));
            }
            else {
                this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:PutObject'],
                    resources: [
                        path.join(`arn:aws:s3:::${props.s3OutputBucket}`, s3TempOutputPrefix, '/'),
                        path.join(`arn:aws:s3:::${props.s3OutputBucket}`, s3TempOutputPrefix, '/*'),
                    ],
                }));
            }
        }
        else {
            for (var policyStatement of props.outputPolicyStatements) {
                this.textractAsyncCallFunction.addToRolePolicy(policyStatement);
            }
        }
        // this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['sns:*'], resources: [this.textractAsyncSNS.topicArn] }));
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['dynamodb:PutItem', 'dynamodb:GetItem'], resources: [this.taskTokenTable.tableArn] }));
        this.textractAsyncReceiveSNSFunction = new lambda.DockerImageFunction(this, 'TextractAsyncSNSFunction', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/textract_async_sns_listener/')),
            memorySize: 128,
            architecture: lambda.Architecture.X86_64,
            timeout: aws_cdk_lib_1.Duration.seconds(900),
            environment: {
                TOKEN_STORE_DDB: this.taskTokenTableName,
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_TEMP_OUTPUT_PREFIX: props.s3TempOutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
            },
        });
        this.textractAsyncSNS.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(this.textractAsyncReceiveSNSFunction));
        this.textractAsyncReceiveSNSFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['dynamodb:GetItem'], resources: [this.taskTokenTable.tableArn] }));
        const workflow_chain = sfn.Chain.start(textractAsyncCallTask);
        this.stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: workflow_chain,
            timeout: aws_cdk_lib_1.Duration.hours(textractStateMachineTimeoutMinutes),
        });
        this.textractAsyncReceiveSNSFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskSuccess', 'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        // =========
        // DASHBOARD
        // =========
        if (enableCloudWatchMetricsAndDashboard) {
            const appName = this.node.tryGetContext('appName');
            const customMetricNamespace = 'TextractConstructGenericAsync';
            // OPERATIONAL
            const asyncDurationMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-DurationFilter`, {
                logGroup: this.textractAsyncReceiveSNSFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'Duration',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'durationMs')
                    .whereString('message', '=', `textract_async_${textractAPI}_job_duration_in_ms:`),
                metricValue: '$durationMs',
            });
            this.asyncDurationMetric = asyncDurationMetricFilter.metric({ statistic: 'avg' });
            const asyncNumberPagesMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesFilter`, {
                logGroup: this.textractAsyncReceiveSNSFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'NumberPages',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                    .whereString('message', '=', `textract_async_${textractAPI}_number_of_pages_processed:`),
                metricValue: '$pages',
            });
            this.asyncNumberPagesMetric = asyncNumberPagesMetricFilter.metric({ statistic: 'sum' });
            // TODO: expose all filters as properties of Construct, so users can build their own metrics
            const asyncJobFinshedMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-JobsFinishedFilter`, {
                logGroup: this.textractAsyncReceiveSNSFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'JobsFinished',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                    .whereString('message', '=', `textract_async_${textractAPI}_number_of_pages_processed:`),
                metricValue: '1',
            });
            this.asyncJobFinshedMetric = asyncJobFinshedMetricFilter.metric({ statistic: 'sum' });
            const asyncJobStartedMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-JobsStartedFilter`, {
                logGroup: this.textractAsyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'JobsStarted',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message')
                    .whereString('message', '=', `textract_async_${textractAPI}_job_started`),
                metricValue: '1',
            });
            this.asyncJobStartedMetric = asyncJobStartedMetricFilter.metric({ statistic: 'sum' });
            const asyncNumberPagesSendMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesSendFilter`, {
                logGroup: this.textractAsyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'NumberPages',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                    .whereString('message', '=', `textract_async_${textractAPI}_number_of_pages_send_to_process:`),
                metricValue: '$pages',
            });
            this.asyncNumberPagesSendMetric = asyncNumberPagesSendMetricFilter.metric({ statistic: 'sum' });
            // OPERATIONAL STOP
            // CALCUATED OPERATIONAL METRICS
            const pagesPerSecond = new cloudwatch.MathExpression({
                expression: 'pages / (duration / 1000)',
                usingMetrics: {
                    pages: this.asyncNumberPagesMetric,
                    duration: this.asyncDurationMetric,
                },
            });
            const openJobs = new cloudwatch.MathExpression({
                expression: 'startedJobs - finishedJobs',
                usingMetrics: {
                    startedJobs: this.asyncJobStartedMetric,
                    finishedJobs: this.asyncJobFinshedMetric,
                },
            });
            // CALCUATED OPERATIONAL METRICS STOP
            const errorFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-ErrorFilter`, {
                logGroup: this.textractAsyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'Errors',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('ERROR', 'Error', 'error'),
                metricValue: '1',
            });
            const limitExceededExceptionFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-limitExceededExceptionFilter`, {
                logGroup: this.textractAsyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'LimitExceededException',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.LimitExceededException'),
                metricValue: '1',
            });
            const throttlingException = new aws_logs_1.MetricFilter(this, `${appName}-throttlingExceptionFilter`, {
                logGroup: this.textractAsyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'ThrottlingException',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ThrottlingException'),
                metricValue: '1',
            });
            const provisionedThroughputExceededException = new aws_logs_1.MetricFilter(this, `${appName}-provisionedThroughputFilter`, {
                logGroup: this.textractAsyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'ProvisionedThroughputExceededException',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ProvisionedThroughputExceededException'),
                metricValue: '1',
            });
            const internalServerError = new aws_logs_1.MetricFilter(this, `${appName}-internalServerErrorFilter`, {
                logGroup: this.textractAsyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'InternalServerError',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.InternalServerError'),
                metricValue: '1',
            });
            const errorMetric = errorFilterMetric.metric({ statistic: 'sum' });
            const limitExceededMetric = limitExceededExceptionFilterMetric.metric({ statistic: 'sum' });
            const throttlingMetric = throttlingException.metric({ statistic: 'sum' });
            const provisionedThroughputMetric = provisionedThroughputExceededException.metric({ statistic: 'sum' });
            const internalServerErrorMetric = internalServerError.metric({ statistic: 'sum' });
            const textractStartDocumentTextThrottle = new cloudwatch.Metric({
                namespace: 'AWS/Textract',
                metricName: 'ThrottledCount',
                dimensionsMap: {
                    Operation: 'StartDocumentTextDetection',
                },
                statistic: 'sum',
            });
            // DASHBOARD LAMBDA
            const dashboardWidth = 24;
            // const widgetStandardHeight=9;
            // DASHBOARD
            new cloudwatch.Dashboard(this, '${appName}-TestDashboard', {
                end: 'end',
                periodOverride: cloudwatch.PeriodOverride.AUTO,
                start: 'start',
                widgets: [
                    [
                        new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Operational Data Row widgets', width: dashboardWidth })),
                    ],
                    [
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'PagesPerSecond', left: [pagesPerSecond], width: Math.floor(dashboardWidth / 2) })),
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'OpenJobs', left: [openJobs], width: Math.floor(dashboardWidth / 2) })),
                    ],
                    [
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'Duration', left: [this.asyncDurationMetric], width: Math.floor(dashboardWidth / 4) })),
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPages', left: [this.asyncNumberPagesMetric], width: Math.floor(dashboardWidth / 4) })),
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPages', left: [this.asyncNumberPagesSendMetric], width: Math.floor(dashboardWidth / 4) })),
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'JobsStartedAndFinished', left: [this.asyncJobFinshedMetric], right: [this.asyncJobStartedMetric], width: Math.floor(dashboardWidth / 4) })),
                    ],
                    [
                        new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Async Textract Exceptions Row', width: dashboardWidth })),
                    ],
                    [
                        new cloudwatch.GraphWidget({ title: 'Errors', left: [errorMetric], width: Math.floor(dashboardWidth / 5) }),
                        new cloudwatch.GraphWidget({ title: 'LimitExceeded', left: [limitExceededMetric], width: Math.floor(dashboardWidth / 5) }),
                        new cloudwatch.GraphWidget({ title: 'Throttling', left: [throttlingMetric], width: Math.floor(dashboardWidth / 5) }),
                        new cloudwatch.GraphWidget({ title: 'ProvisionedThrougput', left: [provisionedThroughputMetric], width: Math.floor(dashboardWidth / 5) }),
                        new cloudwatch.GraphWidget({ title: 'InternalServerError', left: [internalServerErrorMetric], width: Math.floor(dashboardWidth / 5) }),
                    ],
                    [
                        new cloudwatch.TextWidget({ markdown: '# Textract', width: dashboardWidth }),
                    ],
                    [
                        new cloudwatch.GraphWidget({ title: 'Textract-StartDetectText-ThrottledCount', left: [textractStartDocumentTextThrottle] }),
                    ],
                    [
                        new cloudwatch.TextWidget({ markdown: '# textractAsyncCallFunction', width: dashboardWidth }),
                    ],
                    [
                        new cloudwatch.GraphWidget({ title: 'Async-Function-Errors', left: [this.textractAsyncCallFunction.metricErrors()], width: Math.floor(dashboardWidth / 3) }),
                        new cloudwatch.GraphWidget({ title: 'Async-Function-Invocations', left: [this.textractAsyncCallFunction.metricInvocations()], width: Math.floor(dashboardWidth / 3) }),
                        new cloudwatch.GraphWidget({ title: 'Async-Function-Throttles', left: [this.textractAsyncCallFunction.metricThrottles()], width: Math.floor(dashboardWidth / 3) }),
                    ],
                    [
                        new cloudwatch.TextWidget({ markdown: '# textractAsyncReceiveSNSFunction', width: dashboardWidth }),
                    ],
                    [
                        new cloudwatch.GraphWidget({ title: 'SNS-Function-Errors', left: [this.textractAsyncReceiveSNSFunction.metricErrors()], width: Math.floor(dashboardWidth / 3) }),
                        new cloudwatch.GraphWidget({ title: 'SNS-Function-Invocations', left: [this.textractAsyncReceiveSNSFunction.metricInvocations()], width: Math.floor(dashboardWidth / 3) }),
                        new cloudwatch.GraphWidget({ title: 'SNS-Function-Throttles', left: [this.textractAsyncReceiveSNSFunction.metricThrottles()], width: Math.floor(dashboardWidth / 3) }),
                    ],
                ],
            });
        }
        // END DASHBOARD
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
         * @internal
         */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
         * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
         *
         * the scoped access policy should be generated accordingly.
         *
         * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
         */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.TextractGenericAsyncSfnTask = TextractGenericAsyncSfnTask;
_a = JSII_RTTI_SYMBOL_1;
TextractGenericAsyncSfnTask[_a] = { fqn: "amazon-textract-idp-cdk-constructs.TextractGenericAsyncSfnTask", version: "0.0.17" };
TextractGenericAsyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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