import { ExtensionStore } from '../stores/ExtensionStore';
import { UIStore } from '../stores/UIStore';
import { PollingRequesterRef, didPollingManager } from './DIDPollingManager';
import { actions } from './Actions';
import { computeCollectionState } from './Helpers';
export class NotebookPollingListener {
    constructor(notebookListener) {
        this.activePolling = [];
        this.activeNotebookAttachmentDIDs = [];
        this.notebookListener = notebookListener;
        this.pollingRef = new PollingRequesterRef();
        // Listen to change in attachments
        ExtensionStore.subscribe(s => s.activeNotebookAttachment, attachments => {
            this.removeUnfocusedDIDs(attachments);
            this.processNewAttachments(attachments);
            this.activeNotebookAttachmentDIDs = (attachments === null || attachments === void 0 ? void 0 : attachments.map(a => a.did)) || [];
        });
        // Listen to change in file status
        UIStore.subscribe(s => s.fileDetails, (fileDetails, state, prevFileDetails) => {
            if (!fileDetails) {
                return;
            }
            const listenedFileDetails = Object.keys(fileDetails)
                .filter(did => this.activeNotebookAttachmentDIDs.includes(did))
                .map(did => ({
                did,
                file: {
                    current: fileDetails[did],
                    prev: prevFileDetails[did]
                }
            }));
            listenedFileDetails.forEach(({ did, file }) => {
                var _a;
                if (file.current.status === 'REPLICATING') {
                    this.enablePolling(did, 'file');
                }
                else {
                    if (this.activePolling.includes(did)) {
                        this.disablePolling(did);
                    }
                    if (file.current.status === 'OK' && ((_a = file.prev) === null || _a === void 0 ? void 0 : _a.status) === 'REPLICATING') {
                        const { activeNotebookPanel } = ExtensionStore.getRawState();
                        this.notebookListener.reinjectSpecificDID(activeNotebookPanel, did);
                    }
                }
            });
        });
        // Listen to change in collection status
        UIStore.subscribe(s => s.collectionDetails, (collectionDetails, state, prevCollectionDetails) => {
            if (!collectionDetails) {
                return;
            }
            const listenedCollectionDetails = Object.keys(collectionDetails)
                .filter(did => this.activeNotebookAttachmentDIDs.includes(did))
                .map(did => ({
                did,
                file: {
                    current: collectionDetails[did],
                    prev: prevCollectionDetails[did]
                }
            }));
            listenedCollectionDetails.forEach(({ did, file }) => {
                const currentCollectionState = computeCollectionState(file.current);
                if (currentCollectionState === 'REPLICATING') {
                    this.enablePolling(did, 'collection');
                }
                else {
                    if (this.activePolling.includes(did)) {
                        this.disablePolling(did);
                    }
                    const prevCollectionState = computeCollectionState(file.prev);
                    if (currentCollectionState === 'AVAILABLE' && prevCollectionState === 'REPLICATING') {
                        const { activeNotebookPanel } = ExtensionStore.getRawState();
                        this.notebookListener.reinjectSpecificDID(activeNotebookPanel, did);
                    }
                }
            });
        });
    }
    removeUnfocusedDIDs(attachments = []) {
        const shouldNoLongerPoll = this.activePolling.filter(did => !attachments.find(a => a.did === did));
        shouldNoLongerPoll.forEach(did => this.disablePolling(did));
    }
    processNewAttachments(attachments = []) {
        const newlyAdded = attachments.filter(a => !this.activePolling.find(did => did === a.did));
        newlyAdded.forEach(attachment => {
            this.shouldEnablePolling(attachment).then(shouldEnable => {
                if (shouldEnable) {
                    this.enablePolling(attachment.did, attachment.type);
                }
            });
        });
    }
    async shouldEnablePolling(attachment) {
        const { activeInstance } = UIStore.getRawState();
        if (attachment.type === 'file') {
            const didDetails = await actions.getFileDIDDetails(activeInstance.name, attachment.did);
            return didDetails.status === 'REPLICATING';
        }
        else {
            const didDetails = await actions.getCollectionDIDDetails(activeInstance.name, attachment.did);
            return didDetails.find(d => d.status === 'REPLICATING');
        }
    }
    enablePolling(did, type) {
        didPollingManager.requestPolling(did, type, this.pollingRef, false);
        this.activePolling.push(did);
    }
    disablePolling(did) {
        didPollingManager.disablePolling(did, this.pollingRef);
        this.activePolling = this.activePolling.filter(d => d !== did);
    }
}
