#  Copyright (c) 2022 Mira Geoscience Ltd.
#
#  This file is part of geoapps.
#
#  geoapps is distributed under the terms and conditions of the MIT License
#  (see LICENSE file at the root of this source code package).

from __future__ import annotations


def parameters():
    """
    Dictionary of parameters defining geophysical acquisition systems
    """
    return {
        "Airborne TEM Survey": {
            "type": "time",
            "data_type": "dBzdt",
            "flag": None,
            "channel_start_index": 1,
            "bird_offset": [0, 0, 0],
        },
        "AeroTEM (2007)": {
            "type": "time",
            "flag": "Zoff",
            "channel_start_index": 1,
            "channels": {
                "[1]": 58.1e-6,
                "[2]": 85.9e-6,
                "[3]": 113.7e-6,
                "[4]": 141.4e-6,
                "[5]": 169.2e-6,
                "[6]": 197.0e-6,
                "[7]": 238.7e-6,
                "[8]": 294.2e-6,
                "[9]": 349.8e-6,
                "[10]": 405.3e-6,
                "[11]": 474.8e-6,
                "[12]": 558.1e-6,
                "[13]": 655.3e-6,
                "[14]": 794.2e-6,
                "[15]": 988.7e-6,
                "[16]": 1280.3e-6,
                "[17]": 1738.7e-6,
            },
            "uncertainty": [
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
            ],
            "waveform": [
                [-1.10e-03, 1e-8],
                [-8.2500e-04, 5.0e-01],
                [-5.50e-04, 1.0e00],
                [-2.7500e-04, 5.0e-01],
                [0.0e00, 0.0e00],
                [2.50e-05, 0.0e00],
                [5.0e-05, 0.0e00],
                [7.50e-05, 0.0e00],
                [1.0e-04, 0.0e00],
                [1.2500e-04, 0.0e00],
                [1.50e-04, 0.0e00],
                [1.7500e-04, 0.0e00],
                [2.0e-04, 0.0e00],
                [2.2500e-04, 0.0e00],
                [2.50e-04, 0.0e00],
                [3.0550e-04, 0.0e00],
                [3.6100e-04, 0.0e00],
                [4.1650e-04, 0.0e00],
                [4.7200e-04, 0.0e00],
                [5.2750e-04, 0.0e00],
                [6.0750e-04, 0.0e00],
                [6.8750e-04, 0.0e00],
                [7.6750e-04, 0.0e00],
                [8.4750e-04, 0.0e00],
                [9.2750e-04, 0.0e00],
                [1.1275e-03, 0.0e00],
                [1.3275e-03, 0.0e00],
                [1.5275e-03, 0.0e00],
                [1.7275e-03, 0.0e00],
                [1.9275e-03, 0.0e00],
                [2.1275e-03, 0.0e00],
            ],
            "tx_offsets": [[0, 0, 0]],
            "bird_offset": [0, 0, -40],
            "comment": "normalization accounts for 2.5m radius loop * 8 turns * 69 A current, nanoTesla",
            "normalization": [2.9e-4, 1e-9],
            "tx_specs": {"type": "CircularLoop", "a": 1.0, "I": 1.0},
            "data_type": "dBzdt",
        },
        "AeroTEM (2010)": {
            "type": "time",
            "flag": "Zoff",
            "channel_start_index": 1,
            "channels": {
                "[1]": 67.8e-6,
                "[2]": 95.6e-6,
                "[3]": 123.4e-6,
                "[4]": 151.2e-6,
                "[5]": 178.9e-6,
                "[6]": 206.7e-6,
                "[7]": 262.3e-6,
                "[8]": 345.6e-6,
                "[9]": 428.9e-6,
                "[10]": 512.3e-6,
                "[11]": 623.4e-6,
                "[12]": 762.3e-6,
                "[13]": 928.9e-6,
                "[14]": 1165.0e-6,
                "[15]": 1526.2e-6,
                "[16]": 2081.7e-6,
                "[17]": 2942.8e-6,
            },
            "uncertainty": [
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
                [0.05, 5e-0],
            ],
            "waveform": [
                [-1.10e-03, 1e-8],
                [-8.2500e-04, 5.0e-01],
                [-5.50e-04, 1.0e00],
                [-2.7500e-04, 5.0e-01],
                [0.0e00, 0.0e00],
                [2.50e-05, 0.0e00],
                [5.0e-05, 0.0e00],
                [7.50e-05, 0.0e00],
                [1.0e-04, 0.0e00],
                [1.2500e-04, 0.0e00],
                [1.50e-04, 0.0e00],
                [1.7500e-04, 0.0e00],
                [2.0e-04, 0.0e00],
                [2.2500e-04, 0.0e00],
                [2.50e-04, 0.0e00],
                [3.0550e-04, 0.0e00],
                [3.6100e-04, 0.0e00],
                [4.1650e-04, 0.0e00],
                [4.7200e-04, 0.0e00],
                [5.2750e-04, 0.0e00],
                [6.0750e-04, 0.0e00],
                [6.8750e-04, 0.0e00],
                [7.6750e-04, 0.0e00],
                [8.4750e-04, 0.0e00],
                [9.2750e-04, 0.0e00],
                [1.1275e-03, 0.0e00],
                [1.3275e-03, 0.0e00],
                [1.5275e-03, 0.0e00],
                [1.7275e-03, 0.0e00],
                [1.9275e-03, 0.0e00],
                [2.1275e-03, 0.0e00],
                [2.3275e-03, 0.0e00],
                [2.5275e-03, 0.0e00],
                [2.7275e-03, 0.0e00],
                [2.9275e-03, 0.0e00],
                [3.1275e-03, 0.0e00],
            ],
            "tx_offsets": [[0, 0, 0]],
            "bird_offset": [0, 0, -40],
            "comment": "normalization accounts for 2.5m radius loop, 8 turns * 69 A current, nanoTesla",
            "normalization": [2.9e-4, 1e-9],
            "tx_specs": {"type": "CircularLoop", "a": 1.0, "I": 1.0},
            "data_type": "dBzdt",
        },
        "DIGHEM": {
            "type": "frequency",
            "flag": "CPI900",
            "channel_start_index": 0,
            "channels": {
                "CPI900": 900,
                "CPI7200": 7200,
                "CPI56K": 56000,
                "CPQ900": 900,
                "CPQ7200": 7200,
                "CPQ56K": 56000,
            },
            "components": {
                "CPI900": "real",
                "CPQ900": "imag",
                "CPI7200": "real",
                "CPQ7200": "imag",
                "CPI56K": "real",
                "CPQ56K": "imag",
            },
            "tx_offsets": [
                [8, 0, 0],
                [8, 0, 0],
                [6.3, 0, 0],
                [8, 0, 0],
                [8, 0, 0],
                [6.3, 0, 0],
            ],
            "bird_offset": [0, 0, 0],
            "uncertainty": [
                [0.0, 2],
                [0.0, 5],
                [0.0, 10],
                [0.0, 2],
                [0.0, 5],
                [0.0, 10],
            ],
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "normalization": "ppm",
        },
        "GENESIS (2014)": {
            "type": "time",
            "flag": "emz_step_final",
            "channel_start_index": 1,
            "channels": {
                "0": 9e-6,
                "1": 26e-6,
                "2": 52.0e-6,
                "3": 95e-6,
                "4": 156e-6,
                "5": 243e-6,
                "6": 365e-6,
                "7": 547e-6,
                "8": 833e-6,
                "9": 1259e-6,
                "10": 1858e-6,
            },
            "uncertainty": [
                [0.05, 100],
                [0.05, 100],
                [0.05, 100],
                [0.05, 100],
                [0.05, 100],
                [0.05, 100],
                [0.05, 2000],
                [0.05, 100],
                [0.05, 100],
                [0.05, 100],
                [0.05, 100],
            ],
            "waveform": "stepoff",
            "tx_offsets": [[-90, 0, -43]],
            "bird_offset": [-90, 0, -43],
            "comment": "normalization accounts for unit dipole moment at the tx_offset, in part-per-million",
            "normalization": "ppm",
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "data_type": "Bz",
        },
        "GEOTEM 75 Hz - 2082 Pulse": {
            "type": "time",
            "flag": "EM_chan",
            "channel_start_index": 5,
            "channels": {
                "1": -1953e-6,
                "2": -1562e-6,
                "3": -989e-6,
                "4": -416e-6,
                "5": 163e-6,
                "6": 235e-6,
                "7": 365e-6,
                "8": 521e-6,
                "9": 703e-6,
                "10": 912e-6,
                "11": 1146e-6,
                "12": 1407e-6,
                "13": 1693e-6,
                "14": 2005e-6,
                "15": 2344e-6,
                "16": 2709e-6,
                "17": 3073e-6,
                "18": 3464e-6,
                "19": 3880e-6,
                "20": 4297e-6,
            },
            "uncertainty": [
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
                [0.05, 40.0],
            ],
            "waveform": [
                [-2.08000000e-03, 1.22464680e-16],
                [-1.83000000e-03, 3.68686212e-01],
                [-1.58000000e-03, 6.85427422e-01],
                [-1.33000000e-03, 9.05597273e-01],
                [-1.08000000e-03, 9.98175554e-01],
                [-8.30000000e-04, 9.50118712e-01],
                [-5.80000000e-04, 7.68197578e-01],
                [-3.30000000e-04, 4.78043417e-01],
                [-8.00000000e-05, 1.20536680e-01],
                [0.00000000e00, 0.00000000e00],
                [1.00000000e-04, 0.00000000e00],
                [2.00000000e-04, 0.00000000e00],
                [3.00000000e-04, 0.00000000e00],
                [4.00000000e-04, 0.00000000e00],
                [5.00000000e-04, 0.00000000e00],
                [6.00000000e-04, 0.00000000e00],
                [7.00000000e-04, 0.00000000e00],
                [8.00000000e-04, 0.00000000e00],
                [9.00000000e-04, 0.00000000e00],
                [1.00000000e-03, 0.00000000e00],
                [1.10000000e-03, 0.00000000e00],
                [1.20000000e-03, 0.00000000e00],
                [1.30000000e-03, 0.00000000e00],
                [1.40000000e-03, 0.00000000e00],
                [1.50000000e-03, 0.00000000e00],
                [1.60000000e-03, 0.00000000e00],
                [1.70000000e-03, 0.00000000e00],
                [1.80000000e-03, 0.00000000e00],
                [1.90000000e-03, 0.00000000e00],
                [2.00000000e-03, 0.00000000e00],
                [2.10000000e-03, 0.00000000e00],
                [2.20000000e-03, 0.00000000e00],
                [2.30000000e-03, 0.00000000e00],
                [2.40000000e-03, 0.00000000e00],
                [2.50000000e-03, 0.00000000e00],
                [2.60000000e-03, 0.00000000e00],
                [2.70000000e-03, 0.00000000e00],
                [2.80000000e-03, 0.00000000e00],
                [2.90000000e-03, 0.00000000e00],
                [3.00000000e-03, 0.00000000e00],
                [3.10000000e-03, 0.00000000e00],
                [3.20000000e-03, 0.00000000e00],
                [3.30000000e-03, 0.00000000e00],
                [3.40000000e-03, 0.00000000e00],
                [3.50000000e-03, 0.00000000e00],
                [3.60000000e-03, 0.00000000e00],
                [3.70000000e-03, 0.00000000e00],
                [3.80000000e-03, 0.00000000e00],
                [3.90000000e-03, 0.00000000e00],
                [4.00000000e-03, 0.00000000e00],
                [4.10000000e-03, 0.00000000e00],
                [4.20000000e-03, 0.00000000e00],
                [4.30000000e-03, 0.00000000e00],
                [4.40000000e-03, 0.00000000e00],
            ],
            "tx_offsets": [[-123, 0, -56]],
            "bird_offset": [-123, 0, -56],
            "comment": "normalization accounts for unit dipole moment at the tx_offset, in part-per-million",
            "normalization": "ppm",
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "data_type": "Bz",
        },
        "HELITEM2 (25C, 7.5Hz)": {
            "type": "time",
            "flag": "emz_db",
            "channel_start_index": 0,
            "channels": {
                "[0]": 0.0012044,
                "[1]": 0.0012247,
                "[2]": 0.0012492,
                "[3]": 0.0012776,
                "[4]": 0.0013183,
                "[5]": 0.0013712,
                "[6]": 0.0014363,
                "[7]": 0.0015218,
                "[8]": 0.0016316,
                "[9]": 0.0017741,
                "[10]": 0.0019612,
                "[11]": 0.0022013,
                "[12]": 0.0025106,
                "[13]": 0.0029093,
                "[14]": 0.0034220,
                "[15]": 0.0040853,
                "[16]": 0.0049398,
                "[17]": 0.0060384,
                "[18]": 0.0074544,
                "[19]": 0.0092814,
                "[20]": 0.0116414,
                "[21]": 0.0146891,
                "[22]": 0.0186198,
                "[23]": 0.0236857,
                "[24]": 0.0302205,
            },
            "uncertainty": [
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
            ],
            "waveform": [
                [-3.2764e-02, 0.0000e00],
                [-3.2715e-02, 2.4658e-04],
                [-3.2707e-02, -2.2671e-03],
                [-3.2690e-02, 1.7103e-02],
                [-3.2682e-02, 2.2281e-02],
                [-3.2674e-02, 4.8842e-02],
                [-3.2666e-02, 6.2788e-02],
                [-3.2658e-02, 8.6137e-02],
                [-3.2544e-02, 3.4966e-01],
                [-3.2422e-02, 6.1264e-01],
                [-3.2365e-02, 7.2428e-01],
                [-3.2308e-02, 8.2712e-01],
                [-3.2259e-02, 9.0801e-01],
                [-3.2251e-02, 9.1983e-01],
                [-3.2243e-02, 9.2401e-01],
                [-3.2235e-02, 9.3466e-01],
                [-3.2227e-02, 9.3298e-01],
                [-3.0835e-02, 9.4237e-01],
                [-2.9443e-02, 9.5029e-01],
                [-2.7743e-02, 9.5871e-01],
                [-2.6042e-02, 9.6592e-01],
                [-2.4353e-02, 9.7201e-01],
                [-2.2664e-02, 9.7721e-01],
                [-2.0923e-02, 9.8172e-01],
                [-1.9181e-02, 9.8550e-01],
                [-1.7604e-02, 9.8834e-01],
                [-1.6027e-02, 9.9070e-01],
                [-1.4450e-02, 9.9264e-01],
                [-1.2873e-02, 9.9423e-01],
                [-1.1296e-02, 9.9553e-01],
                [-9.5924e-03, 9.9664e-01],
                [-7.8893e-03, 9.9748e-01],
                [-6.1861e-03, 9.9809e-01],
                [-4.4829e-03, 9.9848e-01],
                [-2.7797e-03, 9.9868e-01],
                [-1.0766e-03, 9.9876e-01],
                [6.2660e-04, 0.0000e00],
                [6.5101e-04, 0.0000e00],
                [7.1612e-04, 0.0000e00],
                [7.6495e-04, 0.0000e00],
                [8.3005e-04, 0.0000e00],
                [8.9515e-04, 0.0000e00],
                [1.0905e-03, 0.0000e00],
                [1.1067e-03, 0.0000e00],
                [1.1230e-03, 0.0000e00],
                [2.7616e-03, 0.0000e00],
                [4.4002e-03, 0.0000e00],
                [6.0388e-03, 0.0000e00],
                [7.6774e-03, 0.0000e00],
                [9.3160e-03, 0.0000e00],
                [1.0955e-02, 0.0000e00],
                [1.2593e-02, 0.0000e00],
                [1.4232e-02, 0.0000e00],
                [1.5870e-02, 0.0000e00],
                [1.7509e-02, 0.0000e00],
                [1.9148e-02, 0.0000e00],
                [2.0786e-02, 0.0000e00],
                [2.2425e-02, 0.0000e00],
                [2.4063e-02, 0.0000e00],
                [2.5702e-02, 0.0000e00],
                [2.7340e-02, 0.0000e00],
                [2.8979e-02, 0.0000e00],
                [3.0618e-02, 0.0000e00],
                [3.2256e-02, 0.0000e00],
                [3.3895e-02, 0.0000e00],
            ],
            "tx_offsets": [[17.5, 0, 34.6]],
            "bird_offset": [17.5, 0, 34.6],
            "comment": "normalization accounts for a loop 961 m**2 area * 4 turns * 146 A current, nanoTesla",
            "normalization": [1.7818e-6, 1e-9],
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "data_type": "dBzdt",
        },
        "HELITEM (35C)": {
            "type": "time",
            "flag": "emz_db",
            "channel_start_index": 5,
            "channels": {
                "[1]": -0.007772,
                "[2]": -0.003654,
                "[3]": -0.002678,
                "[4]": -0.000122,
                "[5]": 0.000228,
                "[6]": 0.000269,
                "[7]": 0.000326,
                "[8]": 0.000399,
                "[9]": 0.000488,
                "[10]": 0.000594,
                "[11]": 0.000716,
                "[12]": 0.000854,
                "[13]": 0.001009,
                "[14]": 0.001196,
                "[15]": 0.001424,
                "[16]": 0.001693,
                "[17]": 0.002018,
                "[18]": 0.002417,
                "[19]": 0.002905,
                "[20]": 0.003499,
                "[21]": 0.004215,
                "[22]": 0.005078,
                "[23]": 0.006128,
                "[24]": 0.007406,
                "[25]": 0.008952,
                "[26]": 0.010824,
                "[27]": 0.013094,
                "[28]": 0.015845,
                "[29]": 0.019173,
                "[30]": 0.02321,
            },
            "uncertainty": [
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
                [0.05, 2e-1],
            ],
            "waveform": [
                [-8.000e-03, 1.000e-03],
                [-7.750e-03, 9.802e-02],
                [-7.500e-03, 1.950e-01],
                [-7.250e-03, 2.902e-01],
                [-7.000e-03, 3.826e-01],
                [-6.750e-03, 4.713e-01],
                [-6.500e-03, 5.555e-01],
                [-6.250e-03, 6.344e-01],
                [-6.000e-03, 7.071e-01],
                [-5.750e-03, 7.730e-01],
                [-5.500e-03, 8.315e-01],
                [-5.250e-03, 8.820e-01],
                [-5.000e-03, 9.239e-01],
                [-4.750e-03, 9.569e-01],
                [-4.500e-03, 9.808e-01],
                [-4.250e-03, 9.951e-01],
                [-4.000e-03, 1.000e00],
                [-3.750e-03, 9.951e-01],
                [-3.500e-03, 9.808e-01],
                [-3.250e-03, 9.569e-01],
                [-3.000e-03, 9.239e-01],
                [-2.750e-03, 8.820e-01],
                [-2.500e-03, 8.315e-01],
                [-2.250e-03, 7.730e-01],
                [-2.000e-03, 7.071e-01],
                [-1.750e-03, 6.344e-01],
                [-1.500e-03, 5.555e-01],
                [-1.250e-03, 4.713e-01],
                [-1.000e-03, 3.826e-01],
                [-7.500e-04, 2.902e-01],
                [-5.000e-04, 1.950e-01],
                [-2.500e-04, 9.802e-02],
                [0.000e00, 0.000e00],
                [5.000e-05, 0.000e00],
                [1.000e-04, 0.000e00],
                [1.500e-04, 0.000e00],
                [2.000e-04, 0.000e00],
                [2.500e-04, 0.000e00],
                [3.000e-04, 0.000e00],
                [3.500e-04, 0.000e00],
                [4.000e-04, 0.000e00],
                [4.500e-04, 0.000e00],
                [5.000e-04, 0.000e00],
                [5.500e-04, 0.000e00],
                [6.000e-04, 0.000e00],
                [6.500e-04, 0.000e00],
                [7.000e-04, 0.000e00],
                [7.500e-04, 0.000e00],
                [8.000e-04, 0.000e00],
                [8.500e-04, 0.000e00],
                [9.000e-04, 0.000e00],
                [9.500e-04, 0.000e00],
                [1.000e-03, 0.000e00],
                [1.050e-03, 0.000e00],
                [1.100e-03, 0.000e00],
                [1.150e-03, 0.000e00],
                [1.200e-03, 0.000e00],
                [1.225e-03, 0.000e00],
                [1.475e-03, 0.000e00],
                [1.725e-03, 0.000e00],
                [1.975e-03, 0.000e00],
                [2.225e-03, 0.000e00],
                [2.475e-03, 0.000e00],
                [2.725e-03, 0.000e00],
                [2.975e-03, 0.000e00],
                [3.225e-03, 0.000e00],
                [3.475e-03, 0.000e00],
                [3.725e-03, 0.000e00],
                [3.975e-03, 0.000e00],
                [4.225e-03, 0.000e00],
                [4.475e-03, 0.000e00],
                [4.725e-03, 0.000e00],
                [4.975e-03, 0.000e00],
                [5.225e-03, 0.000e00],
                [5.475e-03, 0.000e00],
                [5.725e-03, 0.000e00],
                [5.975e-03, 0.000e00],
                [6.225e-03, 0.000e00],
                [6.475e-03, 0.000e00],
                [6.725e-03, 0.000e00],
                [6.975e-03, 0.000e00],
                [7.225e-03, 0.000e00],
                [7.475e-03, 0.000e00],
                [7.725e-03, 0.000e00],
                [7.975e-03, 0.000e00],
                [8.225e-03, 0.000e00],
                [8.475e-03, 0.000e00],
                [8.500e-03, 0.000e00],
                [9.250e-03, 0.000e00],
                [1.000e-02, 0.000e00],
                [1.075e-02, 0.000e00],
                [1.150e-02, 0.000e00],
                [1.225e-02, 0.000e00],
                [1.300e-02, 0.000e00],
                [1.375e-02, 0.000e00],
                [1.450e-02, 0.000e00],
                [1.525e-02, 0.000e00],
                [1.600e-02, 0.000e00],
                [1.675e-02, 0.000e00],
                [1.750e-02, 0.000e00],
                [1.825e-02, 0.000e00],
                [1.900e-02, 0.000e00],
                [1.975e-02, 0.000e00],
                [2.050e-02, 0.000e00],
                [2.125e-02, 0.000e00],
                [2.200e-02, 0.000e00],
                [2.275e-02, 0.000e00],
                [2.350e-02, 0.000e00],
                [2.425e-02, 0.000e00],
            ],
            "tx_offsets": [[12.5, 0, 26.8]],
            "bird_offset": [12.5, 0, 26.8],
            "comment": "normalization accounts for a loop 961 m**2 area * 4 turns * 363 A current, nanoTesla",
            "normalization": [7.167e-7, 1e-9],
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "data_type": "dBzdt",
        },
        "Hummingbird": {
            "type": "frequency",
            "flag": "CPI880",
            "channel_start_index": 0,
            "channels": {
                "CPI880": 880,
                "CPI6600": 6600,
                "CPI34K": 34000,
                "CPQ880": 880,
                "CPQ6600": 6600,
                "CPQ34K": 34000,
            },
            "components": {
                "CPI880": "real",
                "CPQ880": "imag",
                "CPI6600": "real",
                "CPQ6600": "imag",
                "CPI34K": "real",
                "CPQ34K": "imag",
            },
            "tx_offsets": [
                [6.025, 0, 0],
                [6.2, 0, 0],
                [4.87, 0, 0],
                [6.025, 0, 0],
                [6.2, 0, 0],
                [4.87, 0, 0],
            ],
            "bird_offset": [0, 0, 0],
            "uncertainty": [
                [0.0, 2],
                [0.0, 5],
                [0.0, 10],
                [0.0, 2],
                [0.0, 5],
                [0.0, 10],
            ],
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "normalization": "ppm",
        },
        "QUESTEM (1996)": {
            "type": "time",
            "flag": "EMX",
            "channel_start_index": 3,
            "channels": {
                "[1]": 90.3e-6,
                "[2]": 142.4e-6,
                "[3]": 0.2466e-3,
                "[4]": 0.3507e-3,
                "[5]": 0.4549e-3,
                "[6]": 0.5590e-3,
                "[7]": 0.6632e-3,
                "[8]": 0.8194e-3,
                "[9]": 1.0278e-3,
                "[10]": 1.2361e-3,
                "[11]": 1.4965e-3,
                "[12]": 1.7048e-3,
                "[13]": 1.9652e-3,
                "[14]": 2.2777e-3,
                "[15]": 2.7464e-3,
                "[16]": 3.2672e-3,
                "[17]": 3.7880e-3,
                "[18]": 4.2046e-3,
            },
            "uncertainty": [
                [0.05, 50],
                [0.05, 50],
                [0.05, 50],
                [0.05, 50],
                [0.05, 50],
                [0.05, 50],
                [0.05, 50],
                [0.05, 50],
                [0.05, 50],
                [0.05, 50],
                [0.05, 50],
                [0.05, 50],
                [0.05, 50],
                [0.05, 50],
                [0.05, 50],
                [0.05, 50],
                [0.05, 50],
                [0.05, 50],
            ],
            "waveform": [
                [-2.00e-03, 1e-4],
                [-1.95e-03, 7.80e-02],
                [-1.90e-03, 1.56e-01],
                [-1.85e-03, 2.33e-01],
                [-1.80e-03, 3.09e-01],
                [-1.75e-03, 3.83e-01],
                [-1.70e-03, 4.54e-01],
                [-1.65e-03, 5.22e-01],
                [-1.60e-03, 5.88e-01],
                [-1.55e-03, 6.49e-01],
                [-1.50e-03, 7.07e-01],
                [-1.45e-03, 7.60e-01],
                [-1.40e-03, 8.09e-01],
                [-1.35e-03, 8.53e-01],
                [-1.30e-03, 8.91e-01],
                [-1.25e-03, 9.24e-01],
                [-1.20e-03, 9.51e-01],
                [-1.15e-03, 9.72e-01],
                [-1.10e-03, 9.88e-01],
                [-1.05e-03, 9.97e-01],
                [-1.00e-03, 1.00e00],
                [-9.50e-04, 9.97e-01],
                [-9.00e-04, 9.88e-01],
                [-8.50e-04, 9.72e-01],
                [-8.00e-04, 9.51e-01],
                [-7.50e-04, 9.24e-01],
                [-7.00e-04, 8.91e-01],
                [-6.50e-04, 8.53e-01],
                [-6.00e-04, 8.09e-01],
                [-5.50e-04, 7.60e-01],
                [-5.00e-04, 7.07e-01],
                [-4.50e-04, 6.49e-01],
                [-4.00e-04, 5.88e-01],
                [-3.50e-04, 5.22e-01],
                [-3.00e-04, 4.54e-01],
                [-2.50e-04, 3.83e-01],
                [-2.00e-04, 3.09e-01],
                [-1.50e-04, 2.33e-01],
                [-1.00e-04, 1.56e-01],
                [-5.00e-05, 7.80e-02],
                [0.00e00, 0.00e00],
                [1.50e-04, 0.00e00],
                [3.00e-04, 0.00e00],
                [4.50e-04, 0.00e00],
                [6.00e-04, 0.00e00],
                [7.50e-04, 0.00e00],
                [9.00e-04, 0.00e00],
                [1.05e-03, 0.00e00],
                [1.20e-03, 0.00e00],
                [1.35e-03, 0.00e00],
                [1.50e-03, 0.00e00],
                [1.65e-03, 0.00e00],
                [1.80e-03, 0.00e00],
                [1.95e-03, 0.00e00],
                [2.10e-03, 0.00e00],
                [2.25e-03, 0.00e00],
                [2.40e-03, 0.00e00],
                [2.55e-03, 0.00e00],
                [2.70e-03, 0.00e00],
                [2.85e-03, 0.00e00],
                [3.00e-03, 0.00e00],
                [3.15e-03, 0.00e00],
                [3.30e-03, 0.00e00],
                [3.45e-03, 0.00e00],
                [3.60e-03, 0.00e00],
                [3.75e-03, 0.00e00],
                [3.90e-03, 0.00e00],
                [4.05e-03, 0.00e00],
                [4.20e-03, 0.00e00],
                [4.35e-03, 0.00e00],
            ],
            "tx_offsets": [[127, 0, -75]],
            "bird_offset": [127, 0, -75],
            "comment": "normalization accounts for unit dipole moment at the tx_offset, in part-per-million",
            "normalization": "ppm",
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "data_type": "Bz",
        },
        "Resolve": {
            "type": "frequency",
            "flag": "CPI400",
            "channel_start_index": 0,
            "channels": {
                "CPI400": 385,
                "CPI1800": 1790,
                "CPI8200": 8208,
                "CPI40K": 39840,
                "CPI140K": 132660,
                "CPQ400": 385,
                "CPQ1800": 1790,
                "CPQ8200": 8208,
                "CPQ40K": 39840,
                "CPQ140K": 132660,
            },
            "components": {
                "CPI400": "real",
                "CPQ400": "imag",
                "CPI1800": "real",
                "CPQ1800": "imag",
                "CPI8200": "real",
                "CPQ8200": "imag",
                "CPI40K": "real",
                "CPQ40K": "imag",
                "CPI140K": "real",
                "CPQ140K": "imag",
            },
            "tx_offsets": [
                [7.86, 0, 0],
                [7.86, 0, 0],
                [7.86, 0, 0],
                [7.86, 0, 0],
                [7.86, 0, 0],
                [7.86, 0, 0],
                [7.86, 0, 0],
                [7.86, 0, 0],
                [7.86, 0, 0],
                [7.86, 0, 0],
            ],
            "bird_offset": [0, 0, 0],
            "uncertainty": [
                [0.0, 7.5],
                [0.0, 25],
                [0.0, 125],
                [0.0, 350],
                [0.0, 800],
                [0.0, 15],
                [0.0, 50],
                [0.0, 200],
                [0.0, 475],
                [0.0, 350],
            ],
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "normalization": "ppm",
        },
        "SandersGFEM": {
            "type": "frequency",
            "flag": "P9",
            "channel_start_index": 1,
            "channels": {
                "P912": 912,
                "P3005": 3005,
                "P11962": 11962,
                "P24510": 24510,
                "Q912": 912,
                "Q3005": 3005,
                "Q11962": 11962,
                "Q24510": 24510,
            },
            "components": {
                "P912": "real",
                "P3005": "real",
                "P11962": "real",
                "P24510": "real",
                "Q912": "imag",
                "Q3005": "imag",
                "Q11962": "imag",
                "Q24510": "imag",
            },
            "tx_offsets": [
                [21.35, 0, 0],
                [21.35, 0, 0],
                [21.38, 0, 0],
                [21.38, 0, 0],
                [21.35, 0, 0],
                [21.35, 0, 0],
                [21.38, 0, 0],
                [21.38, 0, 0],
            ],
            "bird_offset": [0, 0, 0],
            "uncertainty": [
                [0.0, 75],
                [0.0, 150],
                [0.0, 500],
                [0.0, 800],
                [0.0, 125],
                [0.0, 300],
                [0.0, 500],
                [0.0, 500],
            ],
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "normalization": "ppm",
        },
        "Skytem 304M (HM)": {
            "type": "time",
            "flag": "HM",
            "channel_start_index": 8,
            "channels": {
                "[1]": 0.43e-6,
                "[2]": 1.43e-6,
                "[3]": 3.43e-6,
                "[4]": 5.43e-6,
                "[5]": 7.43e-6,
                "[6]": 9.43e-6,
                "[7]": 11.43e-6,
                "[8]": 13.43e-6,
                "[9]": 16.43e-6,
                "[10]": 20.43e-6,
                "[11]": 25.43e-6,
                "[12]": 31.43e-6,
                "[13]": 39.43e-6,
                "[14]": 49.43e-6,
                "[15]": 62.43e-6,
                "[16]": 78.43e-6,
                "[17]": 98.43e-6,
                "[18]": 123.43e-6,
                "[19]": 154.43e-6,
                "[20]": 194.43e-6,
                "[21]": 245.43e-6,
                "[22]": 308.43e-6,
                "[23]": 389.43e-6,
                "[24]": 490.43e-6,
                "[25]": 617.43e-6,
                "[26]": 778.43e-6,
                "[27]": 980.43e-6,
                "[28]": 1235.43e-6,
                "[29]": 1557.43e-6,
                "[30]": 1963.43e-6,
                "[31]": 2474.43e-6,
                "[32]": 3120.43e-6,
                "[33]": 3912.43e-6,
                "[34]": 4880.43e-6,
                "[35]": 6065.43e-6,
                "[36]": 7517.43e-6,
                "[37]": 9293.43e-6,
                "[38]": 11473.43e-6,
            },
            "uncertainty": [
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
            ],
            "waveform": [
                [-4.895e-03, 1.000e-05],
                [-4.640e-03, 9.24906690e-01],
                [-4.385e-03, 9.32202966e-01],
                [-4.130e-03, 9.39232247e-01],
                [-3.875e-03, 9.46198635e-01],
                [-3.620e-03, 9.53165023e-01],
                [-3.365e-03, 9.60131411e-01],
                [-3.110e-03, 9.67097799e-01],
                [-2.855e-03, 9.72342365e-01],
                [-2.600e-03, 9.76662739e-01],
                [-2.345e-03, 9.80880874e-01],
                [-2.090e-03, 9.84342079e-01],
                [-1.835e-03, 9.87803283e-01],
                [-1.580e-03, 9.91264488e-01],
                [-1.325e-03, 9.94371859e-01],
                [-1.070e-03, 9.97464146e-01],
                [-8.150e-04, 1.000e00],
                [-5.600e-04, 1.000e00],
                [-3.050e-04, 1.000e00],
                [-5.000e-05, 1.000e00],
                [-4.500e-05, 1.000e00],
                [-4.000e-05, 9.57188199e-01],
                [-3.500e-05, 8.40405552e-01],
                [-3.000e-05, 7.19785772e-01],
                [-2.500e-05, 5.99026151e-01],
                [-2.000e-05, 4.77564967e-01],
                [-1.500e-05, 3.56103783e-01],
                [-1.000e-05, 2.34673120e-01],
                [-5.000e-06, 1.13453783e-01],
                [0.0, 0.0],
                [5.000e-06, 0.0],
                [1.000e-05, 0.0],
                [1.500e-05, 0.0],
                [2.000e-05, 0.0],
                [2.500e-05, 0.0],
                [3.000e-05, 0.0],
                [3.500e-05, 0.0],
                [4.000e-05, 0.0],
                [4.500e-05, 0.0],
                [5.000e-05, 0.0],
                [5.500e-05, 0.0],
                [6.000e-05, 0.0],
                [6.500e-05, 0.0],
                [7.000e-05, 0.0],
                [7.500e-05, 0.0],
                [8.000e-05, 0.0],
                [8.500e-05, 0.0],
                [9.000e-05, 0.0],
                [9.500e-05, 0.0],
                [1.000e-04, 0.0],
                [1.050e-04, 0.0],
                [1.100e-04, 0.0],
                [1.150e-04, 0.0],
                [1.200e-04, 0.0],
                [1.250e-04, 0.0],
                [1.750e-04, 0.0],
                [2.250e-04, 0.0],
                [2.750e-04, 0.0],
                [3.250e-04, 0.0],
                [3.750e-04, 0.0],
                [4.250e-04, 0.0],
                [4.750e-04, 0.0],
                [5.250e-04, 0.0],
                [5.750e-04, 0.0],
                [6.250e-04, 0.0],
                [6.750e-04, 0.0],
                [7.250e-04, 0.0],
                [7.750e-04, 0.0],
                [8.250e-04, 0.0],
                [8.750e-04, 0.0],
                [9.250e-04, 0.0],
                [9.750e-04, 0.0],
                [1.025e-03, 0.0],
                [1.075e-03, 0.0],
                [1.125e-03, 0.0],
                [1.175e-03, 0.0],
                [1.225e-03, 0.0],
                [1.275e-03, 0.0],
                [1.325e-03, 0.0],
                [1.375e-03, 0.0],
                [1.425e-03, 0.0],
                [1.475e-03, 0.0],
                [1.775e-03, 0.0],
                [2.075e-03, 0.0],
                [2.375e-03, 0.0],
                [2.675e-03, 0.0],
                [2.975e-03, 0.0],
                [3.275e-03, 0.0],
                [3.575e-03, 0.0],
                [3.875e-03, 0.0],
                [4.175e-03, 0.0],
                [4.475e-03, 0.0],
                [4.775e-03, 0.0],
                [5.075e-03, 0.0],
                [5.375e-03, 0.0],
                [5.675e-03, 0.0],
                [5.975e-03, 0.0],
                [6.275e-03, 0.0],
                [6.575e-03, 0.0],
                [6.875e-03, 0.0],
                [7.175e-03, 0.0],
                [7.475e-03, 0.0],
                [7.775e-03, 0.0],
                [8.075e-03, 0.0],
                [8.375e-03, 0.0],
                [8.675e-03, 0.0],
                [8.975e-03, 0.0],
                [9.275e-03, 0.0],
                [9.575e-03, 0.0],
                [9.875e-03, 0.0],
                [1.0175e-02, 0.0],
                [1.0475e-02, 0.0],
                [1.0775e-02, 0.0],
                [1.1075e-02, 0.0],
                [1.1375e-02, 0.0],
                [1.1675e-02, 0.0],
                [1.1975e-02, 0.0],
            ],
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "tx_offsets": [[-13.25, 0, 2.0]],
            "bird_offset": [-13.25, 0, 2.0],
            "normalization": [1e-12],
            "data_type": "dBzdt",
        },
        "Skytem 306HP (LM)": {
            "type": "time",
            "flag": "LM",
            "channel_start_index": 16,
            "channels": {
                "[1]": 0.3e-6,
                "[2]": 1.0e-6,
                "[3]": 1.7e-6,
                "[4]": 2.4e-6,
                "[5]": 3.2e-6,
                "[6]": 4.0e-6,
                "[7]": 4.8e-6,
                "[8]": 5.7e-6,
                "[9]": 6.6e-6,
                "[10]": 7.6e-6,
                "[11]": 8.7e-6,
                "[12]": 9.8e-6,
                "[13]": 1.11e-5,
                "[14]": 1.25e-5,
                "[15]": 1.4e-5,
                "[16]": 1.57e-5,
                "[17]": 1.75e-5,
                "[18]": 1.94e-5,
                "[19]": 2.16e-5,
                "[20]": 2.40e-5,
                "[21]": 2.66e-5,
                "[22]": 2.95e-5,
                "[23]": 3.27e-5,
                "[24]": 3.63e-5,
                "[25]": 4.02e-5,
                "[26]": 4.45e-5,
                "[27]": 4.93e-5,
                "[28]": 5.45e-5,
                "[29]": 6.03e-5,
                "[30]": 6.67e-5,
                "[31]": 7.37e-5,
                "[32]": 8.15e-5,
                "[33]": 9.01e-5,
                "[34]": 9.96e-5,
                "[35]": 1.10e-4,
                "[36]": 1.22e-4,
                "[37]": 1.35e-4,
                "[38]": 1.49e-4,
                "[39]": 1.65e-4,
                "[40]": 1.82e-4,
                "[41]": 2.01e-4,
                "[42]": 2.22e-4,
                "[43]": 2.46e-5,
                "[44]": 2.71e-4,
                "[45]": 3.00e-4,
                "[46]": 3.32e-4,
                "[47]": 3.66e-4,
                "[48]": 4.05e-4,
                "[49]": 4.48e-4,
                "[50]": 4.92e-4,
            },
            "uncertainty": [
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
            ],
            "waveform": [
                [-9.68000000e-04, 1e-4],
                [-9.18000000e-04, 0.67193831],
                [-8.68000000e-04, 0.83631068],
                [-8.18000000e-04, 0.91287783],
                [-7.68000000e-04, 0.95157844],
                [-7.18000000e-04, 0.97608318],
                [-6.68000000e-04, 0.98498653],
                [-6.18000000e-04, 0.99388987],
                [-5.68000000e-04, 0.99553716],
                [-5.18000000e-04, 0.99593248],
                [-4.68000000e-04, 0.99632781],
                [-4.18000000e-04, 0.99672314],
                [-3.68000000e-04, 0.99711847],
                [-3.18000000e-04, 0.9975138],
                [-2.68000000e-04, 0.99790913],
                [-2.18000000e-04, 0.99830446],
                [-1.68000000e-04, 0.99869978],
                [-1.18000000e-04, 0.99909511],
                [-6.80000000e-05, 0.99949044],
                [-1.80000000e-05, 0.99988577],
                [-1.60000000e-05, 0.99990158],
                [-1.40000000e-05, 0.9999174],
                [-1.20000000e-05, 0.99993321],
                [-1.00000000e-05, 0.99994902],
                [-8.00000000e-06, 0.99996484],
                [-6.00000000e-06, 0.99998065],
                [-4.00000000e-06, 0.99999646],
                [-2.00000000e-06, 0.51096262],
                [0.00000000e00, 0.00000000e00],
                [2.00000000e-06, 0.00000000e00],
                [1.20000000e-05, 0.00000000e00],
                [2.20000000e-05, 0.00000000e00],
                [3.20000000e-05, 0.00000000e00],
                [4.20000000e-05, 0.00000000e00],
                [5.20000000e-05, 0.00000000e00],
                [6.20000000e-05, 0.00000000e00],
                [7.20000000e-05, 0.00000000e00],
                [8.20000000e-05, 0.00000000e00],
                [9.20000000e-05, 0.00000000e00],
                [1.02000000e-04, 0.00000000e00],
                [1.12000000e-04, 0.00000000e00],
                [1.22000000e-04, 0.00000000e00],
                [1.32000000e-04, 0.00000000e00],
                [1.42000000e-04, 0.00000000e00],
                [1.52000000e-04, 0.00000000e00],
                [1.62000000e-04, 0.00000000e00],
                [1.72000000e-04, 0.00000000e00],
                [1.82000000e-04, 0.00000000e00],
                [1.92000000e-04, 0.00000000e00],
                [2.02000000e-04, 0.00000000e00],
                [2.52000000e-04, 0.00000000e00],
                [3.02000000e-04, 0.00000000e00],
                [3.52000000e-04, 0.00000000e00],
                [4.02000000e-04, 0.00000000e00],
                [4.52000000e-04, 0.00000000e00],
                [5.02000000e-04, 0.00000000e00],
                [5.52000000e-04, 0.00000000e00],
                [6.02000000e-04, 0.00000000e00],
                [6.52000000e-04, 0.00000000e00],
                [7.02000000e-04, 0.00000000e00],
                [7.52000000e-04, 0.00000000e00],
                [8.02000000e-04, 0.00000000e00],
                [8.52000000e-04, 0.00000000e00],
                [9.02000000e-04, 0.00000000e00],
                [9.52000000e-04, 0.00000000e00],
                [1.00200000e-03, 0.00000000e00],
                [1.05200000e-03, 0.00000000e00],
                [1.10200000e-03, 0.00000000e00],
                [1.15200000e-03, 0.00000000e00],
                [1.20200000e-03, 0.00000000e00],
                [1.25200000e-03, 0.00000000e00],
                [1.30200000e-03, 0.00000000e00],
                [1.35200000e-03, 0.00000000e00],
                [1.40200000e-03, 0.00000000e00],
                [1.45200000e-03, 0.00000000e00],
                [1.50200000e-03, 0.00000000e00],
                [1.55200000e-03, 0.00000000e00],
                [1.60200000e-03, 0.00000000e00],
                [1.65200000e-03, 0.00000000e00],
                [1.70200000e-03, 0.00000000e00],
                [1.75200000e-03, 0.00000000e00],
                [1.80200000e-03, 0.00000000e00],
                [1.85200000e-03, 0.00000000e00],
                [1.90200000e-03, 0.00000000e00],
                [1.95200000e-03, 0.00000000e00],
            ],
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "tx_offsets": [[-13.25, 0, 2.0]],
            "bird_offset": [-13.25, 0, 2.0],
            "normalization": [1e-12],
            "data_type": "dBzdt",
        },
        "Skytem 306M HP (HM)": {
            "type": "time",
            "flag": "HM",
            "channel_start_index": 15,
            "channels": {
                "[1]": 7.0e-7,
                "[2]": 2.1e-5,
                "[3]": 3.6e-5,
                "[4]": 5.3e-5,
                "[5]": 7.2e-5,
                "[6]": 9.3e-5,
                "[7]": 1.18e-5,
                "[8]": 1.49e-5,
                "[9]": 1.85e-5,
                "[10]": 2.28e-5,
                "[11]": 2.81e-5,
                "[12]": 3.46e-5,
                "[13]": 4.25e-5,
                "[14]": 5.2e-5,
                "[15]": 6.36e-5,
                "[16]": 7.78e-5,
                "[17]": 9.51e-5,
                "[18]": 1.16e-4,
                "[19]": 1.42e-4,
                "[20]": 1.74e-4,
                "[21]": 2.12e-4,
                "[22]": 2.59e-4,
                "[23]": 3.17e-4,
                "[24]": 3.87e-4,
                "[25]": 4.72e-4,
                "[26]": 5.77e-4,
                "[27]": 7.05e-4,
                "[28]": 8.61e-4,
                "[29]": 1.05e-3,
                "[30]": 1.28e-3,
                "[31]": 1.57e-3,
                "[32]": 1.92e-3,
                "[33]": 2.34e-3,
                "[34]": 2.86e-3,
                "[35]": 3.49e-3,
                "[36]": 4.26e-3,
                "[37]": 5.21e-3,
                "[38]": 6.36e-3,
                "[39]": 7.77e-3,
                "[40]": 9.49e-3,
                "[41]": 1.11e-2,
            },
            "uncertainty": [
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
            ],
            "waveform": [
                [-4.895e-03, 0.93336193],
                [-4.640e-03, 0.93950274],
                [-4.385e-03, 0.94564356],
                [-4.130e-03, 0.95178438],
                [-3.875e-03, 0.95792519],
                [-3.620e-03, 0.96406601],
                [-3.365e-03, 0.97020682],
                [-3.110e-03, 0.97634764],
                [-2.855e-03, 0.98248846],
                [-2.600e-03, 0.98862927],
                [-2.345e-03, 0.9908726],
                [-2.090e-03, 0.99199595],
                [-1.835e-03, 0.9931193],
                [-1.580e-03, 0.99424264],
                [-1.325e-03, 0.99536599],
                [-1.070e-03, 0.99648934],
                [-8.150e-04, 0.99761269],
                [-5.600e-04, 0.99873604],
                [-3.050e-04, 0.99985938],
                [-5.000e-05, 0.17615372],
                [-4.500e-05, 0.15743786],
                [-4.000e-05, 0.13872199],
                [-3.500e-05, 0.12000613],
                [-3.000e-05, 0.10129026],
                [-2.500e-05, 0.0825744],
                [-2.000e-05, 0.06385853],
                [-1.500e-05, 0.0451426],
                [-1.000e-05, 0.00],
                [-5.000e-06, 0.00],
                [0.0, 0.0],
                [5.000e-06, 0.0],
                [1.000e-05, 0.0],
                [1.500e-05, 0.0],
                [2.000e-05, 0.0],
                [2.500e-05, 0.0],
                [3.000e-05, 0.0],
                [3.500e-05, 0.0],
                [4.000e-05, 0.0],
                [4.500e-05, 0.0],
                [5.000e-05, 0.0],
                [5.500e-05, 0.0],
                [6.000e-05, 0.0],
                [6.500e-05, 0.0],
                [7.000e-05, 0.0],
                [7.500e-05, 0.0],
                [8.000e-05, 0.0],
                [8.500e-05, 0.0],
                [9.000e-05, 0.0],
                [9.500e-05, 0.0],
                [1.000e-04, 0.0],
                [1.050e-04, 0.0],
                [1.100e-04, 0.0],
                [1.150e-04, 0.0],
                [1.200e-04, 0.0],
                [1.250e-04, 0.0],
                [1.750e-04, 0.0],
                [2.250e-04, 0.0],
                [2.750e-04, 0.0],
                [3.250e-04, 0.0],
                [3.750e-04, 0.0],
                [4.250e-04, 0.0],
                [4.750e-04, 0.0],
                [5.250e-04, 0.0],
                [5.750e-04, 0.0],
                [6.250e-04, 0.0],
                [6.750e-04, 0.0],
                [7.250e-04, 0.0],
                [7.750e-04, 0.0],
                [8.250e-04, 0.0],
                [8.750e-04, 0.0],
                [9.250e-04, 0.0],
                [9.750e-04, 0.0],
                [1.025e-03, 0.0],
                [1.075e-03, 0.0],
                [1.125e-03, 0.0],
                [1.175e-03, 0.0],
                [1.225e-03, 0.0],
                [1.275e-03, 0.0],
                [1.325e-03, 0.0],
                [1.375e-03, 0.0],
                [1.425e-03, 0.0],
                [1.475e-03, 0.0],
                [1.775e-03, 0.0],
                [2.075e-03, 0.0],
                [2.375e-03, 0.0],
                [2.675e-03, 0.0],
                [2.975e-03, 0.0],
                [3.275e-03, 0.0],
                [3.575e-03, 0.0],
                [3.875e-03, 0.0],
                [4.175e-03, 0.0],
                [4.475e-03, 0.0],
                [4.775e-03, 0.0],
                [5.075e-03, 0.0],
                [5.375e-03, 0.0],
                [5.675e-03, 0.0],
                [5.975e-03, 0.0],
                [6.275e-03, 0.0],
                [6.575e-03, 0.0],
                [6.875e-03, 0.0],
                [7.175e-03, 0.0],
                [7.475e-03, 0.0],
                [7.775e-03, 0.0],
                [8.075e-03, 0.0],
                [8.375e-03, 0.0],
                [8.675e-03, 0.0],
                [8.975e-03, 0.0],
                [9.275e-03, 0.0],
                [9.575e-03, 0.0],
                [9.875e-03, 0.0],
                [1.0175e-02, 0.0],
                [1.0475e-02, 0.0],
                [1.0775e-02, 0.0],
                [1.1075e-02, 0.0],
                [1.1375e-02, 0.0],
                [1.1675e-02, 0.0],
                [1.1975e-02, 0.0],
            ],
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "tx_offsets": [[-13.25, 0, 2.0]],
            "bird_offset": [-13.25, 0, 2.0],
            "normalization": [1e-12],
            "data_type": "dBzdt",
        },
        "Skytem 312HP (HM)": {
            "type": "time",
            "flag": "HM",
            "channel_start_index": 10,
            "channels": {
                "[1]": -1.7850e-06,
                "[2]": 2.850e-07,
                "[3]": 1.7150e-06,
                "[4]": 3.7150e-06,
                "[5]": 5.7150e-06,
                "[6]": 7.7150e-06,
                "[7]": 9.7150e-06,
                "[8]": 1.2215e-05,
                "[9]": 1.5715e-05,
                "[10]": 2.0215e-05,
                "[11]": 2.5715e-05,
                "[12]": 3.2715e-05,
                "[13]": 4.1715e-05,
                "[14]": 5.3215e-05,
                "[15]": 6.7715e-05,
                "[16]": 8.5715e-05,
                "[17]": 1.082150e-04,
                "[18]": 1.362150e-04,
                "[19]": 1.717150e-04,
                "[20]": 2.172150e-04,
                "[21]": 2.742150e-04,
                "[22]": 3.462150e-04,
                "[23]": 4.372150e-04,
                "[24]": 5.512150e-04,
                "[25]": 6.952150e-04,
                "[26]": 8.767150e-04,
                "[27]": 1.1052150e-03,
                "[28]": 1.3937150e-03,
                "[29]": 1.7577150e-03,
                "[30]": 2.2162150e-03,
                "[31]": 2.7947150e-03,
                "[32]": 3.5137150e-03,
                "[33]": 4.3937150e-03,
                "[34]": 5.4702150e-03,
                "[35]": 6.7887150e-03,
                "[36]": 8.4027150e-03,
                "[37]": 1.0380715e-02,
            },
            "uncertainty": [
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
                [0.05, 1e-2],
            ],
            "waveform": [
                [-4.0e-03, 1.0e-08],
                [-3.90e-03, 1.11e-01],
                [-3.80e-03, 2.22e-01],
                [-3.70e-03, 3.33e-01],
                [-3.60e-03, 4.44e-01],
                [-3.50e-03, 5.55e-01],
                [-3.40e-03, 6.66e-01],
                [-3.30e-03, 7.77e-01],
                [-3.20e-03, 8.88e-01],
                [-3.10e-03, 1.0e00],
                [-3.0e-03, 1.0e00],
                [-2.90e-03, 1.0e00],
                [-2.80e-03, 1.0e00],
                [-2.70e-03, 1.0e00],
                [-2.60e-03, 1.0e00],
                [-2.50e-03, 1.0e00],
                [-2.40e-03, 1.0e00],
                [-2.30e-03, 1.0e00],
                [-2.20e-03, 1.0e00],
                [-2.10e-03, 1.0e00],
                [-2.0e-03, 1.0e00],
                [-1.90e-03, 1.0e00],
                [-1.80e-03, 1.0e00],
                [-1.70e-03, 1.0e00],
                [-1.60e-03, 1.0e00],
                [-1.50e-03, 1.0e00],
                [-1.40e-03, 1.0e00],
                [-1.30e-03, 1.0e00],
                [-1.20e-03, 1.0e00],
                [-1.10e-03, 1.0e00],
                [-1.0e-03, 1.0e00],
                [-9.0e-04, 9.0e-01],
                [-8.0e-04, 8.0e-01],
                [-7.0e-04, 7.0e-01],
                [-6.0e-04, 6.0e-01],
                [-5.0e-04, 5.0e-01],
                [-4.0e-04, 4.0e-01],
                [-3.0e-04, 3.0e-01],
                [-2.0e-04, 2.0e-01],
                [-1.0e-04, 1.0e-01],
                [0.0e-00, 0.0e00],
                [2.0e-05, 0.0e00],
                [4.0e-05, 0.0e00],
                [6.0e-05, 0.0e00],
                [8.0e-05, 0.0e00],
                [1.0e-04, 0.0e00],
                [1.20e-04, 0.0e00],
                [1.40e-04, 0.0e00],
                [1.60e-04, 0.0e00],
                [1.80e-04, 0.0e00],
                [2.0e-04, 0.0e00],
                [2.60e-04, 0.0e00],
                [3.20e-04, 0.0e00],
                [3.80e-04, 0.0e00],
                [4.40e-04, 0.0e00],
                [5.0e-04, 0.0e00],
                [5.60e-04, 0.0e00],
                [6.20e-04, 0.0e00],
                [6.80e-04, 0.0e00],
                [7.40e-04, 0.0e00],
                [8.0e-04, 0.0e00],
                [8.60e-04, 0.0e00],
                [9.20e-04, 0.0e00],
                [9.80e-04, 0.0e00],
                [1.04e-03, 0.0e00],
                [1.10e-03, 0.0e00],
                [1.16e-03, 0.0e00],
                [1.20e-03, 0.0e00],
                [1.70e-03, 0.0e00],
                [2.20e-03, 0.0e00],
                [2.70e-03, 0.0e00],
                [3.20e-03, 0.0e00],
                [3.70e-03, 0.0e00],
                [4.20e-03, 0.0e00],
                [4.70e-03, 0.0e00],
                [5.20e-03, 0.0e00],
                [5.70e-03, 0.0e00],
                [6.20e-03, 0.0e00],
                [6.70e-03, 0.0e00],
                [7.20e-03, 0.0e00],
                [7.70e-03, 0.0e00],
                [8.20e-03, 0.0e00],
                [8.70e-03, 0.0e00],
                [9.20e-03, 0.0e00],
                [9.70e-03, 0.0e00],
                [1.02e-02, 0.0e00],
                [1.07e-02, 0.0e00],
            ],
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "tx_offsets": [[-13.25, 0, 2.0]],
            "bird_offset": [-13.25, 0, 2.0],
            "normalization": [1e-12],
            "data_type": "dBzdt",
        },
        "Skytem 312HP v2 (HM)": {
            "type": "time",
            "flag": "HM",
            "channel_start_index": 18,
            "channels": {
                "[1]": 3.0275e-5,
                "[2]": 3.1775e-5,
                "[3]": 3.3775e-5,
                "[4]": 3.5776e-5,
                "[5]": 3.7776e-5,
                "[6]": 3.9770e-5,
                "[7]": 4.1770e-5,
                "[8]": 4.4270e-5,
                "[9]": 4.7770e-5,
                "[10]": 5.227e-5,
                "[11]": 5.777e-5,
                "[12]": 6.477e-5,
                "[13]": 7.377e-5,
                "[14]": 8.527e-5,
                "[15]": 9.977e-5,
                "[16]": 0.00011777,
                "[17]": 0.00014026,
                "[18]": 0.00016826,
                "[19]": 0.00020376,
                "[20]": 0.00024926,
                "[21]": 0.00030626,
                "[22]": 0.00037826,
                "[23]": 0.00046926,
                "[24]": 0.00058325,
                "[25]": 0.00072726,
                "[26]": 0.00090876,
                "[27]": 0.00113656,
                "[28]": 0.00142556,
                "[29]": 0.00178956,
                "[30]": 0.00224756,
                "[31]": 0.00282656,
                "[32]": 0.00354556,
                "[33]": 0.00442556,
                "[34]": 0.00550156,
                "[35]": 0.00582056,
                "[36]": 0.00843456,
                "[37]": 0.01041256,
            },
            "uncertainty": [
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
            ],
            "waveform": [
                [-4.0e-03, 1.0e-08],
                [-3.90e-03, 1.11e-01],
                [-3.80e-03, 2.22e-01],
                [-3.70e-03, 3.33e-01],
                [-3.60e-03, 4.44e-01],
                [-3.50e-03, 5.55e-01],
                [-3.40e-03, 6.66e-01],
                [-3.30e-03, 7.77e-01],
                [-3.20e-03, 8.88e-01],
                [-3.10e-03, 1.0e00],
                [-3.0e-03, 1.0e00],
                [-2.90e-03, 1.0e00],
                [-2.80e-03, 1.0e00],
                [-2.70e-03, 1.0e00],
                [-2.60e-03, 1.0e00],
                [-2.50e-03, 1.0e00],
                [-2.40e-03, 1.0e00],
                [-2.30e-03, 1.0e00],
                [-2.20e-03, 1.0e00],
                [-2.10e-03, 1.0e00],
                [-2.0e-03, 1.0e00],
                [-1.90e-03, 1.0e00],
                [-1.80e-03, 1.0e00],
                [-1.70e-03, 1.0e00],
                [-1.60e-03, 1.0e00],
                [-1.50e-03, 1.0e00],
                [-1.40e-03, 1.0e00],
                [-1.30e-03, 1.0e00],
                [-1.20e-03, 1.0e00],
                [-1.10e-03, 1.0e00],
                [-1.0e-03, 1.0e00],
                [-9.0e-04, 1.0e00],
                [-8.0e-04, 1.0e00],
                [-7.0e-04, 1.0e00],
                [-6.0e-04, 1.0e00],
                [-5.0e-04, 1.0e00],
                [-4.0e-04, 1.0e00],
                [-3.0e-04, 1.0e00],
                [-2.0e-04, 6.66e-01],
                [-1.0e-04, 3.33e-01],
                [0.00e00, 0.00e00],
                [5.00e-05, 0.00e00],
                [1.00e-04, 0.00e00],
                [1.50e-04, 0.00e00],
                [2.00e-04, 0.00e00],
                [2.50e-04, 0.00e00],
                [3.00e-04, 0.00e00],
                [3.50e-04, 0.00e00],
                [4.00e-04, 0.00e00],
                [4.50e-04, 0.00e00],
                [5.00e-04, 0.00e00],
                [5.50e-04, 0.00e00],
                [6.00e-04, 0.00e00],
                [6.50e-04, 0.00e00],
                [7.00e-04, 0.00e00],
                [7.50e-04, 0.00e00],
                [8.00e-04, 0.00e00],
                [8.50e-04, 0.00e00],
                [9.00e-04, 0.00e00],
                [9.50e-04, 0.00e00],
                [1.00e-03, 0.00e00],
                [1.05e-03, 0.00e00],
                [1.10e-03, 0.00e00],
                [1.15e-03, 0.00e00],
                [1.20e-03, 0.00e00],
                [1.70e-03, 0.00e00],
                [2.20e-03, 0.00e00],
                [2.70e-03, 0.00e00],
                [3.20e-03, 0.00e00],
                [3.70e-03, 0.00e00],
                [4.20e-03, 0.00e00],
                [4.70e-03, 0.00e00],
                [5.20e-03, 0.00e00],
                [5.70e-03, 0.00e00],
                [6.20e-03, 0.00e00],
                [6.70e-03, 0.00e00],
                [7.20e-03, 0.00e00],
                [7.70e-03, 0.00e00],
                [8.20e-03, 0.00e00],
                [8.70e-03, 0.00e00],
                [9.20e-03, 0.00e00],
                [9.70e-03, 0.00e00],
                [1.02e-02, 0.00e00],
            ],
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "tx_offsets": [[-13.25, 0, 2.0]],
            "bird_offset": [-13.25, 0, 2.0],
            "normalization": [1e-12],
            "data_type": "dBzdt",
        },
        "Skytem 312HP v3 (HM)": {
            "type": "time",
            "flag": "HM",
            "channel_start_index": 18,
            "channels": {
                "[1]": 7.1499e-07,
                "[2]": 2.2149e-06,
                "[3]": 4.2149e-06,
                "[4]": 6.2149e-06,
                "[5]": 8.2149e-06,
                "[6]": 1.02144e-05,
                "[7]": 1.22144e-05,
                "[8]": 1.47145e-05,
                "[9]": 1.82149e-05,
                "[10]": 2.2715e-05,
                "[11]": 2.8215e-05,
                "[12]": 3.5215e-05,
                "[13]": 4.4215e-05,
                "[14]": 5.57149e-05,
                "[15]": 7.02149e-05,
                "[16]": 8.82149e-05,
                "[17]": 0.0001107149,
                "[18]": 0.0001387149,
                "[19]": 0.0001742150,
                "[20]": 0.0002197150,
                "[21]": 0.000276715,
                "[22]": 0.000348715,
                "[23]": 0.000439715,
                "[24]": 0.000553715,
                "[25]": 0.000697715,
                "[26]": 0.000879215,
                "[27]": 0.001107715,
                "[28]": 0.001396215,
                "[29]": 0.001760215,
                "[30]": 0.002218715,
                "[31]": 0.002797215,
                "[32]": 0.003516215,
                "[33]": 0.004396215,
                "[34]": 0.005472715,
                "[35]": 0.006791215,
                "[36]": 0.008405215,
                "[37]": 0.010383215,
            },
            "uncertainty": [
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
                [0.1, 1e-2],
            ],
            "waveform": [
                [-4.0e-03, 1.0e-08],
                [-3.90e-03, 1.11e-01],
                [-3.80e-03, 2.22e-01],
                [-3.70e-03, 3.33e-01],
                [-3.60e-03, 4.44e-01],
                [-3.50e-03, 5.55e-01],
                [-3.40e-03, 6.66e-01],
                [-3.30e-03, 7.77e-01],
                [-3.20e-03, 8.88e-01],
                [-3.10e-03, 1.0e00],
                [-3.0e-03, 1.0e00],
                [-2.90e-03, 1.0e00],
                [-2.80e-03, 1.0e00],
                [-2.70e-03, 1.0e00],
                [-2.60e-03, 1.0e00],
                [-2.50e-03, 1.0e00],
                [-2.40e-03, 1.0e00],
                [-2.30e-03, 1.0e00],
                [-2.20e-03, 1.0e00],
                [-2.10e-03, 1.0e00],
                [-2.0e-03, 1.0e00],
                [-1.90e-03, 1.0e00],
                [-1.80e-03, 1.0e00],
                [-1.70e-03, 1.0e00],
                [-1.60e-03, 1.0e00],
                [-1.50e-03, 1.0e00],
                [-1.40e-03, 1.0e00],
                [-1.30e-03, 1.0e00],
                [-1.20e-03, 1.0e00],
                [-1.10e-03, 1.0e00],
                [-1.0e-03, 1.0e00],
                [-9.0e-04, 1.0e00],
                [-8.0e-04, 1.0e00],
                [-7.0e-04, 1.0e00],
                [-6.0e-04, 1.0e00],
                [-5.0e-04, 1.0e00],
                [-4.0e-04, 1.0e00],
                [-3.0e-04, 1.0e00],
                [-2.0e-04, 6.66e-01],
                [-1.0e-04, 3.33e-01],
                [0.00e00, 0.00e00],
                [5.00e-05, 0.00e00],
                [1.00e-04, 0.00e00],
                [1.50e-04, 0.00e00],
                [2.00e-04, 0.00e00],
                [2.50e-04, 0.00e00],
                [3.00e-04, 0.00e00],
                [3.50e-04, 0.00e00],
                [4.00e-04, 0.00e00],
                [4.50e-04, 0.00e00],
                [5.00e-04, 0.00e00],
                [5.50e-04, 0.00e00],
                [6.00e-04, 0.00e00],
                [6.50e-04, 0.00e00],
                [7.00e-04, 0.00e00],
                [7.50e-04, 0.00e00],
                [8.00e-04, 0.00e00],
                [8.50e-04, 0.00e00],
                [9.00e-04, 0.00e00],
                [9.50e-04, 0.00e00],
                [1.00e-03, 0.00e00],
                [1.05e-03, 0.00e00],
                [1.10e-03, 0.00e00],
                [1.15e-03, 0.00e00],
                [1.20e-03, 0.00e00],
                [1.70e-03, 0.00e00],
                [2.20e-03, 0.00e00],
                [2.70e-03, 0.00e00],
                [3.20e-03, 0.00e00],
                [3.70e-03, 0.00e00],
                [4.20e-03, 0.00e00],
                [4.70e-03, 0.00e00],
                [5.20e-03, 0.00e00],
                [5.70e-03, 0.00e00],
                [6.20e-03, 0.00e00],
                [6.70e-03, 0.00e00],
                [7.20e-03, 0.00e00],
                [7.70e-03, 0.00e00],
                [8.20e-03, 0.00e00],
                [8.70e-03, 0.00e00],
                [9.20e-03, 0.00e00],
                [9.70e-03, 0.00e00],
                [1.02e-02, 0.00e00],
            ],
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "tx_offsets": [[-13.25, 0, 2.0]],
            "bird_offset": [-13.25, 0, 2.0],
            "normalization": [1e-12],
            "data_type": "dBzdt",
        },
        "Skytem 312HP v2 (LM)": {
            "type": "time",
            "flag": "LM",
            "channel_start_index": 10,
            "channels": {
                "[1]": -1.73922e-05,
                "[2]": -1.58923e-05,
                "[3]": -1.38922e-05,
                "[4]": -1.18912e-05,
                "[5]": -9.891e-06,
                "[6]": -7.897e-06,
                "[7]": -5.897e-06,
                "[8]": -3.397e-06,
                "[9]": 1.03e-07,
                "[10]": 4.603e-06,
                "[11]": 1.0103e-05,
                "[12]": 1.7103e-05,
                "[13]": 2.6103e-05,
                "[14]": 3.7603e-05,
                "[15]": 5.2103e-05,
                "[16]": 7.0103e-05,
                "[17]": 9.2593e-05,
                "[18]": 0.000120593,
                "[19]": 0.000156093,
                "[20]": 0.000201593,
                "[21]": 0.000258593,
                "[22]": 0.000330593,
                "[23]": 0.000421593,
                "[24]": 0.000535593,
                "[25]": 0.000679593,
                "[26]": 0.000861,
                "[27]": 0.0011,
                "[28]": 0.001377893,
                "[29]": 0.001741893,
                "[30]": 0.002199893,
                "[31]": 0.002778893,
                "[32]": 0.003497893,
                "[33]": 0.004377893,
                "[34]": 0.00545389,
                "[35]": 0.005772893,
                "[36]": 0.008386893,
                "[37]": 0.010364893,
            },
            "uncertainty": [
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
            ],
            "waveform": [
                [-8.18000000e-04, 1.92433144e-04],
                [-7.68000000e-04, 8.36713627e-02],
                [-7.18000000e-04, 1.51880444e-01],
                [-6.68000000e-04, 2.20089525e-01],
                [-6.18000000e-04, 2.82624877e-01],
                [-5.68000000e-04, 3.40805095e-01],
                [-5.18000000e-04, 3.98985314e-01],
                [-4.68000000e-04, 4.57165532e-01],
                [-4.18000000e-04, 5.17091331e-01],
                [-3.68000000e-04, 5.77759762e-01],
                [-3.18000000e-04, 6.38428192e-01],
                [-2.68000000e-04, 6.99096623e-01],
                [-2.18000000e-04, 7.59765054e-01],
                [-1.68000000e-04, 8.20433484e-01],
                [-1.18000000e-04, 8.81101915e-01],
                [-6.80000000e-05, 9.40674793e-01],
                [-1.80000000e-05, 9.98704760e-01],
                [-1.60000000e-05, 8.06732495e-01],
                [-1.40000000e-05, 5.13343178e-01],
                [-1.20000000e-05, 2.70179503e-01],
                [-1.00000000e-05, 1.07502126e-01],
                [-8.00000000e-06, 2.85859885e-02],
                [-6.00000000e-06, 2.21551233e-02],
                [-4.00000000e-06, 2.71962192e-02],
                [-2.00000000e-06, 1.43181338e-02],
                [0.00000000e00, 0.00000000e00],
                [2.00000000e-06, 0.00000000e00],
                [1.20000000e-05, 0.00000000e00],
                [2.20000000e-05, 0.00000000e00],
                [3.20000000e-05, 0.00000000e00],
                [4.20000000e-05, 0.00000000e00],
                [5.20000000e-05, 0.00000000e00],
                [6.20000000e-05, 0.00000000e00],
                [7.20000000e-05, 0.00000000e00],
                [8.20000000e-05, 0.00000000e00],
                [9.20000000e-05, 0.00000000e00],
                [1.02000000e-04, 0.00000000e00],
                [1.12000000e-04, 0.00000000e00],
                [1.22000000e-04, 0.00000000e00],
                [1.32000000e-04, 0.00000000e00],
                [1.42000000e-04, 0.00000000e00],
                [1.52000000e-04, 0.00000000e00],
                [1.62000000e-04, 0.00000000e00],
                [1.72000000e-04, 0.00000000e00],
                [1.82000000e-04, 0.00000000e00],
                [1.92000000e-04, 0.00000000e00],
                [2.02000000e-04, 0.00000000e00],
                [2.52000000e-04, 0.00000000e00],
                [3.02000000e-04, 0.00000000e00],
                [3.52000000e-04, 0.00000000e00],
                [4.02000000e-04, 0.00000000e00],
                [4.52000000e-04, 0.00000000e00],
                [5.02000000e-04, 0.00000000e00],
                [5.52000000e-04, 0.00000000e00],
                [6.02000000e-04, 0.00000000e00],
                [6.52000000e-04, 0.00000000e00],
                [7.02000000e-04, 0.00000000e00],
                [7.52000000e-04, 0.00000000e00],
                [8.02000000e-04, 0.00000000e00],
                [8.52000000e-04, 0.00000000e00],
                [9.02000000e-04, 0.00000000e00],
                [9.52000000e-04, 0.00000000e00],
                [1.00200000e-03, 0.00000000e00],
                [1.05200000e-03, 0.00000000e00],
                [1.10200000e-03, 0.00000000e00],
                [1.15200000e-03, 0.00000000e00],
                [1.20200000e-03, 0.00000000e00],
                [1.25200000e-03, 0.00000000e00],
                [1.30200000e-03, 0.00000000e00],
                [1.35200000e-03, 0.00000000e00],
                [1.40200000e-03, 0.00000000e00],
                [1.45200000e-03, 0.00000000e00],
                [1.50200000e-03, 0.00000000e00],
                [1.55200000e-03, 0.00000000e00],
                [1.60200000e-03, 0.00000000e00],
                [1.65200000e-03, 0.00000000e00],
                [1.70200000e-03, 0.00000000e00],
                [1.75200000e-03, 0.00000000e00],
                [1.80200000e-03, 0.00000000e00],
                [1.85200000e-03, 0.00000000e00],
                [1.90200000e-03, 0.00000000e00],
                [1.95200000e-03, 0.00000000e00],
            ],
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "tx_offsets": [[-13.25, 0, 2.0]],
            "bird_offset": [-13.25, 0, 2.0],
            "normalization": [1e-12],
            "data_type": "dBzdt",
        },
        "Skytem 312HP v3 (LM)": {
            "type": "time",
            "flag": "LM",
            "channel_start_index": 8,
            "channels": {
                "[1]": -1.1485e-05,
                "[2]": -9.985998e-06,
                "[3]": -7.985999e-06,
                "[4]": -5.9859994e-06,
                "[5]": -3.985999e-06,
                "[6]": -1.985999e-06,
                "[7]": 1.5e-08,
                "[8]": 2.515e-06,
                "[9]": 6.015e-06,
                "[10]": 1.0515e-05,
                "[11]": 1.6015998e-05,
                "[12]": 2.3015e-05,
                "[13]": 3.2015e-05,
                "[14]": 4.3515e-05,
                "[15]": 5.8015e-05,
                "[16]": 7.6015e-05,
                "[17]": 9.8515e-05,
                "[18]": 0.000126515,
                "[19]": 0.000162015,
                "[20]": 0.000207515,
                "[21]": 0.000264515,
                "[22]": 0.00033651596,
                "[23]": 0.00042751595,
                "[24]": 0.000541515,
                "[25]": 0.0006855159,
                "[26]": 0.000867015,
                "[27]": 0.0010955158,
                "[28]": 0.0013840157,
            },
            "uncertainty": [
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
                [0.1, 1e-1],
            ],
            "waveform": [
                [-8.18000000e-04, 1.92433144e-04],
                [-7.68000000e-04, 8.36713627e-02],
                [-7.18000000e-04, 1.51880444e-01],
                [-6.68000000e-04, 2.20089525e-01],
                [-6.18000000e-04, 2.82624877e-01],
                [-5.68000000e-04, 3.40805095e-01],
                [-5.18000000e-04, 3.98985314e-01],
                [-4.68000000e-04, 4.57165532e-01],
                [-4.18000000e-04, 5.17091331e-01],
                [-3.68000000e-04, 5.77759762e-01],
                [-3.18000000e-04, 6.38428192e-01],
                [-2.68000000e-04, 6.99096623e-01],
                [-2.18000000e-04, 7.59765054e-01],
                [-1.68000000e-04, 8.20433484e-01],
                [-1.18000000e-04, 8.81101915e-01],
                [-6.80000000e-05, 9.40674793e-01],
                [-1.80000000e-05, 9.98704760e-01],
                [-1.60000000e-05, 8.06732495e-01],
                [-1.40000000e-05, 5.13343178e-01],
                [-1.20000000e-05, 2.70179503e-01],
                [-1.00000000e-05, 1.07502126e-01],
                [-8.00000000e-06, 2.85859885e-02],
                [-6.00000000e-06, 2.21551233e-02],
                [-4.00000000e-06, 2.71962192e-02],
                [-2.00000000e-06, 1.43181338e-02],
                [0.00000000e00, 0.00000000e00],
                [2.00000000e-06, 0.00000000e00],
                [4.00000000e-06, 0.00000000e00],
                [6.00000000e-06, 0.00000000e00],
                [8.00000000e-06, 0.00000000e00],
                [1.00000000e-05, 0.00000000e00],
                [1.20000000e-05, 0.00000000e00],
                [2.20000000e-05, 0.00000000e00],
                [3.20000000e-05, 0.00000000e00],
                [4.20000000e-05, 0.00000000e00],
                [5.20000000e-05, 0.00000000e00],
                [6.20000000e-05, 0.00000000e00],
                [7.20000000e-05, 0.00000000e00],
                [8.20000000e-05, 0.00000000e00],
                [9.20000000e-05, 0.00000000e00],
                [1.02000000e-04, 0.00000000e00],
                [1.12000000e-04, 0.00000000e00],
                [1.22000000e-04, 0.00000000e00],
                [1.32000000e-04, 0.00000000e00],
                [1.42000000e-04, 0.00000000e00],
                [1.52000000e-04, 0.00000000e00],
                [1.62000000e-04, 0.00000000e00],
                [1.72000000e-04, 0.00000000e00],
                [1.82000000e-04, 0.00000000e00],
                [1.92000000e-04, 0.00000000e00],
                [2.02000000e-04, 0.00000000e00],
                [2.52000000e-04, 0.00000000e00],
                [3.02000000e-04, 0.00000000e00],
                [3.52000000e-04, 0.00000000e00],
                [4.02000000e-04, 0.00000000e00],
                [4.52000000e-04, 0.00000000e00],
                [5.02000000e-04, 0.00000000e00],
                [5.52000000e-04, 0.00000000e00],
                [6.02000000e-04, 0.00000000e00],
                [6.52000000e-04, 0.00000000e00],
                [7.02000000e-04, 0.00000000e00],
                [7.52000000e-04, 0.00000000e00],
                [8.02000000e-04, 0.00000000e00],
                [8.52000000e-04, 0.00000000e00],
                [9.02000000e-04, 0.00000000e00],
                [9.52000000e-04, 0.00000000e00],
                [1.00200000e-03, 0.00000000e00],
                [1.05200000e-03, 0.00000000e00],
                [1.10200000e-03, 0.00000000e00],
                [1.15200000e-03, 0.00000000e00],
                [1.20200000e-03, 0.00000000e00],
                [1.25200000e-03, 0.00000000e00],
                [1.30200000e-03, 0.00000000e00],
                [1.35200000e-03, 0.00000000e00],
                [1.40200000e-03, 0.00000000e00],
                [1.45200000e-03, 0.00000000e00],
                [1.50200000e-03, 0.00000000e00],
                [1.55200000e-03, 0.00000000e00],
                [1.60200000e-03, 0.00000000e00],
                [1.65200000e-03, 0.00000000e00],
                [1.70200000e-03, 0.00000000e00],
                [1.75200000e-03, 0.00000000e00],
                [1.80200000e-03, 0.00000000e00],
                [1.85200000e-03, 0.00000000e00],
                [1.90200000e-03, 0.00000000e00],
                [1.95200000e-03, 0.00000000e00],
            ],
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "tx_offsets": [[-13.25, 0, 2.0]],
            "bird_offset": [-13.25, 0, 2.0],
            "normalization": [1e-12],
            "data_type": "dBzdt",
        },
        "Skytem 516M (HM)": {
            "type": "time",
            "flag": "HM",
            "channel_start_index": 20,
            "channels": {
                "[1]": 1.2148000e-05,
                "[2]": 1.3648000e-05,
                "[3]": 1.5648000e-05,
                "[4]": 1.7648000e-05,
                "[5]": 1.9648000e-05,
                "[6]": 2.1648000e-05,
                "[7]": 2.3648000e-05,
                "[8]": 2.6148000e-05,
                "[9]": 2.9648000e-05,
                "[10]": 3.4148000e-05,
                "[11]": 3.9648000e-05,
                "[12]": 4.6648000e-05,
                "[13]": 5.5648000e-05,
                "[14]": 6.7148000e-05,
                "[15]": 8.1648000e-05,
                "[16]": 9.9648000e-05,
                "[17]": 1.2214800e-04,
                "[18]": 1.5014800e-04,
                "[19]": 1.8564800e-04,
                "[20]": 2.3114800e-04,
                "[21]": 2.8814800e-04,
                "[22]": 3.6014800e-04,
                "[23]": 4.5114800e-04,
                "[24]": 5.6514800e-04,
                "[25]": 7.0914800e-04,
                "[26]": 8.9064800e-04,
                "[27]": 1.1136480e-03,
                "[28]": 1.3826480e-03,
                "[29]": 1.7041480e-03,
                "[30]": 2.0836480e-03,
                "[31]": 2.5276480e-03,
                "[32]": 3.0421480e-03,
                "[33]": 3.6316480e-03,
                "[34]": 4.3191480e-03,
                "[35]": 5.1371480e-03,
                "[36]": 6.1106480e-03,
                "[37]": 7.2696480e-03,
                "[38]": 8.6486480e-03,
                "[39]": 1.0288648e-02,
            },
            "uncertainty": [
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
                [0.1, 1.0e-3],
            ],
            "waveform": [
                [-3.650e-03, 1e-8],
                [-3.400e-03, 1.000e00],
                [-3.150e-03, 1.000e00],
                [-2.900e-03, 1.000e00],
                [-2.650e-03, 1.000e00],
                [-2.400e-03, 1.000e00],
                [-2.150e-03, 1.000e00],
                [-1.900e-03, 1.000e00],
                [-1.650e-03, 1.000e00],
                [-1.400e-03, 1.000e00],
                [-1.150e-03, 1.000e00],
                [-9.000e-04, 1.000e00],
                [-8.500e-04, 1.000e00],
                [-8.000e-04, 1.000],
                [-7.500e-04, 1.00],
                [-7.000e-04, 0.93],
                [-6.500e-04, 0.86],
                [-6.000e-04, 0.80],
                [-5.500e-04, 0.73],
                [-5.000e-04, 0.66],
                [-4.500e-04, 0.60],
                [-4.000e-04, 0.53],
                [-3.500e-04, 0.46],
                [-3.000e-04, 0.40],
                [-2.500e-04, 0.33],
                [-2.000e-04, 0.26],
                [-1.500e-04, 0.20],
                [-1.000e-04, 0.13],
                [-5.000e-05, 0.06],
                [0.00000e00, 0.00],
                [2.000e-06, 0.000e00],
                [4.000e-06, 0.000e00],
                [6.000e-06, 0.000e00],
                [8.000e-06, 0.000e00],
                [1.000e-05, 0.000e00],
                [1.200e-05, 0.000e00],
                [1.400e-05, 0.000e00],
                [1.600e-05, 0.000e00],
                [1.800e-05, 0.000e00],
                [2.000e-05, 0.000e00],
                [2.200e-05, 0.000e00],
                [2.400e-05, 0.000e00],
                [2.600e-05, 0.000e00],
                [3.100e-05, 0.000e00],
                [3.600e-05, 0.000e00],
                [4.100e-05, 0.000e00],
                [4.600e-05, 0.000e00],
                [5.100e-05, 0.000e00],
                [5.600e-05, 0.000e00],
                [6.100e-05, 0.000e00],
                [6.600e-05, 0.000e00],
                [7.100e-05, 0.000e00],
                [7.600e-05, 0.000e00],
                [8.100e-05, 0.000e00],
                [8.600e-05, 0.000e00],
                [9.100e-05, 0.000e00],
                [9.600e-05, 0.000e00],
                [1.010e-04, 0.000e00],
                [1.060e-04, 0.000e00],
                [1.110e-04, 0.000e00],
                [1.160e-04, 0.000e00],
                [1.210e-04, 0.000e00],
                [1.260e-04, 0.000e00],
                [1.310e-04, 0.000e00],
                [1.360e-04, 0.000e00],
                [1.410e-04, 0.000e00],
                [1.460e-04, 0.000e00],
                [1.510e-04, 0.000e00],
                [2.010e-04, 0.000e00],
                [2.510e-04, 0.000e00],
                [3.010e-04, 0.000e00],
                [3.510e-04, 0.000e00],
                [4.010e-04, 0.000e00],
                [4.510e-04, 0.000e00],
                [5.010e-04, 0.000e00],
                [5.510e-04, 0.000e00],
                [6.010e-04, 0.000e00],
                [6.510e-04, 0.000e00],
                [7.010e-04, 0.000e00],
                [7.510e-04, 0.000e00],
                [8.010e-04, 0.000e00],
                [8.510e-04, 0.000e00],
                [9.010e-04, 0.000e00],
                [9.510e-04, 0.000e00],
                [1.001e-03, 0.000e00],
                [1.051e-03, 0.000e00],
                [1.101e-03, 0.000e00],
                [1.151e-03, 0.000e00],
                [1.201e-03, 0.000e00],
                [1.251e-03, 0.000e00],
                [1.301e-03, 0.000e00],
                [1.351e-03, 0.000e00],
                [1.401e-03, 0.000e00],
                [1.451e-03, 0.000e00],
                [1.501e-03, 0.000e00],
                [1.551e-03, 0.000e00],
                [1.601e-03, 0.000e00],
                [1.651e-03, 0.000e00],
                [1.701e-03, 0.000e00],
                [1.751e-03, 0.000e00],
            ],
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "tx_offsets": [[-16.7, 0, 2.0]],
            "bird_offset": [-16.7, 0, 2.0],
            "normalization": [1e-12],
            "data_type": "dBzdt",
        },
        "Spectrem (2000)": {
            "type": "time",
            "flag": "EM_Z",
            "channel_start_index": 1,
            "channels": {
                "[1]": 10.85e-6,
                "[2]": 54.25e-6,
                "[3]": 119.35e-6,
                "[4]": 249.55e-6,
                "[5]": 509.96e-6,
                "[6]": 1030.82e-6,
                "[7]": 2072.49e-6,
                "[8]": 4155.82e-6,
            },
            "uncertainty": [
                [0.1, 32.0],
                [0.1, 16.0],
                [0.1, 8.0],
                [0.1, 4.0],
                [0.1, 2.0],
                [0.1, 1.0],
                [0.1, 0.25],
                [0.1, 0.1],
            ],
            "waveform": "stepoff",
            "tx_offsets": [[-136, 0, -39]],
            "bird_offset": [-136, 0, -39],
            "comment": "normalization accounts for unit dipole moment at the tx_offset, in part-per-2000",
            "normalization": "pp2t",
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "data_type": "Bz",
        },
        "Spectrem Plus": {
            "type": "time",
            "flag": "em_z",
            "channel_start_index": 1,
            "channels": {
                "[1]": 6.5e-6,
                "[2]": 26e-6,
                "[3]": 52.1e-6,
                "[4]": 104.2e-6,
                "[5]": 208.3e-6,
                "[6]": 416.7e-6,
                "[7]": 833.3e-6,
                "[8]": 1666.7e-6,
                "[9]": 3333.3e-6,
                "[10]": 6666.7e-6,
                "[11]": 13333.3e-6,
            },
            "uncertainty": [
                [0.05, 2000.0],
                [0.05, 2000.0],
                [0.05, 2000.0],
                [0.05, 2000.0],
                [0.05, 2000.0],
                [0.05, 2000.0],
                [0.05, 2000],
                [0.05, 2000.0],
                [0.05, 2000.0],
                [0.05, 2000.0],
                [0.05, 2000],
            ],
            "waveform": "stepoff",
            "tx_offsets": [[-131, 0, -36]],
            "bird_offset": [-131, 0, -36],
            "comment": "normalization accounts for unit dipole moment at the tx_offset, in part-per-million",
            "normalization": "ppm",
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "data_type": "Bz",
        },
        "Spectrem (2003)": {
            "type": "time",
            "flag": "EM_Z",
            "channel_start_index": 1,
            "channels": {
                "[1]": 21.7e-6,
                "[2]": 54.3e-6,
                "[3]": 119.4e-6,
                "[4]": 249.6e-6,
                "[5]": 501.0e-6,
                "[6]": 1030.8e-6,
                "[7]": 2072.5e-6,
                "[8]": 4155.8e-6,
            },
            "uncertainty": [
                [0.1, 10000.0],
                [0.1, 5000.0],
                [0.1, 3000.0],
                [0.1, 1500.0],
                [0.1, 900.0],
                [0.1, 700.0],
                [0.1, 600.0],
                [0.1, 500.0],
            ],
            "waveform": "stepoff",
            "tx_offsets": [[-136, 0, -39]],
            "bird_offset": [-136, 0, -39],
            "comment": "normalization accounts for unit dipole moment at the tx_offset, in part-per-2000",
            "normalization": "ppm",
            "tx_specs": {"type": "VMD", "a": 1.0, "I": 1.0},
            "data_type": "Bz",
        },
        "VTEM (2007)": {
            "type": "time",
            "flag": "Sf",
            "channel_start_index": 9,
            "channels": {
                "[1]": 0e1,
                "[2]": 0e1,
                "[3]": 0e1,
                "[4]": 0e1,
                "[5]": 0e1,
                "[6]": 0e1,
                "[7]": 0e1,
                "[8]": 0e1,
                "[9]": 99e-6,
                "[10]": 120e-6,
                "[11]": 141e-6,
                "[12]": 167e-6,
                "[13]": 198e-6,
                "[14]": 234e-6,
                "[15]": 281e-6,
                "[16]": 339e-6,
                "[17]": 406e-6,
                "[18]": 484e-6,
                "[19]": 573e-6,
                "[20]": 682e-6,
                "[21]": 818e-6,
                "[22]": 974e-6,
                "[23]": 1151e-6,
                "[24]": 1370e-6,
                "[25]": 1641e-6,
                "[26]": 1953e-6,
                "[27]": 2307e-6,
                "[28]": 2745e-6,
                "[29]": 3286e-6,
                "[30]": 3911e-6,
                "[31]": 4620e-6,
                "[32]": 5495e-6,
                "[33]": 6578e-6,
                "[34]": 7828e-6,
                "[35]": 9245e-6,
            },
            "uncertainty": [
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
                [0.1, 5e-4],
            ],
            "waveform": [
                [-4.30e-03, 1.0e-08],
                [-4.20e-03, 3.34253e-02],
                [-4.10e-03, 1.16092e-01],
                [-4.0e-03, 1.97080e-01],
                [-3.90e-03, 2.75748e-01],
                [-3.80e-03, 3.51544e-01],
                [-3.70e-03, 4.23928e-01],
                [-3.60e-03, 4.92386e-01],
                [-3.50e-03, 5.56438e-01],
                [-3.40e-03, 6.15645e-01],
                [-3.30e-03, 6.69603e-01],
                [-3.20e-03, 7.17955e-01],
                [-3.10e-03, 7.60389e-01],
                [-3.0e-03, 7.96642e-01],
                [-2.90e-03, 8.26499e-01],
                [-2.80e-03, 8.49796e-01],
                [-2.70e-03, 8.66421e-01],
                [-2.60e-03, 8.78934e-01],
                [-2.50e-03, 8.91465e-01],
                [-2.40e-03, 9.03901e-01],
                [-2.30e-03, 9.16161e-01],
                [-2.20e-03, 9.28239e-01],
                [-2.10e-03, 9.40151e-01],
                [-2.0e-03, 9.51908e-01],
                [-1.90e-03, 9.63509e-01],
                [-1.80e-03, 9.74953e-01],
                [-1.70e-03, 9.86240e-01],
                [-1.60e-03, 9.97372e-01],
                [-1.50e-03, 1.0e00],
                [-1.40e-03, 9.65225e-01],
                [-1.30e-03, 9.23590e-01],
                [-1.20e-03, 8.75348e-01],
                [-1.10e-03, 8.20965e-01],
                [-1.0e-03, 7.60913e-01],
                [-9.0e-04, 6.95697e-01],
                [-8.0e-04, 6.25858e-01],
                [-7.0e-04, 5.51972e-01],
                [-6.0e-04, 4.74644e-01],
                [-5.0e-04, 3.94497e-01],
                [-4.0e-04, 3.12171e-01],
                [-3.0e-04, 2.28318e-01],
                [-2.0e-04, 1.43599e-01],
                [-1.0e-04, 5.86805e-02],
                [0.0e00, 0.0e00],
                [2.0e-05, 0.0e00],
                [4.0e-05, 0.0e00],
                [6.0e-05, 0.0e00],
                [8.0e-05, 0.0e00],
                [1.0e-04, 0.0e00],
                [1.20e-04, 0.0e00],
                [1.40e-04, 0.0e00],
                [1.60e-04, 0.0e00],
                [1.80e-04, 0.0e00],
                [2.0e-04, 0.0e00],
                [2.20e-04, 0.0e00],
                [2.40e-04, 0.0e00],
                [2.60e-04, 0.0e00],
                [2.80e-04, 0.0e00],
                [3.0e-04, 0.0e00],
                [3.90e-04, 0.0e00],
                [4.70e-04, 0.0e00],
                [5.50e-04, 0.0e00],
                [6.30e-04, 0.0e00],
                [7.10e-04, 0.0e00],
                [7.90e-04, 0.0e00],
                [8.70e-04, 0.0e00],
                [9.50e-04, 0.0e00],
                [1.03e-03, 0.0e00],
                [1.11e-03, 0.0e00],
                [1.19e-03, 0.0e00],
                [1.27e-03, 0.0e00],
                [1.35e-03, 0.0e00],
                [1.43e-03, 0.0e00],
                [1.51e-03, 0.0e00],
                [1.59e-03, 0.0e00],
                [1.67e-03, 0.0e00],
                [1.75e-03, 0.0e00],
                [1.83e-03, 0.0e00],
                [1.91e-03, 0.0e00],
                [1.99e-03, 0.0e00],
                [2.07e-03, 0.0e00],
                [2.15e-03, 0.0e00],
                [2.23e-03, 0.0e00],
                [2.31e-03, 0.0e00],
                [2.35e-03, 0.0e00],
                [2.65e-03, 0.0e00],
                [2.95e-03, 0.0e00],
                [3.25e-03, 0.0e00],
                [3.55e-03, 0.0e00],
                [3.85e-03, 0.0e00],
                [4.15e-03, 0.0e00],
                [4.45e-03, 0.0e00],
                [4.75e-03, 0.0e00],
                [5.05e-03, 0.0e00],
                [5.35e-03, 0.0e00],
                [5.65e-03, 0.0e00],
                [5.95e-03, 0.0e00],
                [6.25e-03, 0.0e00],
                [6.55e-03, 0.0e00],
                [6.85e-03, 0.0e00],
                [7.15e-03, 0.0e00],
                [7.45e-03, 0.0e00],
                [7.75e-03, 0.0e00],
                [8.05e-03, 0.0e00],
                [8.35e-03, 0.0e00],
                [8.65e-03, 0.0e00],
                [8.95e-03, 0.0e00],
                [9.25e-03, 0.0e00],
                [9.55e-03, 0.0e00],
                [9.85e-03, 0.0e00],
                [1.0150e-02, 0.0e00],
                [1.0450e-02, 0.0e00],
                [1.0750e-02, 0.0e00],
            ],
            "tx_offsets": [[0, 0, 0]],
            "bird_offset": [0, 0, 0],
            "normalization": [531, 1e-12],
            "tx_specs": {"type": "CircularLoop", "a": 13.0, "I": 1.0},
            "data_type": "dBzdt",
        },
        "VTEM Plus": {
            "type": "time",
            "flag": "SFz",
            "channel_start_index": 14,
            "channels": {
                "[1]": 0e1,
                "[2]": 0e1,
                "[3]": 0e1,
                "[4]": 21e-6,
                "[5]": 26e-6,
                "[6]": 31e-6,
                "[7]": 36e-6,
                "[8]": 42e-6,
                "[9]": 48e-6,
                "[10]": 55e-6,
                "[11]": 63e-6,
                "[12]": 73e-6,
                "[13]": 83e-6,
                "[14]": 96e-6,
                "[15]": 110e-6,
                "[16]": 126e-6,
                "[17]": 145e-6,
                "[18]": 167e-6,
                "[19]": 192e-6,
                "[20]": 220e-6,
                "[21]": 253e-6,
                "[22]": 290e-6,
                "[23]": 333e-6,
                "[24]": 383e-6,
                "[25]": 440e-6,
                "[26]": 505e-6,
                "[27]": 580e-6,
                "[28]": 667e-6,
                "[29]": 766e-6,
                "[30]": 880e-6,
                "[31]": 1010e-6,
                "[32]": 1161e-6,
                "[33]": 1333e-6,
                "[34]": 1531e-6,
                "[35]": 1760e-6,
                "[36]": 2021e-6,
                "[37]": 2323e-6,
                "[38]": 2667e-6,
                "[39]": 3063e-6,
                "[40]": 3521e-6,
                "[41]": 4042e-6,
                "[42]": 4641e-6,
                "[43]": 5333e-6,
                "[44]": 6125e-6,
                "[45]": 7036e-6,
                "[46]": 8083e-6,
            },
            "uncertainty": [
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
            ],
            "waveform": [
                [-6.90000000e-03, 1.65109034e-01],
                [-6.80000000e-03, 2.38693737e-01],
                [-6.70000000e-03, 3.10076270e-01],
                [-6.60000000e-03, 3.81995918e-01],
                [-6.50000000e-03, 3.96712859e-01],
                [-6.40000000e-03, 3.93651305e-01],
                [-6.30000000e-03, 3.91771404e-01],
                [-6.20000000e-03, 4.42206467e-01],
                [-6.10000000e-03, 5.04189494e-01],
                [-6.00000000e-03, 5.65259426e-01],
                [-5.90000000e-03, 6.19078311e-01],
                [-5.80000000e-03, 6.68385433e-01],
                [-5.70000000e-03, 6.94328070e-01],
                [-5.60000000e-03, 6.89547749e-01],
                [-5.50000000e-03, 6.84230315e-01],
                [-5.40000000e-03, 7.04586959e-01],
                [-5.30000000e-03, 7.41540445e-01],
                [-5.20000000e-03, 7.75539800e-01],
                [-5.10000000e-03, 8.04221721e-01],
                [-5.00000000e-03, 8.28499302e-01],
                [-4.90000000e-03, 8.43699646e-01],
                [-4.80000000e-03, 8.38489634e-01],
                [-4.70000000e-03, 8.32420238e-01],
                [-4.60000000e-03, 8.29788377e-01],
                [-4.50000000e-03, 8.41658610e-01],
                [-4.40000000e-03, 8.54119669e-01],
                [-4.30000000e-03, 8.65775056e-01],
                [-4.20000000e-03, 8.77376732e-01],
                [-4.10000000e-03, 8.88172736e-01],
                [-4.00000000e-03, 8.86829950e-01],
                [-3.90000000e-03, 8.80491997e-01],
                [-3.80000000e-03, 8.74315179e-01],
                [-3.70000000e-03, 8.81888495e-01],
                [-3.60000000e-03, 8.93221613e-01],
                [-3.50000000e-03, 9.05038135e-01],
                [-3.40000000e-03, 9.16156408e-01],
                [-3.30000000e-03, 9.27704372e-01],
                [-3.20000000e-03, 9.31249329e-01],
                [-3.10000000e-03, 9.24642819e-01],
                [-3.00000000e-03, 9.17660329e-01],
                [-2.90000000e-03, 9.20023633e-01],
                [-2.80000000e-03, 9.30336234e-01],
                [-2.70000000e-03, 9.41669352e-01],
                [-2.60000000e-03, 9.51767107e-01],
                [-2.50000000e-03, 9.62885380e-01],
                [-2.40000000e-03, 9.71479214e-01],
                [-2.30000000e-03, 9.65248684e-01],
                [-2.20000000e-03, 9.58373617e-01],
                [-2.10000000e-03, 9.54291546e-01],
                [-2.00000000e-03, 9.64443012e-01],
                [-1.90000000e-03, 9.74809324e-01],
                [-1.80000000e-03, 9.85068214e-01],
                [-1.70000000e-03, 9.95219680e-01],
                [-1.60000000e-03, 1.00000000e00],
                [-1.50000000e-03, 9.70136427e-01],
                [-1.40000000e-03, 9.32753250e-01],
                [-1.30000000e-03, 8.93651305e-01],
                [-1.20000000e-03, 8.44505317e-01],
                [-1.10000000e-03, 7.92512622e-01],
                [-1.00000000e-03, 7.35900741e-01],
                [-9.00000000e-04, 6.74938232e-01],
                [-8.00000000e-04, 6.10108497e-01],
                [-7.00000000e-04, 5.41894940e-01],
                [-6.00000000e-04, 4.70727253e-01],
                [-5.00000000e-04, 3.89300677e-01],
                [-4.00000000e-04, 3.17595875e-01],
                [-3.00000000e-04, 2.36491567e-01],
                [-2.00000000e-04, 1.62638307e-01],
                [-1.00000000e-04, 8.43807068e-02],
                [-0.00000000e00, 0.00000000e00],
                [5.00000000e-06, 0.00000000e00],
                [1.00000000e-05, 0.00000000e00],
                [1.50000000e-05, 0.00000000e00],
                [2.00000000e-05, 0.00000000e00],
                [2.50000000e-05, 0.00000000e00],
                [3.00000000e-05, 0.00000000e00],
                [3.50000000e-05, 0.00000000e00],
                [4.00000000e-05, 0.00000000e00],
                [4.50000000e-05, 0.00000000e00],
                [5.00000000e-05, 0.00000000e00],
                [5.50000000e-05, 0.00000000e00],
                [6.00000000e-05, 0.00000000e00],
                [6.50000000e-05, 0.00000000e00],
                [7.00000000e-05, 0.00000000e00],
                [7.50000000e-05, 0.00000000e00],
                [8.00000000e-05, 0.00000000e00],
                [8.50000000e-05, 0.00000000e00],
                [9.00000000e-05, 0.00000000e00],
                [9.50000000e-05, 0.00000000e00],
                [1.00000000e-04, 0.00000000e00],
                [1.05000000e-04, 0.00000000e00],
                [1.10000000e-04, 0.00000000e00],
                [1.20000000e-04, 0.00000000e00],
                [1.30000000e-04, 0.00000000e00],
                [1.40000000e-04, 0.00000000e00],
                [1.50000000e-04, 0.00000000e00],
                [1.60000000e-04, 0.00000000e00],
                [1.70000000e-04, 0.00000000e00],
                [1.80000000e-04, 0.00000000e00],
                [1.90000000e-04, 0.00000000e00],
                [2.00000000e-04, 0.00000000e00],
                [2.10000000e-04, 0.00000000e00],
                [2.20000000e-04, 0.00000000e00],
                [2.30000000e-04, 0.00000000e00],
                [2.40000000e-04, 0.00000000e00],
                [2.50000000e-04, 0.00000000e00],
                [2.60000000e-04, 0.00000000e00],
                [2.70000000e-04, 0.00000000e00],
                [2.80000000e-04, 0.00000000e00],
                [2.90000000e-04, 0.00000000e00],
                [3.00000000e-04, 0.00000000e00],
                [3.50000000e-04, 0.00000000e00],
                [4.00000000e-04, 0.00000000e00],
                [4.50000000e-04, 0.00000000e00],
                [5.00000000e-04, 0.00000000e00],
                [5.50000000e-04, 0.00000000e00],
                [6.00000000e-04, 0.00000000e00],
                [6.50000000e-04, 0.00000000e00],
                [7.00000000e-04, 0.00000000e00],
                [7.50000000e-04, 0.00000000e00],
                [8.00000000e-04, 0.00000000e00],
                [8.50000000e-04, 0.00000000e00],
                [9.00000000e-04, 0.00000000e00],
                [9.50000000e-04, 0.00000000e00],
                [1.00000000e-03, 0.00000000e00],
                [1.05000000e-03, 0.00000000e00],
                [1.10000000e-03, 0.00000000e00],
                [1.15000000e-03, 0.00000000e00],
                [1.20000000e-03, 0.00000000e00],
                [1.25000000e-03, 0.00000000e00],
                [1.30000000e-03, 0.00000000e00],
                [1.35000000e-03, 0.00000000e00],
                [1.40000000e-03, 0.00000000e00],
                [1.65000000e-03, 0.00000000e00],
                [1.90000000e-03, 0.00000000e00],
                [2.15000000e-03, 0.00000000e00],
                [2.40000000e-03, 0.00000000e00],
                [2.65000000e-03, 0.00000000e00],
                [2.90000000e-03, 0.00000000e00],
                [3.15000000e-03, 0.00000000e00],
                [3.40000000e-03, 0.00000000e00],
                [3.65000000e-03, 0.00000000e00],
                [3.90000000e-03, 0.00000000e00],
                [4.15000000e-03, 0.00000000e00],
                [4.40000000e-03, 0.00000000e00],
                [4.65000000e-03, 0.00000000e00],
                [4.90000000e-03, 0.00000000e00],
                [5.15000000e-03, 0.00000000e00],
                [5.40000000e-03, 0.00000000e00],
                [5.65000000e-03, 0.00000000e00],
                [5.90000000e-03, 0.00000000e00],
                [6.15000000e-03, 0.00000000e00],
                [6.40000000e-03, 0.00000000e00],
                [6.65000000e-03, 0.00000000e00],
                [6.90000000e-03, 0.00000000e00],
                [7.15000000e-03, 0.00000000e00],
                [7.40000000e-03, 0.00000000e00],
                [7.65000000e-03, 0.00000000e00],
                [7.90000000e-03, 0.00000000e00],
                [8.15000000e-03, 0.00000000e00],
                [8.40000000e-03, 0.00000000e00],
            ],
            "tx_offsets": [[0, 0, 0]],
            "bird_offset": [-25, 0, -34],
            "normalization": [531, 1e-12],
            "tx_specs": {"type": "CircularLoop", "a": 13.0, "I": 1.0},
            "data_type": "dBzdt",
        },
        "VTEM Max": {
            "type": "time",
            "flag": "SFz",
            "channel_start_index": 14,
            "channels": {
                "[1]": 0e1,
                "[2]": 0e1,
                "[3]": 0e1,
                "[4]": 21e-6,
                "[5]": 26e-6,
                "[6]": 31e-6,
                "[7]": 36e-6,
                "[8]": 42e-6,
                "[9]": 48e-6,
                "[10]": 55e-6,
                "[11]": 63e-6,
                "[12]": 73e-6,
                "[13]": 83e-6,
                "[14]": 96e-6,
                "[15]": 110e-6,
                "[16]": 126e-6,
                "[17]": 145e-6,
                "[18]": 167e-6,
                "[19]": 192e-6,
                "[20]": 220e-6,
                "[21]": 253e-6,
                "[22]": 290e-6,
                "[23]": 333e-6,
                "[24]": 383e-6,
                "[25]": 440e-6,
                "[26]": 505e-6,
                "[27]": 580e-6,
                "[28]": 667e-6,
                "[29]": 766e-6,
                "[30]": 880e-6,
                "[31]": 1010e-6,
                "[32]": 1161e-6,
                "[33]": 1333e-6,
                "[34]": 1531e-6,
                "[35]": 1760e-6,
                "[36]": 2021e-6,
                "[37]": 2323e-6,
                "[38]": 2667e-6,
                "[39]": 3063e-6,
                "[40]": 3521e-6,
                "[41]": 4042e-6,
                "[42]": 4641e-6,
                "[43]": 5333e-6,
                "[44]": 6125e-6,
                "[45]": 7036e-6,
                "[46]": 8083e-6,
            },
            "uncertainty": [
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
            ],
            "waveform": [
                [-6.90000000e-03, 1.65109034e-01],
                [-6.80000000e-03, 2.38693737e-01],
                [-6.70000000e-03, 3.10076270e-01],
                [-6.60000000e-03, 3.81995918e-01],
                [-6.50000000e-03, 3.96712859e-01],
                [-6.40000000e-03, 3.93651305e-01],
                [-6.30000000e-03, 3.91771404e-01],
                [-6.20000000e-03, 4.42206467e-01],
                [-6.10000000e-03, 5.04189494e-01],
                [-6.00000000e-03, 5.65259426e-01],
                [-5.90000000e-03, 6.19078311e-01],
                [-5.80000000e-03, 6.68385433e-01],
                [-5.70000000e-03, 6.94328070e-01],
                [-5.60000000e-03, 6.89547749e-01],
                [-5.50000000e-03, 6.84230315e-01],
                [-5.40000000e-03, 7.04586959e-01],
                [-5.30000000e-03, 7.41540445e-01],
                [-5.20000000e-03, 7.75539800e-01],
                [-5.10000000e-03, 8.04221721e-01],
                [-5.00000000e-03, 8.28499302e-01],
                [-4.90000000e-03, 8.43699646e-01],
                [-4.80000000e-03, 8.38489634e-01],
                [-4.70000000e-03, 8.32420238e-01],
                [-4.60000000e-03, 8.29788377e-01],
                [-4.50000000e-03, 8.41658610e-01],
                [-4.40000000e-03, 8.54119669e-01],
                [-4.30000000e-03, 8.65775056e-01],
                [-4.20000000e-03, 8.77376732e-01],
                [-4.10000000e-03, 8.88172736e-01],
                [-4.00000000e-03, 8.86829950e-01],
                [-3.90000000e-03, 8.80491997e-01],
                [-3.80000000e-03, 8.74315179e-01],
                [-3.70000000e-03, 8.81888495e-01],
                [-3.60000000e-03, 8.93221613e-01],
                [-3.50000000e-03, 9.05038135e-01],
                [-3.40000000e-03, 9.16156408e-01],
                [-3.30000000e-03, 9.27704372e-01],
                [-3.20000000e-03, 9.31249329e-01],
                [-3.10000000e-03, 9.24642819e-01],
                [-3.00000000e-03, 9.17660329e-01],
                [-2.90000000e-03, 9.20023633e-01],
                [-2.80000000e-03, 9.30336234e-01],
                [-2.70000000e-03, 9.41669352e-01],
                [-2.60000000e-03, 9.51767107e-01],
                [-2.50000000e-03, 9.62885380e-01],
                [-2.40000000e-03, 9.71479214e-01],
                [-2.30000000e-03, 9.65248684e-01],
                [-2.20000000e-03, 9.58373617e-01],
                [-2.10000000e-03, 9.54291546e-01],
                [-2.00000000e-03, 9.64443012e-01],
                [-1.90000000e-03, 9.74809324e-01],
                [-1.80000000e-03, 9.85068214e-01],
                [-1.70000000e-03, 9.95219680e-01],
                [-1.60000000e-03, 1.00000000e00],
                [-1.50000000e-03, 9.70136427e-01],
                [-1.40000000e-03, 9.32753250e-01],
                [-1.30000000e-03, 8.93651305e-01],
                [-1.20000000e-03, 8.44505317e-01],
                [-1.10000000e-03, 7.92512622e-01],
                [-1.00000000e-03, 7.35900741e-01],
                [-9.00000000e-04, 6.74938232e-01],
                [-8.00000000e-04, 6.10108497e-01],
                [-7.00000000e-04, 5.41894940e-01],
                [-6.00000000e-04, 4.70727253e-01],
                [-5.00000000e-04, 3.89300677e-01],
                [-4.00000000e-04, 3.17595875e-01],
                [-3.00000000e-04, 2.36491567e-01],
                [-2.00000000e-04, 1.62638307e-01],
                [-1.00000000e-04, 8.43807068e-02],
                [-0.00000000e00, 0.00000000e00],
                [5.00000000e-06, 0.00000000e00],
                [1.00000000e-05, 0.00000000e00],
                [1.50000000e-05, 0.00000000e00],
                [2.00000000e-05, 0.00000000e00],
                [2.50000000e-05, 0.00000000e00],
                [3.00000000e-05, 0.00000000e00],
                [3.50000000e-05, 0.00000000e00],
                [4.00000000e-05, 0.00000000e00],
                [4.50000000e-05, 0.00000000e00],
                [5.00000000e-05, 0.00000000e00],
                [5.50000000e-05, 0.00000000e00],
                [6.00000000e-05, 0.00000000e00],
                [6.50000000e-05, 0.00000000e00],
                [7.00000000e-05, 0.00000000e00],
                [7.50000000e-05, 0.00000000e00],
                [8.00000000e-05, 0.00000000e00],
                [8.50000000e-05, 0.00000000e00],
                [9.00000000e-05, 0.00000000e00],
                [9.50000000e-05, 0.00000000e00],
                [1.00000000e-04, 0.00000000e00],
                [1.05000000e-04, 0.00000000e00],
                [1.10000000e-04, 0.00000000e00],
                [1.20000000e-04, 0.00000000e00],
                [1.30000000e-04, 0.00000000e00],
                [1.40000000e-04, 0.00000000e00],
                [1.50000000e-04, 0.00000000e00],
                [1.60000000e-04, 0.00000000e00],
                [1.70000000e-04, 0.00000000e00],
                [1.80000000e-04, 0.00000000e00],
                [1.90000000e-04, 0.00000000e00],
                [2.00000000e-04, 0.00000000e00],
                [2.10000000e-04, 0.00000000e00],
                [2.20000000e-04, 0.00000000e00],
                [2.30000000e-04, 0.00000000e00],
                [2.40000000e-04, 0.00000000e00],
                [2.50000000e-04, 0.00000000e00],
                [2.60000000e-04, 0.00000000e00],
                [2.70000000e-04, 0.00000000e00],
                [2.80000000e-04, 0.00000000e00],
                [2.90000000e-04, 0.00000000e00],
                [3.00000000e-04, 0.00000000e00],
                [3.50000000e-04, 0.00000000e00],
                [4.00000000e-04, 0.00000000e00],
                [4.50000000e-04, 0.00000000e00],
                [5.00000000e-04, 0.00000000e00],
                [5.50000000e-04, 0.00000000e00],
                [6.00000000e-04, 0.00000000e00],
                [6.50000000e-04, 0.00000000e00],
                [7.00000000e-04, 0.00000000e00],
                [7.50000000e-04, 0.00000000e00],
                [8.00000000e-04, 0.00000000e00],
                [8.50000000e-04, 0.00000000e00],
                [9.00000000e-04, 0.00000000e00],
                [9.50000000e-04, 0.00000000e00],
                [1.00000000e-03, 0.00000000e00],
                [1.05000000e-03, 0.00000000e00],
                [1.10000000e-03, 0.00000000e00],
                [1.15000000e-03, 0.00000000e00],
                [1.20000000e-03, 0.00000000e00],
                [1.25000000e-03, 0.00000000e00],
                [1.30000000e-03, 0.00000000e00],
                [1.35000000e-03, 0.00000000e00],
                [1.40000000e-03, 0.00000000e00],
                [1.65000000e-03, 0.00000000e00],
                [1.90000000e-03, 0.00000000e00],
                [2.15000000e-03, 0.00000000e00],
                [2.40000000e-03, 0.00000000e00],
                [2.65000000e-03, 0.00000000e00],
                [2.90000000e-03, 0.00000000e00],
                [3.15000000e-03, 0.00000000e00],
                [3.40000000e-03, 0.00000000e00],
                [3.65000000e-03, 0.00000000e00],
                [3.90000000e-03, 0.00000000e00],
                [4.15000000e-03, 0.00000000e00],
                [4.40000000e-03, 0.00000000e00],
                [4.65000000e-03, 0.00000000e00],
                [4.90000000e-03, 0.00000000e00],
                [5.15000000e-03, 0.00000000e00],
                [5.40000000e-03, 0.00000000e00],
                [5.65000000e-03, 0.00000000e00],
                [5.90000000e-03, 0.00000000e00],
                [6.15000000e-03, 0.00000000e00],
                [6.40000000e-03, 0.00000000e00],
                [6.65000000e-03, 0.00000000e00],
                [6.90000000e-03, 0.00000000e00],
                [7.15000000e-03, 0.00000000e00],
                [7.40000000e-03, 0.00000000e00],
                [7.65000000e-03, 0.00000000e00],
                [7.90000000e-03, 0.00000000e00],
                [8.15000000e-03, 0.00000000e00],
                [8.40000000e-03, 0.00000000e00],
            ],
            "tx_offsets": [[0, 0, 0]],
            "bird_offset": [-27, 0, -44],
            "normalization": [531, 1e-12],
            "tx_specs": {"type": "CircularLoop", "a": 13.0, "I": 1.0},
            "data_type": "dBzdt",
        },
        "Xcite": {
            "type": "time",
            "flag": "dBdt_Z_F",
            "channel_start_index": 7,
            "channels": {
                "[1]": 6.40000e-03,
                "[2]": 1.28000e-02,
                "[3]": 1.92000e-02,
                "[4]": 2.56000e-02,
                "[5]": 3.20000e-02,
                "[6]": 3.84000e-02,
                "[7]": 4.48000e-02,
                "[8]": 5.12000e-02,
                "[9]": 5.76000e-02,
                "[10]": 6.40000e-02,
                "[11]": 7.04000e-02,
                "[12]": 7.68000e-02,
                "[13]": 8.32000e-02,
                "[14]": 9.28000e-02,
                "[15]": 1.05600e-01,
                "[16]": 1.18400e-01,
                "[17]": 1.34400e-01,
                "[18]": 1.53600e-01,
                "[19]": 1.76000e-01,
                "[20]": 2.01600e-01,
                "[21]": 2.30400e-01,
                "[22]": 2.62400e-01,
                "[23]": 2.97600e-01,
                "[24]": 3.39200e-01,
                "[25]": 3.87200e-01,
                "[26]": 4.41600e-01,
                "[27]": 5.05600e-01,
                "[28]": 5.82400e-01,
                "[29]": 6.72000e-01,
                "[30]": 7.74400e-01,
                "[31]": 8.89600e-01,
                "[32]": 1.02080e00,
                "[33]": 1.17120e00,
                "[34]": 1.34400e00,
                "[35]": 1.54560e00,
                "[36]": 1.77920e00,
                "[37]": 2.04480e00,
                "[38]": 2.34560e00,
                "[39]": 2.69120e00,
                "[40]": 3.08800e00,
                "[41]": 3.54240e00,
                "[42]": 4.06720e00,
                "[43]": 4.67200e00,
                "[44]": 5.36640e00,
                "[45]": 6.16320e00,
                "[46]": 7.07840e00,
                "[47]": 8.13120e00,
                "[48]": 9.33760e00,
                "[49]": 1.07232e01,
                "[50]": 1.24896e01,
            },
            "uncertainty": [
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
                [0.05, 1e-3],
            ],
            "waveform": [
                [-5.50e00, 1.00000000e-08],
                [-5.20e00, 5.00000000e-01],
                [-4.90e00, 9.40112151e-01],
                [-4.60e00, 9.40178855e-01],
                [-4.30e00, 9.40245558e-01],
                [-4.00e00, 9.40312262e-01],
                [-3.70e00, 9.40378966e-01],
                [-3.40e00, 9.40445670e-01],
                [-3.10e00, 9.40512373e-01],
                [-2.80e00, 9.40579077e-01],
                [-2.50e00, 9.40645781e-01],
                [-2.20e00, 9.40712484e-01],
                [-1.90e00, 9.40779188e-01],
                [-1.60e00, 9.40845892e-01],
                [-1.30e00, 9.40912595e-01],
                [-1.00e00, 9.40979299e-01],
                [-7.00e-01, 9.41046003e-01],
                [-4.00e-01, 9.41112706e-01],
                [-3.50e-01, 9.41123824e-01],
                [-3.00e-01, 8.98234294e-01],
                [-2.50e-01, 7.89179185e-01],
                [-2.00e-01, 6.47911463e-01],
                [-1.50e-01, 4.80782576e-01],
                [-1.00e-01, 2.95120400e-01],
                [-5.00e-02, 9.92697471e-02],
                [0.00e-00, 3.68628739e-18],
                [5.00e-03, 0.00000000e00],
                [1.00e-02, 0.00000000e00],
                [1.50e-02, 0.00000000e00],
                [2.00e-02, 0.00000000e00],
                [2.50e-02, 0.00000000e00],
                [3.00e-02, 0.00000000e00],
                [3.50e-02, 0.00000000e00],
                [4.00e-02, 0.00000000e00],
                [4.50e-02, 0.00000000e00],
                [5.00e-02, 0.00000000e00],
                [5.50e-02, 0.00000000e00],
                [6.00e-02, 0.00000000e00],
                [6.50e-02, 0.00000000e00],
                [7.00e-02, 0.00000000e00],
                [7.50e-02, 0.00000000e00],
                [8.00e-02, 0.00000000e00],
                [8.50e-02, 0.00000000e00],
                [9.00e-02, 0.00000000e00],
                [9.50e-02, 0.00000000e00],
                [1.00e-01, 0.00000000e00],
                [1.10e-01, 0.00000000e00],
                [1.20e-01, 0.00000000e00],
                [1.30e-01, 0.00000000e00],
                [1.40e-01, 0.00000000e00],
                [1.50e-01, 0.00000000e00],
                [1.60e-01, 0.00000000e00],
                [1.70e-01, 0.00000000e00],
                [1.80e-01, 0.00000000e00],
                [1.90e-01, 0.00000000e00],
                [2.00e-01, 0.00000000e00],
                [2.10e-01, 0.00000000e00],
                [2.20e-01, 0.00000000e00],
                [2.30e-01, 0.00000000e00],
                [2.40e-01, 0.00000000e00],
                [2.50e-01, 0.00000000e00],
                [2.60e-01, 0.00000000e00],
                [2.70e-01, 0.00000000e00],
                [2.80e-01, 0.00000000e00],
                [2.90e-01, 0.00000000e00],
                [3.00e-01, 0.00000000e00],
                [3.25e-01, 0.00000000e00],
                [3.50e-01, 0.00000000e00],
                [3.75e-01, 0.00000000e00],
                [4.00e-01, 0.00000000e00],
                [4.25e-01, 0.00000000e00],
                [4.50e-01, 0.00000000e00],
                [4.75e-01, 0.00000000e00],
                [5.00e-01, 0.00000000e00],
                [5.25e-01, 0.00000000e00],
                [5.50e-01, 0.00000000e00],
                [5.75e-01, 0.00000000e00],
                [6.00e-01, 0.00000000e00],
                [7.50e-01, 0.00000000e00],
                [9.00e-01, 0.00000000e00],
                [1.05e00, 0.00000000e00],
                [1.20e00, 0.00000000e00],
                [1.35e00, 0.00000000e00],
                [1.50e00, 0.00000000e00],
                [1.65e00, 0.00000000e00],
                [1.80e00, 0.00000000e00],
                [1.95e00, 0.00000000e00],
                [2.10e00, 0.00000000e00],
                [2.25e00, 0.00000000e00],
                [2.40e00, 0.00000000e00],
                [2.55e00, 0.00000000e00],
                [2.70e00, 0.00000000e00],
                [2.85e00, 0.00000000e00],
                [3.00e00, 0.00000000e00],
                [3.15e00, 0.00000000e00],
                [3.30e00, 0.00000000e00],
                [3.45e00, 0.00000000e00],
                [3.60e00, 0.00000000e00],
                [3.75e00, 0.00000000e00],
                [3.90e00, 0.00000000e00],
                [4.00e00, 0.00000000e00],
                [4.50e00, 0.00000000e00],
                [5.00e00, 0.00000000e00],
                [5.50e00, 0.00000000e00],
                [6.00e00, 0.00000000e00],
                [6.50e00, 0.00000000e00],
                [7.00e00, 0.00000000e00],
                [7.50e00, 0.00000000e00],
                [8.00e00, 0.00000000e00],
                [8.50e00, 0.00000000e00],
                [9.00e00, 0.00000000e00],
                [9.50e00, 0.00000000e00],
                [1.00e01, 0.00000000e00],
                [1.05e01, 0.00000000e00],
                [1.10e01, 0.00000000e00],
                [1.15e01, 0.00000000e00],
                [1.20e01, 0.00000000e00],
                [1.25e01, 0.00000000e00],
            ],
            "tx_offsets": [[0, 0, 0]],
            "bird_offset": [0, 0, 0],
            "normalization": [3.1416, 1e-12],
            "tx_specs": {"type": "CircularLoop", "a": 1.0, "I": 1.0},
            "data_type": "dBzdt",
        },
        "Zonge 8Hz": {
            "type": "time",
            "flag": "CH",
            "channel_start_index": 6,
            "channels": {
                "1": 8.620e-05,
                "2": 1.168e-04,
                "3": 1.473e-04,
                "4": 1.778e-04,
                "5": 2.083e-04,
                "6": 2.388e-04,
                "7": 2.846e-04,
                "8": 3.456e-04,
                "9": 4.067e-04,
                "10": 4.830e-04,
                "11": 5.745e-04,
                "12": 6.966e-04,
                "13": 8.644e-04,
                "14": 1.063e-03,
                "15": 1.307e-03,
                "16": 1.612e-03,
                "17": 2.009e-03,
                "18": 2.528e-03,
                "19": 3.169e-03,
                "20": 3.962e-03,
                "21": 4.954e-03,
                "22": 6.220e-03,
                "23": 7.822e-03,
                "24": 9.806e-03,
                "25": 1.231e-02,
                "26": 1.548e-02,
                "27": 1.946e-02,
                "28": 2.449e-02,
            },
            "uncertainty": [
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
                [0.05, 1e-0],
            ],
            "waveform": [
                [-3.0225e-02, 1.0000e-08],
                [-2.5225e-02, 5.0000e-01],
                [-2.0225e-02, 1.0000e00],
                [-1.5225e-02, 1.0000e00],
                [-1.0225e-02, 1.0000e00],
                [-5.2250e-03, 1.0000e00],
                [-2.2500e-04, 1.0000e00],
                [-2.0000e-04, 1.0000e00],
                [-1.7500e-04, 8.7500e-01],
                [-1.5000e-04, 7.5000e-01],
                [-1.2500e-04, 6.2500e-01],
                [-1.0000e-04, 5.0000e-01],
                [-7.5000e-05, 3.7500e-01],
                [-5.0000e-05, 2.5000e-01],
                [-2.5000e-05, 1.2500e-01],
                [-0.0000e00, 0.0000e00],
                [2.5000e-05, 0.0000e00],
                [5.0000e-05, 0.0000e00],
                [7.5000e-05, 0.0000e00],
                [1.0000e-04, 0.0000e00],
                [1.2500e-04, 0.0000e00],
                [1.5000e-04, 0.0000e00],
                [1.7500e-04, 0.0000e00],
                [2.0000e-04, 0.0000e00],
                [2.2500e-04, 0.0000e00],
                [2.5000e-04, 0.0000e00],
                [2.7500e-04, 0.0000e00],
                [3.0000e-04, 0.0000e00],
                [3.7500e-04, 0.0000e00],
                [4.5000e-04, 0.0000e00],
                [5.2500e-04, 0.0000e00],
                [6.0000e-04, 0.0000e00],
                [6.7500e-04, 0.0000e00],
                [7.5000e-04, 0.0000e00],
                [8.2500e-04, 0.0000e00],
                [9.0000e-04, 0.0000e00],
                [9.7500e-04, 0.0000e00],
                [1.0500e-03, 0.0000e00],
                [1.1250e-03, 0.0000e00],
                [1.2000e-03, 0.0000e00],
                [1.2750e-03, 0.0000e00],
                [1.3500e-03, 0.0000e00],
                [1.4250e-03, 0.0000e00],
                [1.8250e-03, 0.0000e00],
                [2.2250e-03, 0.0000e00],
                [2.6250e-03, 0.0000e00],
                [3.0250e-03, 0.0000e00],
                [3.4250e-03, 0.0000e00],
                [3.8250e-03, 0.0000e00],
                [4.2250e-03, 0.0000e00],
                [4.6250e-03, 0.0000e00],
                [5.0250e-03, 0.0000e00],
                [6.0250e-03, 0.0000e00],
                [7.0250e-03, 0.0000e00],
                [8.0250e-03, 0.0000e00],
                [9.0250e-03, 0.0000e00],
                [1.0025e-02, 0.0000e00],
                [1.1025e-02, 0.0000e00],
                [1.2025e-02, 0.0000e00],
                [1.3025e-02, 0.0000e00],
                [1.4025e-02, 0.0000e00],
                [1.5025e-02, 0.0000e00],
                [1.6025e-02, 0.0000e00],
                [1.7025e-02, 0.0000e00],
                [1.8025e-02, 0.0000e00],
                [1.9025e-02, 0.0000e00],
                [2.0025e-02, 0.0000e00],
                [2.1025e-02, 0.0000e00],
                [2.2025e-02, 0.0000e00],
                [2.3025e-02, 0.0000e00],
                [2.4025e-02, 0.0000e00],
                [2.5025e-02, 0.0000e00],
            ],
            "tx_offsets": [[0, 0, 0]],
            "bird_offset": [0, 0, 0],
            "comment": "normalization accounts for 100 m square loop (converted to circular), unit current, in micro-V",
            "normalization": [1e-5, 1e-6],
            "tx_specs": {"type": "CircularLoop", "a": 56.4, "I": 1.0},
            "data_type": "dBzdt",
        },
    }
